/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.astyanax.recipes.locks;

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.netflix.astyanax.ColumnListMutation;
import com.netflix.astyanax.Keyspace;
import com.netflix.astyanax.MutationBatch;
import com.netflix.astyanax.model.Column;
import com.netflix.astyanax.model.ColumnFamily;
import com.netflix.astyanax.model.ColumnList;
import com.netflix.astyanax.model.ColumnMap;
import com.netflix.astyanax.model.ConsistencyLevel;
import com.netflix.astyanax.model.OrderedColumnMap;
import com.netflix.astyanax.recipes.locks.BusyLockException;
import com.netflix.astyanax.recipes.locks.DistributedRowLock;
import com.netflix.astyanax.recipes.locks.StaleLockException;
import com.netflix.astyanax.retry.RetryPolicy;
import com.netflix.astyanax.retry.RunOnce;
import com.netflix.astyanax.serializers.ByteBufferSerializer;
import com.netflix.astyanax.serializers.LongSerializer;
import com.netflix.astyanax.util.RangeBuilder;
import com.netflix.astyanax.util.TimeUUIDUtils;
import java.nio.ByteBuffer;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

public class ColumnPrefixDistributedRowLock<K>
implements DistributedRowLock {
    public static final int LOCK_TIMEOUT = 60;
    public static final TimeUnit DEFAULT_OPERATION_TIMEOUT_UNITS = TimeUnit.MINUTES;
    public static final String DEFAULT_LOCK_PREFIX = "_LOCK_";
    private final ColumnFamily<K, String> columnFamily;
    private final Keyspace keyspace;
    private final K key;
    private long timeout = 60L;
    private TimeUnit timeoutUnits = DEFAULT_OPERATION_TIMEOUT_UNITS;
    private String prefix = "_LOCK_";
    private ConsistencyLevel consistencyLevel = ConsistencyLevel.CL_LOCAL_QUORUM;
    private boolean failOnStaleLock = false;
    private String lockColumn = null;
    private String lockId = null;
    private Set<String> locksToDelete = Sets.newHashSet();
    private ColumnMap<String> columns = null;
    private Integer ttl = null;
    private boolean readDataColumns = false;
    private RetryPolicy backoffPolicy = RunOnce.get();
    private long acquireTime = 0L;
    private int retryCount = 0;

    public ColumnPrefixDistributedRowLock(Keyspace keyspace, ColumnFamily<K, String> columnFamily, K key) {
        this.keyspace = keyspace;
        this.columnFamily = columnFamily;
        this.key = key;
        this.lockId = TimeUUIDUtils.getUniqueTimeUUIDinMicros().toString();
    }

    public ColumnPrefixDistributedRowLock<K> withConsistencyLevel(ConsistencyLevel consistencyLevel) {
        this.consistencyLevel = consistencyLevel;
        return this;
    }

    public ColumnPrefixDistributedRowLock<K> withColumnPrefix(String prefix) {
        this.prefix = prefix;
        return this;
    }

    public ColumnPrefixDistributedRowLock<K> withDataColumns(boolean flag) {
        this.readDataColumns = flag;
        return this;
    }

    public ColumnPrefixDistributedRowLock<K> withLockId(String lockId) {
        this.lockId = lockId;
        return this;
    }

    public ColumnPrefixDistributedRowLock<K> failOnStaleLock(boolean failOnStaleLock) {
        this.failOnStaleLock = failOnStaleLock;
        return this;
    }

    public ColumnPrefixDistributedRowLock<K> expireLockAfter(long timeout, TimeUnit unit) {
        this.timeout = timeout;
        this.timeoutUnits = unit;
        return this;
    }

    public ColumnPrefixDistributedRowLock<K> withTtl(Integer ttl) {
        this.ttl = ttl;
        return this;
    }

    public ColumnPrefixDistributedRowLock<K> withTtl(Integer ttl, TimeUnit units) {
        this.ttl = (int)TimeUnit.SECONDS.convert(ttl.intValue(), units);
        return this;
    }

    public ColumnPrefixDistributedRowLock<K> withBackoff(RetryPolicy policy) {
        this.backoffPolicy = policy;
        return this;
    }

    @Override
    public void acquire() throws Exception {
        Preconditions.checkArgument((this.ttl == null || TimeUnit.SECONDS.convert(this.timeout, this.timeoutUnits) < (long)this.ttl.intValue() ? 1 : 0) != 0, (Object)("Timeout " + this.timeout + " must be less than TTL " + this.ttl));
        RetryPolicy retry = this.backoffPolicy.duplicate();
        this.retryCount = 0;
        while (true) {
            try {
                long curTimeMicros = ColumnPrefixDistributedRowLock.getCurrentTimeMicros();
                MutationBatch m = this.keyspace.prepareMutationBatch().setConsistencyLevel(this.consistencyLevel);
                this.fillLockMutation(m, curTimeMicros, this.ttl);
                m.execute();
                this.verifyLock(curTimeMicros);
                this.acquireTime = System.nanoTime();
                return;
            }
            catch (BusyLockException e) {
                this.release();
                if (!retry.allowRetry()) {
                    throw e;
                }
                ++this.retryCount;
                continue;
            }
            break;
        }
    }

    public ColumnMap<String> acquireLockAndReadRow() throws Exception {
        this.withDataColumns(true);
        this.acquire();
        return this.getDataColumns();
    }

    public void verifyLock(long curTimeInMicros) throws Exception, BusyLockException, StaleLockException {
        if (this.lockColumn == null) {
            throw new IllegalStateException("verifyLock() called without attempting to take the lock");
        }
        Map<String, Long> lockResult = this.readLockColumns(this.readDataColumns);
        for (Map.Entry<String, Long> entry : lockResult.entrySet()) {
            if (entry.getValue() != 0L && curTimeInMicros > entry.getValue()) {
                if (this.failOnStaleLock) {
                    throw new StaleLockException("Stale lock on row '" + this.key + "'.  Manual cleanup requried.");
                }
                this.locksToDelete.add(entry.getKey());
                continue;
            }
            if (entry.getKey().equals(this.lockColumn)) continue;
            throw new BusyLockException("Lock already acquired for row '" + this.key + "' with lock column " + entry.getKey());
        }
    }

    @Override
    public void release() throws Exception {
        if (!this.locksToDelete.isEmpty() || this.lockColumn != null) {
            MutationBatch m = this.keyspace.prepareMutationBatch().setConsistencyLevel(this.consistencyLevel);
            this.fillReleaseMutation(m, false);
            m.execute();
        }
    }

    public void releaseWithMutation(MutationBatch m) throws Exception {
        this.releaseWithMutation(m, false);
    }

    public boolean releaseWithMutation(MutationBatch m, boolean force) throws Exception {
        long elapsed = TimeUnit.MILLISECONDS.convert(System.nanoTime() - this.acquireTime, TimeUnit.NANOSECONDS);
        boolean isStale = false;
        if (this.timeout > 0L && elapsed > TimeUnit.MILLISECONDS.convert(this.timeout, this.timeoutUnits)) {
            isStale = true;
            if (!force) {
                throw new StaleLockException("Lock for '" + this.getKey() + "' became stale");
            }
        }
        m.setConsistencyLevel(this.consistencyLevel);
        this.fillReleaseMutation(m, false);
        m.execute();
        return isStale;
    }

    public Map<String, Long> readLockColumns() throws Exception {
        return this.readLockColumns(false);
    }

    private Map<String, Long> readLockColumns(boolean readDataColumns) throws Exception {
        LinkedHashMap result = Maps.newLinkedHashMap();
        if (readDataColumns) {
            this.columns = new OrderedColumnMap();
            ColumnList lockResult = (ColumnList)this.keyspace.prepareQuery(this.columnFamily).setConsistencyLevel(this.consistencyLevel).getKey(this.key).execute().getResult();
            for (Column c : lockResult) {
                if (((String)c.getName()).startsWith(this.prefix)) {
                    result.put(c.getName(), this.readTimeoutValue(c));
                    continue;
                }
                this.columns.add(c);
            }
        } else {
            ColumnList lockResult = (ColumnList)this.keyspace.prepareQuery(this.columnFamily).setConsistencyLevel(this.consistencyLevel).getKey(this.key).withColumnRange(new RangeBuilder().setStart(this.prefix + "\u0000").setEnd(this.prefix + "\uffff").build()).execute().getResult();
            for (Column c : lockResult) {
                result.put(c.getName(), this.readTimeoutValue(c));
            }
        }
        return result;
    }

    public Map<String, Long> releaseAllLocks() throws Exception {
        return this.releaseLocks(true);
    }

    public Map<String, Long> releaseExpiredLocks() throws Exception {
        return this.releaseLocks(false);
    }

    public Map<String, Long> releaseLocks(boolean force) throws Exception {
        Map<String, Long> locksToDelete = this.readLockColumns();
        MutationBatch m = this.keyspace.prepareMutationBatch().setConsistencyLevel(this.consistencyLevel);
        ColumnListMutation row = m.withRow(this.columnFamily, this.key);
        long now = ColumnPrefixDistributedRowLock.getCurrentTimeMicros();
        for (Map.Entry<String, Long> c : locksToDelete.entrySet()) {
            if (!force && (c.getValue() <= 0L || c.getValue() >= now)) continue;
            row.deleteColumn((Object)c.getKey());
        }
        m.execute();
        return locksToDelete;
    }

    private static long getCurrentTimeMicros() {
        return TimeUnit.MICROSECONDS.convert(System.currentTimeMillis(), TimeUnit.MILLISECONDS);
    }

    public String fillLockMutation(MutationBatch m, Long time, Integer ttl) {
        if (this.lockColumn != null) {
            if (!this.lockColumn.equals(this.prefix + this.lockId)) {
                throw new IllegalStateException("Can't change prefix or lockId after acquiring the lock");
            }
        } else {
            this.lockColumn = this.prefix + this.lockId;
        }
        Long timeoutValue = time == null ? new Long(0L) : time + TimeUnit.MICROSECONDS.convert(this.timeout, this.timeoutUnits);
        m.withRow(this.columnFamily, this.key).putColumn((Object)this.lockColumn, this.generateTimeoutValue(timeoutValue), ttl);
        return this.lockColumn;
    }

    private ByteBuffer generateTimeoutValue(long timeout) {
        if (this.columnFamily.getDefaultValueSerializer() == ByteBufferSerializer.get() || this.columnFamily.getDefaultValueSerializer() == LongSerializer.get()) {
            return LongSerializer.get().toByteBuffer(Long.valueOf(timeout));
        }
        return this.columnFamily.getDefaultValueSerializer().fromString(Long.toString(timeout));
    }

    public long readTimeoutValue(Column<?> column) {
        if (this.columnFamily.getDefaultValueSerializer() == ByteBufferSerializer.get() || this.columnFamily.getDefaultValueSerializer() == LongSerializer.get()) {
            return column.getLongValue();
        }
        return Long.parseLong(column.getStringValue());
    }

    public void fillReleaseMutation(MutationBatch m, boolean excludeCurrentLock) {
        ColumnListMutation row = m.withRow(this.columnFamily, this.key);
        for (String c : this.locksToDelete) {
            row.deleteColumn((Object)c);
        }
        if (!excludeCurrentLock && this.lockColumn != null) {
            row.deleteColumn((Object)this.lockColumn);
        }
        this.locksToDelete.clear();
        this.lockColumn = null;
    }

    public ColumnMap<String> getDataColumns() {
        return this.columns;
    }

    public K getKey() {
        return this.key;
    }

    public Keyspace getKeyspace() {
        return this.keyspace;
    }

    public ConsistencyLevel getConsistencyLevel() {
        return this.consistencyLevel;
    }

    public String getLockColumn() {
        return this.lockColumn;
    }

    public String getLockId() {
        return this.lockId;
    }

    public String getPrefix() {
        return this.prefix;
    }

    public int getRetryCount() {
        return this.retryCount;
    }
}

