/*
 * #%L
 * Vradi :: Swing
 * 
 * $Id: SendingViewManager.java 1715 2010-10-27 19:21:28Z tchemit $
 * $HeadURL: svn+ssh://tchemit@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.3/vradi-swing/src/main/java/com/jurismarches/vradi/ui/email/helpers/SendingViewManager.java $
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package com.jurismarches.vradi.ui.email.helpers;

import com.jurismarches.vradi.beans.FormPagedResult;
import com.jurismarches.vradi.beans.SendingHelper;
import com.jurismarches.vradi.entities.Sending;
import com.jurismarches.vradi.entities.Session;
import com.jurismarches.vradi.entities.Status;
import com.jurismarches.vradi.entities.User;
import com.jurismarches.vradi.entities.XmlStream;
import com.jurismarches.vradi.services.VradiService;
import com.jurismarches.vradi.ui.email.EmailHandler;
import com.jurismarches.vradi.ui.email.SendingViewUI;
import com.jurismarches.vradi.ui.email.renderers.EmailFormsHighlighter;
import com.jurismarches.vradi.ui.helpers.UIHelper;
import com.jurismarches.vradi.ui.offer.OfferTable;
import com.jurismarches.vradi.ui.offer.models.OfferListTableModel;
import com.jurismarches.vradi.ui.task.VradiTask;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.SwingUtilities;
import jaxx.runtime.swing.ErrorDialogUI;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.decorator.Highlighter;
import org.jdesktop.swingx.decorator.HighlighterFactory;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.WikittyService;
import org.nuiton.wikitty.WikittyServiceEvent;
import org.nuiton.wikitty.WikittyServiceListener;

import java.awt.Cursor;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static org.nuiton.i18n.I18n._;

public class SendingViewManager implements WikittyServiceListener {

    /**
     * Logger
     */
    static private final Log log = LogFactory.getLog(SendingViewManager.class);

    protected Session session;
    protected Sending sending;
    protected SendingViewUI ui;
    protected EmailFormsHighlighter emailFormsHighlighter;
    protected OfferListTableModel listTableModel;
    private static SendingViewManager instance;

    protected SendingViewManager() {

        // register each tree on wikitty service
        VradiService.getWikittyService().addWikittyServiceListener(this, WikittyService.ServiceListenerType.ALL);
    }

    // Singleton
    public static SendingViewManager getInstance() {
        if (instance == null) {
            instance = new SendingViewManager();
        }
        return instance;
    }

    public void openUI(SendingViewUI ui, Session session, Sending sending) {

        if (sending == null) {
            return;
        }

        this.ui = ui;
        WikittyProxy wikittyProxy = VradiService.getWikittyProxy();
        this.sending = wikittyProxy.restore(Sending.class, sending.getWikittyId());
        this.session = session;

        // Init session ui
        ui.setBean(sending);

        ui.setSession(session);

        // listTable cellRenderer
        // TODO SLE 30/09/2010 : Refactor it !
        OfferListTableModel.OfferListTableCellRenderer renderer =
                new OfferListTableModel.OfferListTableCellRenderer();

        OfferTable listTable = ui.getListTable();

        listTableModel = (OfferListTableModel) listTable.getModel();

        emailFormsHighlighter = new EmailFormsHighlighter(listTableModel);
        updateDatas();

        listTable.setModel(listTableModel);

        listTable.setDefaultRenderer(Boolean.class, renderer);
        listTable.setDefaultRenderer(Date.class, renderer);
        listTable.setDefaultRenderer(Double.class, renderer);
        listTable.setDefaultRenderer(String.class, renderer);
        listTable.setDefaultRenderer(XmlStream.class, renderer);
        listTable.setDefaultRenderer(Status.class, renderer);

        // listTable highlighter
        Highlighter highlighter =
                HighlighterFactory.createAlternateStriping(
            HighlighterFactory.NOTEPAD, HighlighterFactory.GENERIC_GRAY);

        listTable.addHighlighter(highlighter);
        listTable.addHighlighter(emailFormsHighlighter);

        // Add mouse listener
        MouseListener offerListTableMouseListener = getOfferListTableMouseListener();
        listTable.addMouseListener(offerListTableMouseListener);

        // init ui
        ui.init();
    }

    protected void updateDatas() {
        WikittyProxy wikittyProxy = VradiService.getWikittyProxy();

        FormPagedResult formPage = new FormPagedResult();

        Set<String> formsIdsToDisplay = new HashSet<String>();

        // Get forms
        Set<String> forms = SendingHelper.extractForms(wikittyProxy, sending.getGroupForms());
        if (forms != null) {
            formsIdsToDisplay.addAll(forms);
        }

        // Get deleted
        Set<String> deletedForms = SendingHelper.extractForms(wikittyProxy, sending.getDeletedGroupForms());
        Set<String> addedForms = SendingHelper.extractForms(wikittyProxy, sending.getAddedGroupForms());
        if (deletedForms != null) {
            formsIdsToDisplay.addAll(deletedForms);
        }
        if (!formsIdsToDisplay.isEmpty()) {
            formPage.setFormsToShow(new ArrayList<String>(formsIdsToDisplay));
        }

        emailFormsHighlighter.setFormsAdded(addedForms);
        emailFormsHighlighter.setFormsDeleted(deletedForms);
        listTableModel.setFormPagedResult(formPage);

        emailFormsHighlighter.notifyChange();
    }

    protected void refresh() {
        WikittyProxy wikittyProxy = VradiService.getWikittyProxy();
        sending = wikittyProxy.restore(Sending.class, sending.getWikittyId());

        ui.setBean(sending);

        updateDatas();
    }

    @Override
    public void putWikitty(final WikittyServiceEvent event) {

        if (sending == null) {
            return;
        }

        // map between id and extensions "name" (not extension ids)
        final Map<String, Set<String>> idAndExtensions = event.getIdExtensions();

        SwingUtilities.invokeLater(new Thread(){
            @Override
            public void run() {
                boolean refresh = false;
                for (final String wikittyId : event.getIds()) {
                    if (idAndExtensions != null) {
                        Set<String> wikittyExtensions = idAndExtensions.get(wikittyId);

                        // Sending
                        if (wikittyExtensions.contains(Sending.EXT_SENDING)){

                            if (wikittyId.equals(sending.getWikittyId())) {
                                refresh = true;
                            }
                        }
                    }
                }
                if (refresh) {
                    refresh();
                }
            }
        });
    }

    @Override
    public void removeWikitty(WikittyServiceEvent event) {
    }

    @Override
    public void clearWikitty(WikittyServiceEvent event) {
    }

    @Override
    public void putExtension(WikittyServiceEvent event) {
    }

    @Override
    public void removeExtension(WikittyServiceEvent event) {
    }

    @Override
    public void clearExtension(WikittyServiceEvent event) {
    }

    /**
     * Implementation du click (droit/gauche) sur la table des offres résultats/
     *
     * Comportement :
     * <ul>
     *  <li>double clic : ouvre l'offre
     *  <li>clic droit : changer le status, ajouter a une session de mail
     * </ul>
     *
     * @return le mouse listener
     */
    public MouseListener getOfferListTableMouseListener() {
        // double click on listTable shows selected offer
        return new MouseAdapter() {

            @Override
            public void mouseClicked(final MouseEvent e) {
                JXTable listTable = (JXTable)e.getSource();

                // double clic
                if (e.getButton() == MouseEvent.BUTTON1 &&
                        e.getClickCount() == 2) {
                    try {
                        ui.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));

                        EmailHandler handler = UIHelper.getHandler(ui, EmailHandler.class);
                        handler.selectOffer(ui, listTable);
                    } catch (Exception ex) {
                        if (log.isErrorEnabled()) {
                            log.error("Can't open offer", ex);
                        }
                        ErrorDialogUI.showError(ex);
                    } finally {
                        ui.setCursor(null);
                    }

                // clic droit
                } else if (e.getButton() == MouseEvent.BUTTON3) {
                    int[] selectedRows = listTable.getSelectedRows();
                    int rowClicked = listTable.rowAtPoint(e.getPoint());
                    boolean rowClickedSelected = false;
                    if (selectedRows == null || selectedRows.length == 0) {
                        listTable.getSelectionModel().setSelectionInterval(rowClicked, rowClicked);
                        rowClickedSelected = true;
                    } else {
                        int i = 0;
                        while(i < selectedRows.length && !rowClickedSelected) {
                            rowClickedSelected = rowClicked == selectedRows[i++];
                        }
                    }

                    if (rowClickedSelected) {
                        showOfferListContextMenu(ui, listTable, ui.getOfferListTableModel(), e);
                    }
                }
            }
        };
    }

    /**
     * Le menu contextuel lors d'un clic droit sur la table.
     *
     * @param ui sending view ui
     * @param table of offers
     * @param model of table
     * @param e event du clic
     */
    protected void showOfferListContextMenu(final SendingViewUI ui, final JXTable table, OfferListTableModel model, final MouseEvent e) {

        // Enable menu only if session is active
        boolean enabled = (ui.getSession() != null && ui.isNotClosed());

        JPopupMenu menu = new JPopupMenu();

        // Menu to add user or group sending
        JMenuItem removeFormFromSession = new JMenuItem(_("vradi.email.removeFormFromSession"));

        removeFormFromSession.setEnabled(enabled);

        if (enabled) {
            removeFormFromSession.addActionListener(new RemoveAllFormFromSession(ui, table, model));
        }

        menu.add(removeFormFromSession);

        // Menu to add user or group sending
        JMenuItem removeFormFromSessionForUser = new JMenuItem(_("vradi.email.removeFormFromSessionForUser"));

        removeFormFromSessionForUser.setEnabled(enabled);

        if (enabled) {
            removeFormFromSessionForUser.addActionListener(new RemoveFormFromSessionForUser(ui, table, model));
        }

        menu.add(removeFormFromSessionForUser);

        int[] selectedRows = table.getSelectedRows();
        final List<String> formsIdsToUpdate = new ArrayList<String>();
        for(int row : selectedRows) {
            row = table.convertRowIndexToModel(row);

            FormPagedResult formPage = listTableModel.getFormPagedResult();
            List<String> formsToShow = formPage.getFormsIdsToShow();

            String formId = formsToShow.get(row);
            if (formId != null) {

                Set<String> deletedFormsIds = SendingHelper.extractForms(VradiService.getWikittyProxy(), sending.getDeletedGroupForms());
                if (deletedFormsIds != null) {
                    if (deletedFormsIds.contains(formId)) {
                        formsIdsToUpdate.add(formId);
                    }

                }
            }
        }

        if (!formsIdsToUpdate.isEmpty()) {

            // Display popup menu
            final JMenuItem revertDeletion = new JMenuItem(_("vradi.email.forms.revertDeletion"));
            revertDeletion.addActionListener(new ActionListener() {

                @Override
                public void actionPerformed(ActionEvent e) {
                    EmailHandler handler = UIHelper.getHandler(ui, EmailHandler.class);
                    handler.revertDeletion(ui, sending, formsIdsToUpdate);
                }
            });

            menu.add(revertDeletion);
        }

        menu.show(e.getComponent(), e.getX(), e.getY());
    }

    protected abstract class RemoveFormFromSession implements ActionListener {

        protected List<String> formsIdsToUpdate = new ArrayList<String>();

        protected SendingViewUI ui;
        protected User user;
        protected Session session;
        protected EmailHandler handler;

        protected RemoveFormFromSession(SendingViewUI ui, JXTable listTable, OfferListTableModel tableModel) {
            this.ui = ui;
            handler = UIHelper.getHandler(ui, EmailHandler.class);

            Sending sending = ui.getBean();
            user = VradiService.getWikittyProxy().restore(User.class, sending.getUser());

            session = VradiService.getVradiDataService().getSessionBySending(sending.getWikittyId());

            int[] selectedRows = listTable.getSelectedRows();

            for(int row : selectedRows) {
                row = listTable.convertRowIndexToModel(row);

                FormPagedResult formPage = tableModel.getFormPagedResult();
                List<String> formsIdsToShow = formPage.getFormsIdsToShow();

                String formId = formsIdsToShow.get(row);
                if (formId != null) {
                    formsIdsToUpdate.add(formId);
                }
            }
        }
    }

    protected class RemoveAllFormFromSession extends RemoveFormFromSession {

        public RemoveAllFormFromSession(SendingViewUI ui, JXTable listTable, OfferListTableModel tableModel) {
            super(ui, listTable, tableModel);
        }

        @Override
        public void actionPerformed(ActionEvent e) {
            if (handler.askToRemoveAllForm(ui, EmailDataHelper.convertFormsToString(formsIdsToUpdate))) {
                VradiTask<Void> removeFormsFromSessionTask = new VradiTask<Void>(ui) {
                    @Override
                    public Void doAction() throws Exception {
                        VradiService.getVradiStorageService().removeFormsFromSession(session.getWikittyId(), formsIdsToUpdate);
                        return null;
                    }
                };
                removeFormsFromSessionTask.execute();
            }
        }
    }

    protected class RemoveFormFromSessionForUser extends RemoveFormFromSession {

        public RemoveFormFromSessionForUser(SendingViewUI ui, JXTable listTable, OfferListTableModel tableModel) {
            super(ui, listTable, tableModel);
        }

        @Override
        public void actionPerformed(ActionEvent e) {
            if (handler.askToRemoveUserForm(ui, EmailDataHelper.convertFormsToString(formsIdsToUpdate), user.getName())) {
                handler.removeUserSending(ui, session, formsIdsToUpdate, new Object[]{user}, false);
            }
        }
    }
}
