/*
 * #%L
 * Vradi :: Swing
 * 
 * $Id: ThesaurusDataHelper.java 1715 2010-10-27 19:21:28Z tchemit $
 * $HeadURL: svn+ssh://sletellier@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.3.1/vradi-swing/src/main/java/com/jurismarches/vradi/ui/helpers/ThesaurusDataHelper.java $
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package com.jurismarches.vradi.ui.helpers;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang.xwork.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.wikitty.WikittyProxy;

import com.jurismarches.vradi.entities.RootThesaurus;
import com.jurismarches.vradi.entities.Thesaurus;
import com.jurismarches.vradi.services.VradiService;

public class ThesaurusDataHelper {

    /**
     * Logger
     */
    static private final Log log = LogFactory.getLog(ThesaurusDataHelper.class);

    /**
     * Get the wikitty proxy
     * @return WikittyProxy
     */
    public static WikittyProxy getProxy() {
        return VradiService.getWikittyProxy();
    }

    /**
     * Get corresponding thesaurus
     *
     * @param thesaurusId thesaurus id
     * @return Thesaurus from base
     */
    public static Thesaurus restoreThesaurus(String thesaurusId){
        return getProxy().restore(Thesaurus.class, thesaurusId);
    }

    /**
     * Get corresponding list of thesaurus
     *
     * @param thesaurusIds thesaurus ids
     * @return List of thesaurus from base
     */
    public static List<Thesaurus> restoreThesaurus(List<String> thesaurusIds){
        return getProxy().restore(Thesaurus.class, thesaurusIds);
    }

    public static List<Thesaurus> restoreThesaurus(Set<String> thesaurusIds) {
        return restoreThesaurus(new ArrayList<String>(thesaurusIds));
    }

    /**
     * Get corresponding root thesaurus
     *
     * @param rootThesaurusId root thesaurus id
     * @return Thesaurus from base
     */
    public static RootThesaurus restoreRootThesaurus(String rootThesaurusId){
        return getProxy().restore(RootThesaurus.class, rootThesaurusId);
    }

    /**
     * Get corresponding list root thesaurus
     *
     * @param rootThesaurusIds root thesaurus ids
     * @return List root thesaurus from base
     */
    public static List<RootThesaurus> restoreRootThesaurus(List<String> rootThesaurusIds){
        return getProxy().restore(RootThesaurus.class, rootThesaurusIds);
    }

    /**
     * Get the path with name of thesaurus ({@code root/thesaurus...})
     * @param thesaurus thesaurus concerned
     * @param pathSeparator separator
     * @return path with name
     */
    public static String getNamePath(Thesaurus thesaurus, String pathSeparator) {
        if (thesaurus == null) {
            return "";
        }
        String parentId = thesaurus.getParent();

        // If root
        if (isFirstChild(thesaurus)) {
            return restoreRootThesaurus(thesaurus.getRootThesaurus()).getName() + pathSeparator + concateOrderName(thesaurus);
        }
        Thesaurus parentThesaurus = restoreThesaurus(parentId);
        return getNamePath(parentThesaurus, pathSeparator) + pathSeparator + concateOrderName(thesaurus);
    }

    /**
     * Concatenate order and name of thesaurus in param ({@code order - name}}
     *
     * @param thesaurus thesaurus concerned
     * @return concatenate order and name
     */
    public static String concateOrderName(Thesaurus thesaurus) {
        String result = thesaurus.getName();
        int order = thesaurus.getOrder();
        if (order != 0){
            result = order + " - " + result;
        }
        return result;
    }

    /**
     * Get tags as string ({@code tag1, tag2, ...})
     *
     * @param thesaurus thesaurus concerned
     * @return tags as string
     */
    public static String getTagsAsString(Thesaurus thesaurus) {
        if (thesaurus == null) {
            return "";
        }
        String result = null;
        if (thesaurus.getTags() != null) {
            result = StringUtils.join(thesaurus.getTags(), " ,");
        }
        else {
            result = "";
        }

        return result;
    }

    /**
     * Set tags as string ({@code tag1, tag2, ...})
     *
     * @param thesaurus    thesaurus concerned
     * @param tagsAsString tags as string
     */
    public static void setTagsAsString(Thesaurus thesaurus, String tagsAsString) {
        if (tagsAsString != null) {

            // Clear existing
            thesaurus.clearTags();

            // Resore tags from tags as string
            for (String tag : tagsAsString.split(",")) {
                thesaurus.addTags(tag.trim());
            }
        }
    }

    /**
     * Return number of form associated to thesaurus pass in param
     *
     * @param thesaurus thesaurus concerned
     * @return number of form associated
     */
    public static int getNbFormsForThesaurus(Thesaurus thesaurus) {

        Map.Entry<Thesaurus, Integer> entry =
                getProxy().restoreNode(Thesaurus.class, thesaurus.getWikittyId(), null);

        if (entry == null) {
            return 0;
        }

        return entry.getValue();
    }

    protected static List<String> getStringAsList(String s) {
        List<String> result = new ArrayList<String>();
        result.add(s);
        return result;
    }

    public static String getParentPath(Thesaurus thesaurus, String pathSeparator) {
        if (thesaurus == null) {
            return pathSeparator;
        }
        if (isFirstChild(thesaurus)) {
            RootThesaurus rootThesaurus = restoreRootThesaurus(thesaurus.getRootThesaurus());
            return rootThesaurus.getName();
        }
        String parentId = thesaurus.getParent();
        Thesaurus parentThesaurus = restoreThesaurus(parentId);
        return getParentPath(parentThesaurus, pathSeparator) + pathSeparator + concateOrderName(parentThesaurus);
    }

    public static int getDepth(Thesaurus thesaurus){
        if (thesaurus == null) {
            return 0;
        }
        if (isFirstChild(thesaurus)) {
            return 2;
        }
        String parentId = thesaurus.getParent();
        Thesaurus parent = restoreThesaurus(parentId);
        return 1 + getDepth(parent);
    }

    public static boolean isFirstChild(Thesaurus thesaurus) {
        String rootThesaurus = thesaurus.getRootThesaurus();
        String parent = thesaurus.getParent();
        if (parent == null) {
            return false;
        }
        return parent.equals(rootThesaurus);
    }

    public static String getRequestName(Thesaurus thesaurus) {
        String rootId = thesaurus.getRootThesaurus();
        RootThesaurus rootThesaurus = restoreRootThesaurus(rootId);
        return rootThesaurus.getName() + ":\"" + thesaurus.getName() + "\"";
    }
}