/*
 * #%L
 * Vradi :: Swing
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package com.jurismarches.vradi.ui.offer.thesaurus.helpers;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Set;

import javax.swing.*;
import javax.swing.tree.TreeModel;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.wikitty.WikittyService.ServiceListenerType;

import com.jurismarches.vradi.entities.RootThesaurus;
import com.jurismarches.vradi.entities.Thesaurus;
import com.jurismarches.vradi.services.VradiService;
import com.jurismarches.vradi.ui.helpers.ThesaurusDataHelper;
import com.jurismarches.vradi.ui.helpers.VradiTreeHelper;
import com.jurismarches.vradi.ui.offer.thesaurus.loadors.RootThesaurusNodeLoadors;
import com.jurismarches.vradi.ui.offer.thesaurus.loadors.ThesaurusNodeLoadors;
import com.jurismarches.vradi.ui.tree.VradiDataProvider;
import com.jurismarches.vradi.ui.tree.VradiTreeNode;
import com.jurismarches.vradi.ui.tree.VradiTreeTableNode;

/**
 * Navigation tree helpers.
 *
 * @author sletellier
 * @see jaxx.runtime.swing.nav.tree.NavTreeHelper
 */
public class ThesaurusTreeHelper extends VradiTreeHelper {

    /**
     * Logger
     */
    static private final Log log = LogFactory.getLog(ThesaurusTreeHelper.class);

    protected RootThesaurus rootThesaurus;

    protected Thesaurus thesaurus;

    protected boolean listening = true;

    /**
     * Create helper with first nodes are the root thesaurus
     *
     * @param dataProvider data provider
     */
    public ThesaurusTreeHelper(VradiDataProvider dataProvider) {
        super();
        init(dataProvider);
    }

    /**
     * Create helper with root thesaurus for root node
     *
     * @param dataProvider  data provider
     * @param rootThesaurus root thesaurus for root node
     */
    public ThesaurusTreeHelper(VradiDataProvider dataProvider, RootThesaurus rootThesaurus) {
        super();
        this.rootThesaurus = rootThesaurus;
        init(dataProvider);
    }

    /**
     * Create helper with thesaurus for root node
     *
     * @param dataProvider data provider
     * @param thesaurus    thesaurus for root node
     */
    public ThesaurusTreeHelper(VradiDataProvider dataProvider, Thesaurus thesaurus) {
        super();
        this.thesaurus = thesaurus;
        init(dataProvider);
    }

    protected void init(VradiDataProvider dataProvider) {
        setDataProvider(dataProvider);

        // register each tree on wikitty service
        VradiService.getWikittyService().addWikittyServiceListener(this, ServiceListenerType.ALL);
    }

    public TreeModel createTreeModel() {
        
        VradiTreeNode root;
        if (thesaurus != null) {
            if (log.isDebugEnabled()) {
                log.debug("[Root creation] thesaurus " + thesaurus.getName());
            }

            root = new VradiTreeNode(
                    Thesaurus.class,
                    thesaurus.getWikittyId(),
                    null,
                    getChildLoador(ThesaurusNodeLoadors.class)
            );
        } else if (rootThesaurus != null) {
            if (log.isDebugEnabled()) {
                log.debug("[Root creation] root thesaurus " + rootThesaurus.getName());
            }
            root = new VradiTreeNode(
                    RootThesaurus.class,
                    rootThesaurus.getWikittyId(),
                    null,
                    getChildLoador(ThesaurusNodeLoadors.class)
            );
        } else {
            if (log.isDebugEnabled()) {
                log.debug("[Root creation] all root thesaurus");
            }
            root = new VradiTreeNode(
                    String.class,
                    "Root node",
                    null,
                    getChildLoador(RootThesaurusNodeLoadors.class)
            );
        }

        TreeModel model = createModel(root);

        // Populate childs nodes
        root.populateChilds(getBridge(), getDataProvider());

        registerLoadedIds(root);

        return model;
    }

    /**
     * Try to select thesaurus pass in param
     *
     * @param thesaurus list of thesaurus to select
     */
    public void tryToSelect(List<Thesaurus> thesaurus) {
        tryToSelect(extractIds(thesaurus));
    }

    /**
     * Try to select thesaurus ids pass in param
     *
     * @param thesaurusIds list of thesaurus ids to select
     */
    public void tryToSelect(Collection<String> thesaurusIds) {
        if (thesaurusIds == null) {
            if (log.isDebugEnabled()) {
                log.debug("Try to select, ids are null");
            }
            return;
        }
        List<VradiTreeNode> nodes = new ArrayList<VradiTreeNode>();

        // Find recursivly
        for (String id : thesaurusIds) {

            // Skip if root
            if (!id.equals(getRootNode().getId())) {

                VradiTreeNode rootNode = getRootNode();
                VradiTreeNode nodeFound = findNode(rootNode, id);
                if (nodeFound != null) {
                    nodes.add(nodeFound);
                }

            }
        }

        selectNodes(nodes);
    }

    /**
     * Try to unselect thesaurus pass in param
     *
     * @param thesaurus list of thesaurus to unselect
     */
    public void tryToUnselect(List<Thesaurus> thesaurus) {
        tryToUnselect(extractIds(thesaurus));
    }

    /**
     * Try to unselect thesaurus ids pass in param
     *
     * @param thesaurusIds list of thesaurus ids to select
     */
    public void tryToUnselect(Collection<String> thesaurusIds) {
        if (thesaurusIds == null) {
            return;
        }
        List<VradiTreeNode> nodes = new ArrayList<VradiTreeNode>();

        // Find recursivly
        for (String id : thesaurusIds) {
            VradiTreeNode nodeFound = findNode(getRootNode(), id);
            if (nodeFound != null) {
                nodes.add(nodeFound);
            }
        }

        unSelectNodes(nodes);
    }

    protected List<String> extractIds(List<Thesaurus> beans){
        List<String> ids = new ArrayList<String>();
        if (beans != null){
            for (Thesaurus bean : beans){
                if (bean != null) {
                    ids.add(bean.getWikittyId());
                }
            }
        }
        return ids;
    }

    public static RootThesaurus getRootThesaurus(VradiTreeNode node) {
        if (!node.getInternalClass().equals(RootThesaurus.class)) {
            return null;
        }
        return ThesaurusDataHelper.restoreRootThesaurus(node.getId());
    }

    public static Thesaurus getThesaurus(VradiTreeNode node) {
        if (!node.getInternalClass().equals(Thesaurus.class)) {
            return null;
        }
        return ThesaurusDataHelper.restoreThesaurus(node.getId());
    }

    public static RootThesaurus getRootThesaurus(VradiTreeTableNode node) {
        if (!node.getInternalClass().equals(RootThesaurus.class)) {
            return null;
        }
        return ThesaurusDataHelper.restoreRootThesaurus(node.getId());
    }

    public static Thesaurus getThesaurus(VradiTreeTableNode node) {
        if (!node.getInternalClass().equals(Thesaurus.class)) {
            return null;
        }
        return ThesaurusDataHelper.restoreThesaurus(node.getId());
    }

    protected void refresh(VradiTreeNode node) {

        if (node == null) {
            return;
        }
        
        getBridge().nodeChanged(node);
    }

    @Override
    public void putWikitty(final String wikittyId, Set<String> wikittyExtensions) {

        // Thesaurus
        if (wikittyExtensions.contains(Thesaurus.EXT_THESAURUS)) {

            // Restore
            final Thesaurus thesaurusConcerned = ThesaurusDataHelper.restoreThesaurus(wikittyId);

            if (thesaurusConcerned == null) {
                return;
            }
            
            // Search on parent to creation case
            if (idsLoaded.contains(thesaurusConcerned.getParent())) {

                SwingUtilities.invokeLater(new Thread() {

                    @Override
                    public void run() {
                        VradiTreeNode existingNode = findNode(getRootNode(), wikittyId);

                        // cas modification
                        if (existingNode != null) {

                            // Parent id
                            VradiTreeNode parent = existingNode.getParent();

                            // Move case
                            String newParentId = thesaurusConcerned.getParent();
                            if (parent != null && !parent.getId().equals(newParentId)) {

                                VradiTreeNode newParentNode = findNode(getRootNode(), newParentId);

                                // Delete old
                                getBridge().removeNodeFromParent(existingNode);

                                // Insert
                                // Verify that node is not already created
                                if (findNode(newParentNode, wikittyId) == null) {
                                    insertNode(newParentNode, existingNode);
                                }

                            } else {
                                if (log.isDebugEnabled()) {
                                    log.debug("Modification case :  " + thesaurusConcerned.getName());
                                }

                                // Modification case
                                refresh(existingNode);
                            }
                            return;
                        }

                        // Creation case
                        else {

                            if (log.isDebugEnabled()) {
                                log.debug("Creation case :  " + thesaurusConcerned.getName());
                            }

                            // Find parent node
                            VradiTreeNode parentNode = findNode(getRootNode(), thesaurusConcerned.getParent());

                            // Tri
                            VradiTreeNode newNode = getChildLoador(ThesaurusNodeLoadors.class).createNode(wikittyId, getDataProvider());
                            // Verify that node is not already created
                            if (findNode(parentNode, wikittyId) == null) {
                                insertNode(parentNode, newNode);
                            }
                        }
                    }
                });
            }
        }

        // Root thesaurus
        if (wikittyExtensions.contains(RootThesaurus.EXT_ROOTTHESAURUS)) {

            SwingUtilities.invokeLater(new Thread() {

                @Override
                public void run() {
                    // Find existing
                    VradiTreeNode existingNode = findNode(getRootNode(), wikittyId);

                    // Modification case
                    if (existingNode != null) {
                        refresh(existingNode);
                        return;
                    }
                    if (log.isDebugEnabled()) {
                        log.debug("Refresh root thesaurus : " + wikittyId);
                    }

                    // Creation
                    VradiTreeNode newNode = getChildLoador(RootThesaurusNodeLoadors.class).createNode(wikittyId, getDataProvider());
                    insertNode(getRootNode(), newNode);
                }
            });
        }
    }
}
