package com.jurismarches.vradi.ui.models;

import java.awt.Component;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.swing.JComponent;
import javax.swing.JTree;
import javax.swing.tree.DefaultTreeCellRenderer;

import org.apache.commons.lang.StringUtils;
import org.jdesktop.swingx.treetable.AbstractTreeTableModel;
import org.nuiton.i18n.I18n;
import org.sharengo.wikitty.TreeNodeImpl;

import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.Thesaurus;
import com.jurismarches.vradi.services.dto.VradiCartographyDTO;
import com.jurismarches.vradi.ui.VradiComparators;
import com.jurismarches.vradi.ui.helpers.ToolTipHelper;

/**
 * ThesaurusCartographyModel.
 *
 * @author schorlet
 * @version $Revision: 756 $ $Date: 2010-04-26 11:02:19 +0200 (lun., 26 avril 2010) $
 * @since 14 avr. 2010 14:40:00
 */
public class ThesaurusCartographyModel extends AbstractTreeTableModel {
    
    /** chidren nodes indexed by parent node id */
    final Map<String, List<TreeNodeImpl>> nodeList = new HashMap<String, List<TreeNodeImpl>>();
    /** chidren form indexed by parent node id */
    final Map<String, List<Form>> formList = new HashMap<String, List<Form>>();
    
    /** top level nodes */
    final ArrayList<TreeNodeImpl> topLevels = new ArrayList<TreeNodeImpl>();
    
    /** number of matching forms indexed by node */
    final Map<TreeNodeImpl, Integer> cartography;
    
    public ThesaurusCartographyModel(VradiCartographyDTO cartographyDTO) {
        super(1L);
        this.cartography = cartographyDTO.getCartography();
        List<Form> forms = cartographyDTO.getForms();
        
        // creates the lists of nodes/forms indexed by parent node id
        Set<TreeNodeImpl> keySet = cartography.keySet();
        for (TreeNodeImpl node : keySet) {
            nodeList.put(node.getWikittyId(), new ArrayList<TreeNodeImpl>());
            formList.put(node.getWikittyId(), new ArrayList<Form>());
        }
        
        // fill the node lists with children nodes
        for (TreeNodeImpl node : keySet) {
            String parent = node.getParent();
            if (parent != null && nodeList.containsKey(parent)) {
                List<TreeNodeImpl> list = nodeList.get(parent);
                list.add(node);
            } else {
                // the parent is root
                topLevels.add(node);
            }
        }
        
        // fill the form lists with forms
        for (Form form : forms) {
            Set<String> thesaurus = form.getThesaurus();
            if (thesaurus == null || thesaurus.isEmpty()) {
                continue;
            }
            
            for (String id : thesaurus) {
                if (formList.containsKey(id)) {
                    List<Form> list = formList.get(id);
                    list.add(form);
                }
            }
        }
        
        // sort the node lists
        for (List<TreeNodeImpl> entry : nodeList.values()) {
            if (entry.isEmpty()) {
                continue;
            }
            Collections.sort(entry, VradiComparators.THESAURUS_COMPARATOR_TREENODE);
        }
        
        Collections.sort(topLevels, VradiComparators.THESAURUS_COMPARATOR_TREENODE);
    }
    
    @Override
    public int getColumnCount() {
        return 3;
    }
    
    @Override
    public Object getValueAt(Object node, int column) {
        if (node instanceof TreeNodeImpl) {
            TreeNodeImpl treeNode = (TreeNodeImpl) node;
            
            if (column == 0) {
                return treeNode.getName();
                
            } else if (column == 1) {
                return cartography.get(treeNode);
                
            } else if (column == 2) {
                List<String> tags = (List<String>) treeNode.getField(
                        Thesaurus.EXT_THESAURUS,
                        Thesaurus.TAGS_FIELD);
                return StringUtils.join(tags, ',');
            }
        }
        return null;
    }

    @Override
    public Object getChild(Object parent, int index) {
        if (parent instanceof Long) {
            return topLevels.get(index);
            
        } else if (parent instanceof TreeNodeImpl) {
            TreeNodeImpl treeNode = (TreeNodeImpl) parent;
            String id = treeNode.getWikittyId();
            List<TreeNodeImpl> list = nodeList.get(id);
            return list.get(index);
        }
        return null;
    }

    @Override
    public int getChildCount(Object parent) {
        if (parent instanceof Long) {
            return topLevels.size();
            
        } else if (parent instanceof TreeNodeImpl) {
            TreeNodeImpl treeNode = (TreeNodeImpl) parent;
            String id = treeNode.getWikittyId();
            
            List<TreeNodeImpl> list = nodeList.get(id);
            return list.size();
        }
        return 0;
    }

    @Override
    public int getIndexOfChild(Object parent, Object child) {
        if (parent instanceof Long) {
            return topLevels.indexOf(child);
            
        } else if (parent instanceof TreeNodeImpl) {
            TreeNodeImpl treeNode = (TreeNodeImpl) parent;
            String id = treeNode.getWikittyId();
            List<TreeNodeImpl> list = nodeList.get(id);
            return list.indexOf(child);
        }
        return -1;
    }

    @Override
    public String getColumnName(int column) {
        String columnName = null;
        if (column == 0) {
            columnName = I18n._("vradi.thesaurus.name");
        } else if (column == 1) {
            columnName = I18n._("vradi.thesaurus.nbforms");
        } else if (column == 2) {
            columnName = I18n._("vradi.thesaurus.tags");
        }
        return columnName;
    }
    
    public class TreeCellRenderer extends DefaultTreeCellRenderer {
        private static final long serialVersionUID = 1L;
        
        @Override
        public Component getTreeCellRendererComponent(JTree tree, Object value,
                boolean selected, boolean expanded, boolean leaf, int row,
                boolean hasFocus) {

            Object value2 = value;
            String toolTip = null;
            
            if (value instanceof TreeNodeImpl) {
                TreeNodeImpl node = (TreeNodeImpl) value;
                value2 = node.getName();
                
                List<Form> forms = formList.get(node.getWikittyId());
                toolTip = ToolTipHelper.getToolTip(forms);
            }
            
            JComponent component = (JComponent) super.getTreeCellRendererComponent(
                    tree, value2, selected, expanded, leaf, row, hasFocus);
            component.setToolTipText(toolTip);
            
            return component;
        }
    }
}
