/*
 * *##%
 * Vradi :: Swing
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.ui.search;

import java.awt.Component;
import java.io.File;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.GregorianCalendar;
import java.util.List;

import javax.swing.ComboBoxModel;
import javax.swing.DefaultComboBoxModel;
import javax.swing.DefaultListCellRenderer;
import javax.swing.JComboBox;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.ListCellRenderer;

import jaxx.runtime.JAXXContext;
import jaxx.runtime.context.JAXXInitialContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.sharengo.wikitty.FieldType;
import org.sharengo.wikitty.TreeNode;
import org.sharengo.wikitty.TreeNodeImpl;
import org.sharengo.wikitty.WikittyExtension;

import com.jurismarches.vradi.VradiContext;
import com.jurismarches.vradi.VradiHelper;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.services.dto.VradiFormPageDTO;
import com.jurismarches.vradi.services.dto.VradiThesaurusDTO;
import com.jurismarches.vradi.ui.OfferListTableModel;
import com.jurismarches.vradi.ui.OfferListUI;
import com.jurismarches.vradi.ui.ThesaurusHandler;
import com.jurismarches.vradi.ui.UIHelper;

/**
 * @author letellier
 */
public class SearchHandler {

    static private final Log log = LogFactory.getLog(SearchHandler.class);

    static public final String CSV_FILE_EXTENSION = ".csv";

    /**
     * Methode pour initialiser l'ui principale sans l'afficher.
     *
     * @param rootContext le context applicatif
     * @return l'ui instancie et initialisee mais non visible encore
     */
    public SearchUI initUI(JAXXContext rootContext, OfferListTableModel resultTableModel) {
        SearchUI ui = getUI(rootContext);
        if (ui == null) {

            JAXXInitialContext context = new JAXXInitialContext().add(rootContext);

            // show main ui
            context.add(this);
            context.add(resultTableModel);

            ui = new SearchUI(context);

            initCriteria(ui);

            VradiContext.SEARCH_UI_ENTRY_DEF.setContextValue(rootContext, ui);
        }

        return ui;
    }

    public SearchUI initCloneUI(JAXXContext rootContext, OfferListTableModel resultTableModel) {
        return initCloneUI(rootContext, resultTableModel, null);
    }

    public SearchUI initCloneUI(JAXXContext rootContext, OfferListTableModel resultTableModel, List<VradiThesaurusDTO> thesaurus) {

        JAXXInitialContext context = new JAXXInitialContext().add(rootContext);

        // show main ui
        context.add(this);
        context.add(resultTableModel);

        SearchUI ui = new SearchUI(context);

        initCriteria(ui, thesaurus);
        
        return ui;
    }

    SearchUI getUI(JAXXContext context) {
        if (context instanceof SearchUI) {
            return (SearchUI) context;
        }
        
        SearchUI ui = VradiContext.SEARCH_UI_ENTRY_DEF.getContextValue(context);
        return ui;
    }

    public ComboBoxModel getRequestModel() {
        return new DefaultComboBoxModel();
    }

    // From Admin Handler
    public void executeQuery(JAXXContext context) {
        SearchUI searchUI = getUI(context);
        
        if (searchUI != null) {
            String query = searchUI.getRequest();
            JComboBox combo = searchUI.isOfferListUI() ? searchUI.getRequests() : null;
            CriteriaUI criterias = searchUI.getCriterias().isVisible() ? searchUI.getCriterias() : null;
            
            OfferListTableModel listTableModel = searchUI.getResultTableModel();
            VradiFormPageDTO formPageDTO = listTableModel.getFormPageDTO();
            
            executeQuery(searchUI, criterias, formPageDTO, combo, query);
            listTableModel.setFormPageDTO(formPageDTO);
        } else {
            log.warn("searchUI is null !");
        }
    }

    public void executeQuery(JAXXContext context, int pageToShow) {
        SearchUI searchUI = getUI(context);
        
        if (searchUI != null) {
            OfferListTableModel listTableModel = searchUI.getResultTableModel();
            listTableModel.setPageToShow(pageToShow);
        } else {
            log.warn("searchUI is null !");
        }
    }
    
    public java.util.List<String> loadRequests() {
        return VradiHelper.loadRequests();
    }

    public JPanel getCriteriaPanel(JAXXContext context) {
        SearchUI ui = getUI(context);
        CriteriaUI criterias = ui.getCriterias();
        return criterias.getCriteriaPanel();
    }

    public CriteriaUI getCriterias(JAXXContext context) {
        return getUI(context).getCriterias();
    }

    public void initCriteria(JAXXContext context) {
        initCriteria(context, null);
    }

    public void initCriteria(JAXXContext context, List<VradiThesaurusDTO> thesaurus) {
        // Remove existing
        getCriteriaPanel(context).removeAll();

        // Adding root criteria
        VradiThesaurusDTO root = VradiContext.THESAURUS_ENTRY_DEF.getContextValue(context);
        addCriteria(context, root, false, true);

        // Adding saved criteria
        List<String> vradiListCriteria = VradiHelper.getVradiListCriteria();
        for (String value : vradiListCriteria) {
            addCriteria(context, value, true);
        }

        // Adding thesaurus suppl
        if (thesaurus != null){
            for (VradiThesaurusDTO t : thesaurus){
                addUnmodifiableCriteria(context, t);
            }
        }
    }

    public void removeCriteria(JAXXContext context, CriteriaField criteriaField) {
        getCriteriaPanel(context).remove(criteriaField);
        getUI(context).validate();
    }

    public void removeCriteriaAndPersist(JAXXContext context, TreeNode bean) {
        CriteriaField criteriaField = ((CriteriaField) context);
        VradiHelper.removeVradiListCriteria(bean);
        removeCriteria(context, criteriaField);
        getUI(context).validate();
    }

    public void addUnmodifiableCriteria(JAXXContext context, VradiThesaurusDTO thesaurusDTO) {
        CriteriaField criteriaField = addCriteria(context, thesaurusDTO.getParentThesaurus(), false, false);

        // Get combo
        JComboBox combo = criteriaField.getCombo();

        // Select thesaurus
        combo.setSelectedItem(thesaurusDTO.getBean());
    }

    public void addAndPersistCriteria(JAXXContext context, VradiThesaurusDTO thesaurus) {
        VradiHelper.addVradiListCriteria(thesaurus.getBean());
        addCriteria(context, thesaurus, true, true);
    }

    protected void addCriteria(JAXXContext context, String wikittyId, boolean deletable) {

        if (wikittyId == null || wikittyId.isEmpty()){
             return;
        }

        VradiThesaurusDTO criteriaThes = UIHelper.getHandler(context, ThesaurusHandler.class).findThesaurusInRef(context, wikittyId);

        if (criteriaThes != null) {
             addCriteria(context, criteriaThes, deletable, true);
        }
    }

    protected CriteriaField addCriteria(JAXXContext context, VradiThesaurusDTO thesCriteria, boolean deletable, boolean enabled) {
        TreeNode criteria = thesCriteria.getBean();

        // New criteria field
        CriteriaField criteriaField = new CriteriaField(context);

        // Set Name
        criteriaField.setLabelName(criteria.getName());

        // Set bean
        criteriaField.setBean(criteria);

        // Deletable
        criteriaField.setDeletable(deletable);
        criteriaField.setActif(enabled);

        // Get combo
        JComboBox combo = criteriaField.getCombo();

        // Get children treeNode
        Collection<VradiThesaurusDTO> children = thesCriteria.getChildren();
        TreeNode[] thesArray = new TreeNode[children.size() + 1];
        int i = 0;

        // Adding an empty to combo
        TreeNodeImpl empty = new TreeNodeImpl();
        empty.setName(" ");
        thesArray[i] = empty;

        // Adding children
        for (VradiThesaurusDTO child : children){
            i++;
            thesArray[i] = child.getBean();
        }
        
        // Apply model
        combo.setModel(new DefaultComboBoxModel(thesArray));
        // Apply renderer
        combo.setRenderer(VradiContext.get().getDecoratorProviderListCellRenderer());

        // Adding to ui
        criteriaField.init();

        JPanel criteriaPanel = getCriteriaPanel(context);
        criteriaPanel.add(criteriaField);

        getUI(context).validate();

        return criteriaField;
    }

    protected List<String> getDateFormType(WikittyExtension formType) {
        List<String> result = new ArrayList<String>();
        result.add(Form.FQ_FIELD_CREATIONDATE);
        result.add(Form.FQ_FIELD_DATEPUB);
        result.add(Form.FQ_FIELD_DATEPEREMPTION);
        if (formType != null) {
            for (String name : formType.getFieldNames()) {
                FieldType type = formType.getFieldType(name);
                if (type.getType() == FieldType.TYPE.DATE) {
                    result.add(formType.getName() + "." + name);
                }
            }
        }
        return result;
    }


    public DefaultComboBoxModel updateDateSelectionModel(WikittyExtension selected) {
        List<String> datas = getDateFormType(selected);
        return new DefaultComboBoxModel(datas.toArray(new String[datas.size()]));
    }


    public void executeQuery(JAXXContext context, CriteriaUI criterias, VradiFormPageDTO formPageDTO, JComboBox combo, String request) {
        if (criterias == null) {
            // Crtieria panel is hidden
            VradiHelper.executeQuery(request, null, null, null, null, null, formPageDTO);
        } else {
            // Apply criteriain search
            Component[] thesaurusCriteria = getCriteriaPanel(context).getComponents();
            List<String>[] thesaurus = new ArrayList[thesaurusCriteria.length];
            for (int i = 0; i < thesaurus.length; i++) {

                // Get selected
                TreeNode tn = (TreeNode) ((CriteriaField) thesaurusCriteria[i]).getCombo().getSelectedItem();

                // If is not empty
                if (tn != null && !tn.getName().trim().isEmpty()) {

                    // Add parent
                    thesaurus[i] = new ArrayList<String>();
                    String wikittyId = tn.getWikittyId();
                    thesaurus[i].add(wikittyId);

                    // Find thesaurus corresponding in context
                    VradiThesaurusDTO parent = UIHelper.getHandler(context, ThesaurusHandler.class).findThesaurusInRef(context, wikittyId);

                    // Get children recursivly
                    thesaurus[i].addAll(parent.getChildrenRecursif());
                }
            }
            
            Calendar beginDate = new GregorianCalendar();
            beginDate.setTime(criterias.getFromDate().getDate());
            beginDate.set(Calendar.HOUR_OF_DAY, (Integer) criterias.getFromHour().getSelectedItem());
            beginDate.set(Calendar.MINUTE, (Integer) criterias.getFromMinute().getSelectedItem());

            Calendar endDate = new GregorianCalendar();
            endDate.setTime(criterias.getToDate().getDate());
            endDate.set(Calendar.HOUR_OF_DAY, (Integer) criterias.getToHour().getSelectedItem());
            endDate.set(Calendar.MINUTE, (Integer) criterias.getToMinute().getSelectedItem());

            VradiHelper.executeQuery(request, (WikittyExtension) criterias.getFormTypeSelection().getSelectedItem(),
                    (String) criterias.getDateTypeSelection().getSelectedItem(),
                    beginDate.getTime(), endDate.getTime(), thesaurus, formPageDTO);
        }
        
        if (log.isInfoEnabled()) {
            log.info("Total forms found        : " + formPageDTO.getTotalFoundFormNb());
            log.info("Number of forms per page : " + formPageDTO.getNbFormsToShow());
            log.info("Actual page              : " + formPageDTO.getPageToShow());
        }

        if (context instanceof OfferListUI) {
            VradiHelper.addRequestToProperties(request);
        }
        
        // Update models
        if (combo != null) {
            ((DefaultComboBoxModel) combo.getModel()).removeElement(request);
            combo.insertItemAt(request, 0);
            combo.setSelectedIndex(0);
        }
    }

    public ListCellRenderer getDateTypeComboBoxRenderer() {
        return new DefaultListCellRenderer() {
            @Override
            public Component getListCellRendererComponent(JList list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
                String stringValue = null;
                if (value != null) {
                    stringValue = value.toString();
                    stringValue = stringValue.substring(stringValue.indexOf('.') + 1);
                }
                return super.getListCellRendererComponent(list, stringValue, index, isSelected, cellHasFocus);
            }
        };
    }

    public void exportToCSV(JAXXContext context) {
        SearchUI searchUI = getUI(context);
        File file = UIHelper.openFileChooser(CSV_FILE_EXTENSION, "vradi.export.filter.csv",
                searchUI, "vradi.export.validate.label");
        
        if (file != null) {
            VradiFormPageDTO formPageDTO = new VradiFormPageDTO();
            formPageDTO.setNbFormsToShow(1000 * 1000); // max 1 million
            formPageDTO.setPageToShow(1);
            OfferListTableModel tableModel = new OfferListTableModel(formPageDTO);
            
            if (searchUI != null) {
                CriteriaUI criterias = searchUI.getCriterias();
                String request = searchUI.getRequest();
                executeQuery(context, criterias, formPageDTO, null, request);
                
            } else {
                executeQuery(context, null, formPageDTO, null, null);
            }
            
            UIHelper.exportToCSV(file, tableModel);
        }
    }
    
}

