/*
 * *##%
 * Vradi :: Swing
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.ui;

import static org.nuiton.i18n.I18n._;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Collection;
import java.util.List;

import javax.swing.JComponent;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.tree.TreePath;

import jaxx.runtime.JAXXContext;
import jaxx.runtime.context.JAXXInitialContext;
import jaxx.runtime.swing.ErrorDialogUI;
import jaxx.runtime.swing.navigation.NavigationTreeNode;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.sharengo.exceptions.TechnicalException;
import org.sharengo.wikitty.TreeNode;
import org.sharengo.wikitty.TreeNodeImpl;

import com.jurismarches.vradi.VradiContext;
import com.jurismarches.vradi.VradiHelper;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.models.ModifThesaurusModel;
import com.jurismarches.vradi.services.ServiceHelper;
import com.jurismarches.vradi.services.VradiStorageService;
import com.jurismarches.vradi.services.dto.VradiThesaurusDTO;
import com.jurismarches.vradi.ui.admin.AdminHandler;
import com.jurismarches.vradi.ui.admin.ThesaurusEditUI;
import com.jurismarches.vradi.ui.admin.ThesaurusPathChooserUI;
import com.jurismarches.vradi.ui.admin.content.AdminThesaurusUI;
import com.jurismarches.vradi.ui.search.SearchHandler;

/**
 * @author letellier
 */
public class ThesaurusHandler {

    /**
     * to use log facility, just put in your code: log.info(\"...\");
     */
    static private Log log = LogFactory.getLog(ThesaurusHandler.class);

    static public String PREFIX_THESAURUS = "thesaurus";
    static public String PREFIX_EDIT = "edit";
    static public String PREFIX_MOVE = "move";

    /**
     * Display thesaurusUI with rootThesaurus for one form
     *
     * @param rootContext   context
     * @param rootThesaurus root node
     * @param bean          form associated
     * @return thesaurusUI
     */
    public ThesaurusUI initUI(JAXXContext rootContext, VradiThesaurusDTO rootThesaurus, Form bean) {

        return initUI(rootContext, rootThesaurus, bean, false);
    }

    /**
     * Display thesaurusUI with rootThesaurus for one form and can be removed with parameter deletable
     *
     * @param rootContext   context
     * @param rootThesaurus root node
     * @param bean          form associated
     * @param deletable     true if ui can be removed
     * @return thesaurusUI
     */
    public ThesaurusUI initUI(JAXXContext rootContext, VradiThesaurusDTO rootThesaurus, Form bean, boolean deletable) {

        JAXXInitialContext context = new JAXXInitialContext().add(rootContext);

        // show main ui
        context.add(this);

        // Init ui
        ThesaurusUI ui = new ThesaurusUI(context, rootThesaurus);
        ui.setDeletable(deletable);
        ui.setBean(bean);

        return ui;
    }

    /**
     * Init thesaurus in main context
     */
    public void initThesaurusDef() {
        VradiStorageService service = ServiceHelper.getVradiStorageService();
        TreeNode rootThesaurus = null;
        try {
            rootThesaurus = service.getRootThesaurus();
        } catch (TechnicalException e) {
            log.error("Cant get root thesaurus : ", e);
        }
        VradiThesaurusDTO vradiThesaurusDTO = new VradiThesaurusDTO();
        vradiThesaurusDTO.fromWikitty(rootThesaurus);
        VradiContext.THESAURUS_ENTRY_DEF.setContextValue(VradiContext.get(), vradiThesaurusDTO);

        if (log.isDebugEnabled()){
            vradiThesaurusDTO.addPropertyChangeListener(new PropertyChangeListener(){
                @Override
                public void propertyChange(PropertyChangeEvent evt) {
                    log.debug("Property " + evt.getPropertyName() +
                            " of def thesaurus changed, old value : " +
                            evt.getNewValue() + " old : " + evt.getOldValue());
                }
            });
        }
    }

    /**
     * Find thesaurusUI in context
     *
     * @param context
     * @return thesaurusUI in context
     */
    ThesaurusUI getUI(JAXXContext context) {
        if (context instanceof ThesaurusUI) {
            return (ThesaurusUI) context;
        }
        return null;
    }

    public VradiThesaurusDTO findThesaurusInRef(JAXXContext context, String wikityId){
        VradiThesaurusDTO root = VradiContext.THESAURUS_ENTRY_DEF.getContextValue(context);
        return root.findThesaurus(wikityId);
    }

    /**
     * Get thesaurus panel from OfferEditUI
     *
     * @param context
     * @return thesaurus panel
     */
    protected JPanel getThesaurusContainer(JAXXContext context) {
        return getUI(context).getParentContainer(OfferEditUI.class).getThesaurus();
    }

    /**
     * Get OfferEditUI
     *
     * @param context
     * @return offer edit panel
     */
    protected OfferEditUI getEditUI(JAXXContext context) {
        return UIHelper.getHandler(context, OfferEditHandler.class).getUI(context);
    }

    /**
     * Get OfferListUI
     *
     * @param context
     * @return offer list panel
     */
    protected OfferListUI getListUI(JAXXContext context) {
        return UIHelper.getHandler(context, OfferListHandler.class).getUI(context);
    }

    /**
     * Get parent node
     *
     * @param helper
     * @param path
     * @return parent node
     */
    protected NavigationTreeNode getParentNode(ThesaurusTreeHelper helper, TreePath path) {
        if (path == null) {
            return helper.getRootNode();
        } else {
            return (NavigationTreeNode) path.getLastPathComponent();
        }
    }

    /**
     * Add criteria in search panel and save it in properties
     *
     * @param context
     * @param helper
     * @param path
     */
    public void addCriteria(JAXXContext context, ThesaurusTreeHelper helper, TreePath path) {
        OfferListUI ui = getListUI(context);

        // Get selected thesaurus
        NavigationTreeNode parentNode = getParentNode(helper, path);
        VradiThesaurusDTO thesaurus = (VradiThesaurusDTO) parentNode.getBean(context);

        // Adding to offerListUI
        UIHelper.getHandler(context, SearchHandler.class).addAndPersistCriteria(context, thesaurus);

        ui.validate();
    }

    /**
     * Refresh all thesaurus displayed
     *
     * @param context
     */
    public void refreshAllThesaurus(JAXXContext context) {
        List<OfferEditUI> editUIs = VradiContext.OFFERT_EDIT_UI_ENTRY_DEF.getContextValue(VradiContext.get());
        if (log.isDebugEnabled()) {
            log.debug("Refresh ui : " + (editUIs == null ? "is null" : editUIs.size()));
        }
        if (editUIs != null) {
            for (OfferEditUI editUI : editUIs) {
                if (log.isDebugEnabled()) {
                    log.debug("Refresh ui : " + editUI.getData().getObjet());
                }
                OfferEditHandler offerEditHandler = UIHelper.getHandler(context, OfferEditHandler.class);
                offerEditHandler.initThesaurus(context, editUI);

                editUI.revalidate();
            }
        }
    }

    /**
     * Add thesaurus panel not removable in offerEdit and save it in properties
     *
     * @param context
     * @param path
     */
    public void addThesaurusUI(JAXXContext context, TreePath path) {
        if (path != null) {
            // Get selected thesaurus
            NavigationTreeNode parentNode = (NavigationTreeNode) path.getLastPathComponent();

            // Creating new root
            VradiThesaurusDTO thesaurusParent = (VradiThesaurusDTO) parentNode.getBean(context);

            addThesaurusUI(context, thesaurusParent, true, true);
        }
    }

    /**
     * Add thesaurus panel in offerEdit
     *
     * @param context
     * @param root          root thesaurus displayed
     * @param deletable     if ui can be removed
     * @param save          to save in properties
     */
    public void addThesaurusUI(JAXXContext context, VradiThesaurusDTO root, boolean deletable, boolean save) {
        if (root != null) {
            JAXXInitialContext intitialContext = new JAXXInitialContext().add(context);
            ThesaurusUI thesaurusChildUI = new ThesaurusUI(intitialContext, root);

            thesaurusChildUI.setDeletable(deletable);

            getThesaurusContainer(context).add(thesaurusChildUI);

            // Save in properties
            if (save) {
                VradiHelper.addVradiListThesaurus(root.getBean());
            }

            getUI(context).revalidate();
        }
    }

    /**
     * Add colomn in OfferList result table
     *
     * @param context
     * @param helper
     * @param path
     */
    public void addColumn(JAXXContext context, ThesaurusTreeHelper helper, TreePath path) {
        // Get selected thesaurus
        NavigationTreeNode node = getParentNode(helper, path);

        // Creating new root
        VradiThesaurusDTO thesaurus = (VradiThesaurusDTO) node.getBean(context);

        // Save in properties
        VradiHelper.addVradiListThesaurus(thesaurus.getBean());
    }

    /**
     * Remove thesaurus ui and save it in properties
     *
     * @param thesaurusUI   to removed
     */
    public void closeThesaurus(ThesaurusUI thesaurusUI) {
        getThesaurusContainer(thesaurusUI).remove(thesaurusUI);

        // Save in properties
        VradiThesaurusDTO selected = (VradiThesaurusDTO) thesaurusUI.helper.getRootNode().getBean();
        VradiHelper.removeVradiListThesaurus(selected.getBean());

        refreshAllThesaurus(thesaurusUI);
    }

    /**
     * Answer to save thesaurus modification
     *
     * @param content   to be saved
     * @return if it's done
     */
    public boolean answerToSaveBeforeQuit(AdminThesaurusUI content) {
        if (content != null) {

            log.info("current content  to close "
                    + content.getClass().getSimpleName() + " : " + content
                    .isModified());

            String msgs = content.getModifModel().getAllMsgs();

            if (content.isModified()) {
                int n = JOptionPane.showConfirmDialog(content,
                        _("vradi.admin.saveAnswer", msgs),
                        _("vradi.admin.saveTitle"),
                        JOptionPane.YES_NO_CANCEL_OPTION);
                switch (n) {
                    case JOptionPane.NO_OPTION:
                        // annule les modifications
                        cancelThesaurus(content, content.getHelper(), false);
                        break;
                    case JOptionPane.OK_OPTION:
                        // sauvegarde les modifications
                        saveThesaurus(content, content.getHelper(), false);
                        break;
                    case JOptionPane.CANCEL_OPTION:
                        // annule le changement d'onglet
                        return false;

                }
            }
        }
        return true;
    }

    /**
     * Return helper selected node
     *
     * @param context
     * @param helper
     * @return selected node
     */
    protected NavigationTreeNode getSelectedNode(JAXXContext context,
                                                 ThesaurusTreeHelper helper) {
        // Get selected thesaurus
        NavigationTreeNode selectedNode = helper.getSelectedNode(context);
        if (selectedNode == null) {
            selectedNode = helper.getRootNode();
        }
        return selectedNode;
    }

    /**
     * Add new thesaurus to selected node
     *
     * @param context
     * @param helper
     * @param modifThesaurusModel    list of modif viewer
     * @return if it's done
     */
    public boolean addChildThesaurusToSelected(JAXXContext context,
                                               ThesaurusTreeHelper helper,
                                               ModifThesaurusModel modifThesaurusModel) {

        NavigationTreeNode parentNode = getSelectedNode(context, helper);
        if (parentNode == null) {
            parentNode = helper.getRootNode();
        }
        VradiThesaurusDTO created = addChildThesaurus(context, helper, parentNode);

        if (created == null){
            return false;
        }
        // Add created item in modif viewer
        modifThesaurusModel.addCreateModif(created);

        return true;
    }

    /**
     * Add new thesaurus to parent node in param
     *
     * @param context
     * @param helper
     * @param parentNode
     * @return ThesaurusCreated
     */
    protected VradiThesaurusDTO addChildThesaurus(JAXXContext context,
                                        ThesaurusTreeHelper helper,
                                        NavigationTreeNode parentNode) {
        // Add new to parent
        VradiThesaurusDTO thesaurusParent = (VradiThesaurusDTO) parentNode
                .getBean(context);

        // Creating child
        TreeNode newTreeNode = new TreeNodeImpl();

        // Add child to parent thesaurus
        VradiThesaurusDTO newThesaurus = new VradiThesaurusDTO();
        newThesaurus.fromWikitty(newTreeNode);
        newThesaurus.setParentThesaurus(thesaurusParent);

        ThesaurusEditUI edit = new ThesaurusEditUI(context);

        edit.setBean(newThesaurus);
        attachListener(edit, newThesaurus);
        edit.setHelper(helper);
        edit.setTitleThesaurus(_("vradi.adminThesaurus.creationTitle"));

        if (log.isDebugEnabled()) {
            log.debug("Creating thesaurus " + newThesaurus.getName());
        }

        edit.setVisible(true);

        // If name is not empty
        if (edit.isSave()) {
            String name = edit.getName();
            if (name != null && !name.isEmpty()) {
                newThesaurus.setParentThesaurus(thesaurusParent);
                thesaurusParent.addChild(newThesaurus);

                // Mark to save
                newThesaurus.setToCreate(true);

                // Add in tree
                if (log.isDebugEnabled()) {
                    log.debug("Adding thesaurus " + newThesaurus.getName()
                            + " to parent " + parentNode);
                }
                NavigationTreeNode newNode = helper.getBuilder()
                        .addThesaurus(context, parentNode, newThesaurus);

                // Expend
                helper.expendNode(context, parentNode);

                return newThesaurus;
            }
        }
        return null;
    }

    /**
     * Edit selected thesaurus
     *
     * @param context
     * @param helper
     * @param modifThesaurusModel    list of modif viewer
     * @return if it's done
     */
    public boolean editThesaurusSelected(JAXXContext context,
                                         ThesaurusTreeHelper helper,
                                         ModifThesaurusModel modifThesaurusModel) {
        NavigationTreeNode selectedNode = getSelectedNode(context, helper);

        VradiThesaurusDTO selected = ((VradiThesaurusDTO) selectedNode
                .getBean(context));

        ThesaurusEditUI edit = new ThesaurusEditUI(context);

        attachListener(edit, selected, modifThesaurusModel);
        edit.setBean(selected);
        edit.setHelper(helper);
        edit.setModifModel(modifThesaurusModel);
        edit.setTitleThesaurus(
                _("vradi.adminThesaurus.editTitle", selected.getName()));

        if (log.isDebugEnabled()) {
            log.debug("Editing thesaurus " + (selected == null ? " null "
                    : selected.getName()));
        }

        edit.setVisible(true);

        // If name is not empty
        if (edit.isSave()) {

            // Tag to delete
            selected.setToSave(true);

            // Get new seleted (when moved)
            selectedNode = getSelectedNode(edit, helper);

            // Repaint an expend
            helper.repaintNode(context, selectedNode);
            helper.expendNode(context, selectedNode);

            // Update list of change
            modifThesaurusModel.confirmAttributeModif();

            return true;
        }
        // Cancel
        modifThesaurusModel.cancelAttributeModif();
        return false;
    }

    /**
     * Delete selected thesaurus
     *
     * @param context
     * @param helper
     * @param modifThesaurusModel    list of modif viewer
     * @return if it's done
     */
    public boolean deleteThesaurusSelected(JAXXContext context,
                                           ThesaurusTreeHelper helper,
                                           ModifThesaurusModel modifThesaurusModel) {

        // Get selected node
        NavigationTreeNode selectedNode = getSelectedNode(context, helper);

        // Get selected thesaurus
        VradiThesaurusDTO toDelete = (VradiThesaurusDTO) selectedNode.getBean(context);

        // Add delete item in modif viewer
        modifThesaurusModel.addDeleteModif(toDelete);

        // Delete
        return deleteThesaurus(context, helper, selectedNode, false);
    }

    /**
     * Delete selected thesaurus
     *
     * @param context
     * @param helper
     * @param ask       if confirm dialog box must be displayed
     * @return if it's done
     */
    public boolean deleteThesaurusSelected(JAXXContext context,
                                           ThesaurusTreeHelper helper,
                                           boolean ask) {
        NavigationTreeNode selectedNode = getSelectedNode(context, helper);

        return deleteThesaurus(context, helper, selectedNode, ask);
    }

    /**
     * Delete thesaurus in param
     *
     * @param context
     * @param helper
     * @param nodeToDelete  node to delete
     * @param ask           if confirm dialog box must be displayed
     * @return if it's done
     */
    public boolean deleteThesaurus(JAXXContext context,
                                   ThesaurusTreeHelper helper,
                                   NavigationTreeNode nodeToDelete,
                                   boolean ask) {

        VradiThesaurusDTO selected = (VradiThesaurusDTO) nodeToDelete
                .getBean(context);

        int result = JOptionPane.CANCEL_OPTION;
        if (ask) {
            result = JOptionPane.showConfirmDialog(getUI(context),
                    _("vradi.thesaurus.askdeleteThesaurus") + selected
                            .getRecursifName());
        } else {
            result = JOptionPane.OK_OPTION;
        }

        if (result == JOptionPane.OK_OPTION) {

            selected.setToDelete(true);

            helper.getBuilder().removeNodeFromParent(context, nodeToDelete);

            // Selection root node
            AdminThesaurusUI ui = (AdminThesaurusUI)context;
            NavigationTreeNode rootNode = ui.getHelper().getRootNode();
            ui.getHelper().setSelectedNode(context, rootNode);

            return true;
        }
        return false;
    }

    /**
     * Delete thesaurus in base
     *
     * @param toDelete  thesaurus to delete
     */
    protected void deleteTreeNode(VradiThesaurusDTO toDelete) {
        try {
            ServiceHelper.getVradiStorageService()
                    .deleteEntity(toDelete.getBean());
        } catch (TechnicalException e) {
            log.error("Cant delete thesaurus : " + toDelete.getName() + " id : "
                    + toDelete.getWikittyId(), e);
            ErrorDialogUI.showError(e);
        }
    }

    protected void attachListener(final ThesaurusEditUI ui,
                                  final VradiThesaurusDTO thesaurus) {
        attachListener(ui, thesaurus, null);
    }
    protected void attachListener(final ThesaurusEditUI ui,
                                  final VradiThesaurusDTO thesaurus, final ModifThesaurusModel modifThesaurusModel) {
        thesaurus.addPropertyChangeListener(new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                ui.setModified(true);
                if (modifThesaurusModel != null){
                    Object oldValue = evt.getOldValue();
                    if (oldValue instanceof String){
                        modifThesaurusModel.prepareAttributeModif(thesaurus, evt.getPropertyName(), (String) oldValue);
                    }
                }
            }
        });
    }

    /**
     * Cancel editing thesaurus
     *
     * @param context
     * @param helper
     * @param trigerredByCancelButton   source of cancel
     */
    public void cancelThesaurus(JAXXContext context, ThesaurusTreeHelper helper,
                                boolean trigerredByCancelButton) {
        int confirm = -1;
        if(trigerredByCancelButton) {
            confirm = JOptionPane.showConfirmDialog((JComponent)context,
                    _("vradi.admin.cancel.message"),
                    _("vradi.admin.cancel.title"),
                    JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE);
        }

        if(!trigerredByCancelButton || confirm == JOptionPane.YES_OPTION) {
            // Reinit thesaurus ui
            helper = new ThesaurusTreeHelper(context, PREFIX_EDIT);
            AdminThesaurusUI ui = (AdminThesaurusUI) context;
            ui.reinit(helper);
        }
    }

    /**
     * Save edited thesaurus
     *
     * @param context
     * @param helper
     */
    public void saveThesaurus(JAXXContext context, ThesaurusTreeHelper helper, boolean answer) {
        if (answer) {
            AdminThesaurusUI ui = (AdminThesaurusUI) context;

            String msgs = ui.getModifModel().getAllMsgs();

            int n = JOptionPane.showConfirmDialog(ui,
                    _("vradi.admin.saveThesaurusAnswer", msgs),
                    _("vradi.admin.saveTitle"),
                    JOptionPane.YES_NO_OPTION);
            
            if (n == JOptionPane.NO_OPTION){
                return;
            }
        }

        saveThesaurus(helper.getRootThesaurus());

        // Reinit thesaurus ENTRY def
        initThesaurusDef();

        AdminThesaurusUI ui = (AdminThesaurusUI) context;
        ui.setModified(false);
        ui.reinit(helper);

        // Refresh criteria
        UIHelper.getHandler(context, SearchHandler.class).initCriteria(context);

        refreshAllThesaurus(context);
    }

    /**
     * Save thesaurus in param
     *
     * @param toSave
     */
    protected void saveThesaurus(VradiThesaurusDTO toSave) {
        // Get bean
        TreeNode bean = toSave.getBean();

        // To delete
        if (toSave.isToDelete()) {
            // Delete
            deleteTreeNode(toSave);
            if (log.isDebugEnabled()) {
                log.debug("Saving thesaurus : " + bean.getName());
            }
            // Remove from parent
            VradiThesaurusDTO parent = toSave.getParentThesaurus();
            parent.removeChild(toSave);
        }
        // To save
        if (toSave.isToSave() || toSave.isToCreate()) {
            // Save
            try {
                // Set parent to child
                VradiThesaurusDTO parentThesaurus = toSave.getParentThesaurus();

                // If it's not root
                if (parentThesaurus != null) {
                    TreeNode parent = parentThesaurus.getBean();
                    bean.setParent(parent.getWikittyId());
                }

                // Update
                bean = (TreeNode) ServiceHelper.getVradiStorageService().updateEntity(bean);
                if (log.isDebugEnabled()) {
                    log.debug("Saving thesaurus : " + bean.getName());
                }
            } catch (TechnicalException ex) {
                log.error("Cant save thesaurus : " + bean.getName() + " id : "
                        + bean.getWikittyId() + " caused by : ", ex);
                ErrorDialogUI.showError(ex);
            }
            // Its save
            toSave.setToSave(false);
            toSave.setToCreate(false);
        }
        Collection<VradiThesaurusDTO> children = toSave.getChildren();
        if (children != null) {
            for (VradiThesaurusDTO child : children) {
                if (child != null) {
                    saveThesaurus(child);
                }
            }
        }
    }

    /**
     * Move selected thesaurus
     *
     * @param context
     * @param helperFrom    helper of adminThesaurus
     * @param modifThesaurusModel    list of modif viewer
     * @return is modified
     */
    public boolean moveThesaurus(JAXXContext context,
                              ThesaurusTreeHelper helperFrom,
                              ModifThesaurusModel modifThesaurusModel) {

        // Get thesaurus to move
        VradiThesaurusDTO toMove = (VradiThesaurusDTO) helperFrom.getSelectedBean(context);

        // Move
        return moveThesaurus(context, helperFrom, toMove, modifThesaurusModel);
    }
    /**
     * Move selected thesaurus
     *
     * @param context
     * @param helperFrom    helper of adminThesaurus
     * @param beanToMove    bean to move
     * @return is modified
     */
    public boolean moveThesaurus(JAXXContext context,
                              ThesaurusTreeHelper helperFrom,
                              VradiThesaurusDTO beanToMove,
                              ModifThesaurusModel modifThesaurusModel) {


        // Get node to move
        NavigationTreeNode nodeToMove = getSelectedNode(context, helperFrom);

        // Get root thesaurus
        VradiThesaurusDTO rootThesaurus = helperFrom.getRootThesaurus().clone();

        // Remove own branch
        boolean isDeleted = rootThesaurus.removeThesaurusRecursivly(beanToMove);
        if (log.isDebugEnabled()) {
            if (isDeleted) {
                log.debug("Thesaurus " + beanToMove.getName() + " is deleted");
            } else {
                log.debug("Thesaurus " + beanToMove.getName()
                        + " is not deleted (not found)");
            }
        }

        // Create new helper
        ThesaurusTreeHelper selectHelper = new ThesaurusTreeHelper(context,
                rootThesaurus, PREFIX_MOVE);

        // Create select ui
        ThesaurusPathChooserUI selectUI = new ThesaurusPathChooserUI(context,
                selectHelper);

        selectUI.setVisible(true);

        // If select
        if (selectUI.isSaved()) {
            // Get parent selected
            NavigationTreeNode nodeParentSelected = getSelectedNode(selectUI,
                    selectHelper);

            // Get corresponding parent selected node
            VradiThesaurusDTO beanParentSelected = (VradiThesaurusDTO)nodeParentSelected.getBean();
            String parentSelectedPath = beanParentSelected.getPath(PREFIX_EDIT, helperFrom.getPathSeparator());
            if (log.isDebugEnabled()){
                log.debug("Searching node whith path : " + parentSelectedPath);
            }

            // Add modif item in modif viewer
            modifThesaurusModel.addMoveModif(beanToMove,
                    beanToMove.getParentThesaurus()
                            .getNamePath(helperFrom.getPathSeparator()),
                    beanToMove.getParentThesaurus()
                            .getPath(PREFIX_EDIT, helperFrom.getPathSeparator()),
                    parentSelectedPath);

            NavigationTreeNode nodeParentSelectedFrom = helperFrom
                    .findNode(context, parentSelectedPath);

            if (log.isDebugEnabled()) {
                log.debug("Selected node : " + nodeParentSelected.getFullPath()
                        + " finded node : " + nodeParentSelectedFrom
                        .getFullPath());
            }

            // Delete in tree
            deleteThesaurus(context, helperFrom, nodeToMove, false);
            beanToMove.setToDelete(false);

            // Remove from parent
            VradiThesaurusDTO oldParent = beanToMove.getParentThesaurus();
            oldParent.removeChild(beanToMove);

            // Get corresponding parent selected bean
            VradiThesaurusDTO thesaurusParentFrom
                    = (VradiThesaurusDTO) nodeParentSelectedFrom
                    .getBean(context);

            // Attach child
            thesaurusParentFrom.addChild(beanToMove);
            beanToMove.setParentThesaurus(thesaurusParentFrom);

            // Mark to save
            beanToMove.setToSave(true);

            // Add in tree
            NavigationTreeNode newNode = helperFrom.getBuilder()
                    .addThesaurusAndChildrenRecursivly(nodeParentSelectedFrom, beanToMove);

            // Expend
            helperFrom.selectNode(context, nodeParentSelectedFrom);

            // TODO : Binding dont work
            if (context instanceof ThesaurusEditUI){
                ((ThesaurusEditUI) context).getPathField()
                    .setText(beanToMove.getParentPath(helperFrom.getPathSeparator()));
            }

            return true;
        }
        return false;
    }

    public void viewApply(JAXXContext context, List<VradiThesaurusDTO> concerned){
        UIHelper.getHandler(context, AdminHandler.class).openRequest(context, concerned);
    }
}
