/*
 * *##%
 * Vradi :: Swing
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.ui;

import java.awt.Component;
import java.io.File;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

import javax.swing.*;

import jaxx.runtime.JAXXContext;
import jaxx.runtime.context.JAXXInitialContext;
import jaxx.runtime.swing.ErrorDialogUI;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.sharengo.exceptions.TechnicalException;
import org.sharengo.wikitty.BusinessEntity;
import org.sharengo.wikitty.FieldType;
import org.sharengo.wikitty.TreeNode;
import org.sharengo.wikitty.WikittyExtension;

import com.jurismarches.vradi.VradiContext;
import com.jurismarches.vradi.VradiHelper;
import com.jurismarches.vradi.entities.FieldTypeEnum;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.ModificationTag;
import com.jurismarches.vradi.entities.Status;
import com.jurismarches.vradi.entities.VradiUser;
import com.jurismarches.vradi.services.ServiceHelper;
import com.jurismarches.vradi.services.dto.VradiThesaurusDTO;
import com.jurismarches.vradi.ui.editors.CurrencyEditor;
import com.jurismarches.vradi.ui.editors.DateEditor;
import com.jurismarches.vradi.ui.editors.DateTimeEditor;
import com.jurismarches.vradi.ui.editors.EmailEditor;
import com.jurismarches.vradi.ui.editors.NumEditor;
import com.jurismarches.vradi.ui.editors.StringEditor;
import com.jurismarches.vradi.ui.editors.TextEditor;
import com.jurismarches.vradi.ui.editors.UrlEditor;
import com.jurismarches.vradi.ui.editors.VradiEditor;

/**
 * @author letellier
 */
public class OfferEditHandler {

    /**
     * Logger
     */
    static private final Log log = LogFactory.getLog(OfferEditHandler.class);
    private static final String DEFAULT_EDITOR_NAME = "$default";
    public static final String REQUIRED_FIELD_CONTEXT = "required";

    /**
     * Methode pour initialiser l'ui principale sans l'afficher.
     *
     * @param rootContext le context applicatif
     * @param data
     * @return l'ui instancie et initialisee mais non visible encore
     */
    public OfferEditUI initUI(JAXXContext rootContext, Form data) {
        JAXXInitialContext context = new JAXXInitialContext().add(rootContext).add(this);

        // show main ui
        context.add(data);
        UIHelper.getHandler(context, ThesaurusHandler.class);
        OfferEditUI ui = new OfferEditUI(context);

        fillfields(ui, data);

        // Adding in referenciel
        List<OfferEditUI> editUIs = VradiContext.OFFERT_EDIT_UI_ENTRY_DEF.getContextValue(VradiContext.get());
        if (editUIs == null) {
            editUIs = new ArrayList<OfferEditUI>();
            VradiContext.OFFERT_EDIT_UI_ENTRY_DEF.setContextValue(VradiContext.get(), editUIs);
        }
        VradiContext.OFFERT_EDIT_UI_ENTRY_DEF.getContextValue(VradiContext.get()).add(ui);

        // Init thesaurus
        initThesaurus(context, ui);

        // Init propositions
        try {
            List<TreeNode> thesaurusToPropose =
                    ServiceHelper.getVradiStorageService().proposeThesaurus(data, null);
            addPropositions(ui, thesaurusToPropose.toArray(
                    new TreeNode[thesaurusToPropose.size()]));
            
        } catch(TechnicalException eee) {
            log.error("Cant get propositions of thesaurus ", eee);
            ErrorDialogUI.showError(eee);
        }

        if (log.isDebugEnabled()) {
            log.debug("There is " + VradiContext.OFFERT_EDIT_UI_ENTRY_DEF.getContextValue(VradiContext.get()).size() + " OfferEditUI in referenciel");
        }

        return ui;
    }

    public void initThesaurus(JAXXContext context, OfferEditUI ui) {
        // Remove all thsuaurus
        JPanel thesaurusPanel = ui.getThesaurus();
        thesaurusPanel.removeAll();

        ThesaurusHandler thesaurusHandler = UIHelper.getHandler(context, ThesaurusHandler.class);

        List<String> thesaurusId = VradiHelper.getVradiListThesaurus();

        // Get rootThesaurus
        VradiThesaurusDTO rootThesaurus = VradiContext.THESAURUS_ENTRY_DEF.getContextValue(context);

        // Display root thesaurus UI
        thesaurusPanel.add(thesaurusHandler.initUI(context, rootThesaurus, ui.getData(), false));

        // If no properties found and if root dont hav children display root thesaurus
        List<VradiThesaurusDTO> thesaurusSaved = new ArrayList<VradiThesaurusDTO>();

        // Find saved thesaurus
        if (thesaurusId != null) {
            for (String id : thesaurusId) {
                VradiThesaurusDTO savedThesaurus = rootThesaurus.findThesaurus(id);
                if (savedThesaurus != null) {
                    thesaurusSaved.add(savedThesaurus);
                }
            }
        }

        // Display saved thesaurus
        for (VradiThesaurusDTO saved : thesaurusSaved) {
            thesaurusPanel.add(thesaurusHandler.initUI(context, saved, ui.getData(), true));
        }
    }

    protected void fillfields(OfferEditUI ui, Form data) {
        Collection<WikittyExtension> extensions = data.getExtensions();
        for (WikittyExtension extension : extensions) {
            if (!extension.equals(ModificationTag.MODIFICATION_TAG)
                    && !extension.getName().equals(Form.EXT_INFOGENE)
                    && !extension.getName().equals(Form.EXT_FORM)) {
                
                List<String> fieldNames = extension.getFieldNames();
                for (String fieldName : fieldNames) {
                    createField(ui, extension, fieldName, data);
                }
            }
        }
    }

    protected static Map<String, Class<? extends VradiEditor<?, ?>>> editorMapping;

    protected void createField(OfferEditUI ui, WikittyExtension extension, String fieldName, BusinessEntity data) {
        JPanel content = ui.getContent();
        FieldType fieldType = extension.getFieldType(fieldName);
        
        if (!fieldType.isCollection()) {
            VradiEditor editor = getEditor(fieldType, ui);
            
            if (editor != null) {
                String extensionName = extension.getName();
                Object value = data.getField(extensionName, fieldName);
                editor.setFieldName(fieldName);
                
                String fieldDesc = VradiHelper.getFieldTypeDescription(fieldType);
                editor.setFieldDescription(fieldDesc);
                
                editor.setExtensionName(extensionName);
                editor.setExtensionFieldName(fieldName);
                
                editor.setValue(value);
                
                if (fieldType.isNotNull()) {
                    editor.setValidatorContext(REQUIRED_FIELD_CONTEXT);
                }
                content.add(editor);
            }
        }
    }

    OfferEditUI getUI(JAXXContext context) {
        if (context instanceof OfferEditUI) {
            return (OfferEditUI) context;
        }
//        OfferEditUI ui = VradiContext.OFFERT_EDIT_UI_ENTRY_DEF.getContextValue(context);
        return null;
    }

    public void openRequestPopup(JAXXContext context, List<String> queries) {
        RequestFormViewHandler handler = UIHelper.getHandler(context, RequestFormViewHandler.class);
        RequestFormViewUI ui = handler.initUI(context);
        for (String query : queries) {
            ui.getContent().add(new JLabel(query));
        }
        ui.setVisible(true);
    }

    protected void addPropositions(JAXXContext context, TreeNode[] props) {
        OfferEditUI ui = getUI(context);
        DefaultListModel propositionListModel = ui.propositionListModel;
        propositionListModel.removeAllElements();
        for (TreeNode p : props) {
            if (log.isDebugEnabled()){
                log.debug("Add proposition element : " + p);
            }
            VradiThesaurusDTO thesaurus = UIHelper.getHandler(context, ThesaurusHandler.class)
                    .findThesaurusInRef(context, p.getWikittyId());
            propositionListModel.addElement(thesaurus);
        }
    }

    protected void save(JAXXContext context, Form form) {
        log.info("Saving Form: " + form.getObjet());
        try {
            // Decrement forms by thesaurus
            Set<String> listThesaurus = form.getThesaurus();
            ThesaurusHandler thesaurusHandler = UIHelper.getHandler(context, ThesaurusHandler.class);
            if (listThesaurus != null){
                for (String id : listThesaurus) {
                    VradiThesaurusDTO thesaurus = thesaurusHandler.findThesaurusInRef(context, id);
                    thesaurus.decFormsForThesaurus();
                }
            }
    
            // Remove all thesaurus
            form.clearThesaurus();

            // Find all propositions selected
            for (Object s : getUI(context).getPropositionList().getSelectedValues()){
                VradiThesaurusDTO selected = (VradiThesaurusDTO)s;

                // Save thesaurus
                form.addThesaurus(selected.getWikittyId());

                // Increment nb offer by thesaurus
                selected.incFormsForThesaurus();
            }

            // Find all thesaurus selected
            for (Component c : getUI(context).getThesaurus().getComponents()) {
                ThesaurusUI thesaurusUI = (ThesaurusUI) c;
                DefaultListModel model = thesaurusUI.getThesaurusSelectedModel();
                Object[] selected = model.toArray();
    
                for (Object s : selected) {
                    TreeNode treeNode = (TreeNode) s;
                    log.debug(treeNode.getChildren());
                    if (log.isDebugEnabled()) {
                        log.debug("treeNode to save : " + treeNode.getName() + " in form : " + form.getObjet());
                    }
                    // Save thesaurus
                    form.addThesaurus(treeNode.getWikittyId());
    
                    // Increment nb offer by thesaurus
                    VradiThesaurusDTO thesaurus = thesaurusHandler.findThesaurusInRef(context, treeNode.getWikittyId());
                    if(thesaurus != null) {
                        thesaurus.incFormsForThesaurus();
                    }
                }
            }
            
            // Save dynamic content
            OfferEditUI offerEditUI = getUI(context);
            JPanel content = offerEditUI.getContent();
            Component[] components = content.getComponents();
            for (Component c : components) {
                VradiEditor editor = (VradiEditor) c;
                updateFormField(editor, form);
            }
            
            // Save infogene content
            updateFormField(offerEditUI.getObjetEditor(), form);
            updateFormField(offerEditUI.getIdEditor(), form);
            updateFormField(offerEditUI.getSourceTextEditor(), form);
            updateFormField(offerEditUI.getSourceURLEditor(), form);
            updateFormField(offerEditUI.getEntityEditor(), form);
            updateFormField(offerEditUI.getCountryEditor(), form);
            updateFormField(offerEditUI.getDepartmentEditor(), form);
            updateFormField(offerEditUI.getDatePubEditor(), form);
            updateFormField(offerEditUI.getDatePeremeptionEditor(), form);
            updateFormField(offerEditUI.getDescriptionEditor(), form);
            
            if (!form.getExtensions().contains(ModificationTag.MODIFICATION_TAG)) {
                form.addExtension(ModificationTag.MODIFICATION_TAG);
            }
            
            VradiUser vradiUser = context.getContextValue(VradiUser.class);
            form.setField(ModificationTag.EXT_MODIFICATION_TAG, ModificationTag.FIELD_LAST_MODIFIER,
                    vradiUser != null ? vradiUser.getName() : null);
            form.setField(ModificationTag.EXT_MODIFICATION_TAG, ModificationTag.FIELD_LAST_MODIFIED, new Date());
        
            ServiceHelper.getVradiStorageService().updateForm(form);
        } catch (Exception eee) {
            log.error("Cant save form : " + form.getObjet(), eee);
            ErrorDialogUI.showError(eee);
        }
    }

    void updateFormField(VradiEditor editor, Form form) {
        String extensionName = editor.getExtensionName();
        String extensionFieldName = editor.getExtensionFieldName();
        Object value = editor.getValue();
        if (log.isDebugEnabled()) {
            log.debug("updating: " + extensionName + "." + extensionFieldName + "=" + value);
        }
        form.setField(extensionName, extensionFieldName, value);
    }
    
    protected void update(JAXXContext context) {
        VradiMainUIHandler rootHandler = VradiContext.get().getContextValue(VradiMainUIHandler.class);
        rootHandler.goToHome(context);
    }

    protected void test(JAXXContext context, Form form) {
        List<String> queries = new ArrayList<String>();
        try {
            queries = ServiceHelper.getVradiStorageService().getQueriesReturningForm(form);
        } catch (TechnicalException eee) {
            log.error(eee);
            ErrorDialogUI.showError(eee);
        }
        openRequestPopup(context, queries);
    }

    // TODO
    protected void addFile(JAXXContext context) {
        JFileChooser fileChooser = new JFileChooser();
        fileChooser.showOpenDialog(getUI(context));

        File file = fileChooser.getSelectedFile();
//        getUI(context).fileList.addItem(new Item(file.getName(), file.getName(), file.getPath(), false));
    }

    // TODO
    protected void removeFile(JAXXContext context) {
//        java.util.List<Item> itemsToRemove = getUI(context).fileList.getSelectedItems();
//        getUI(context).fileList.removeAllItems(itemsToRemove);
    }

//    public void addThesaurus(JAXXContext context, VradiThesaurusDTO thesToAdd) {
//        for (Component c : getUI(context).getThesaurus().getComponents()) {
//            ThesaurusUI ui = (ThesaurusUI) c;
//            ThesaurusTreeHelper helper = ui.helper;
//            NavigationTreeNode rootNode = helper.getRootNode();
//            VradiThesaurusDTO root = (VradiThesaurusDTO) rootNode.getBean();
//            log.info("Try to add " + thesToAdd.getName() + " to " + root.getName());
//            if (root.addChildRecursif(thesToAdd)) {
//                log.info("succes");
//                helper.setRootThesaurus(root);
//                helper.createTreeModel(context);
//                ui.getThesaurus().setModel(helper.getTreeModel(context));
//                break;
//            }
//        }
//    }

    protected synchronized VradiEditor getEditor(FieldType type, JAXXContext context) {
        if (editorMapping == null) {
            editorMapping = new TreeMap<String, Class<? extends VradiEditor<?, ?>>>();
            editorMapping.put(FieldTypeEnum.STRING.name(), StringEditor.class);
            editorMapping.put(FieldTypeEnum.DATE_TIME.name(), DateTimeEditor.class);
            editorMapping.put(FieldTypeEnum.DATE.name(), DateEditor.class);
            editorMapping.put(FieldTypeEnum.EMAIL.name(), EmailEditor.class);
            editorMapping.put(FieldTypeEnum.NUMERIC.name(), NumEditor.class);
            editorMapping.put(FieldTypeEnum.TEXT.name(), TextEditor.class);
            editorMapping.put(FieldTypeEnum.URL.name(), UrlEditor.class);
            editorMapping.put(FieldTypeEnum.CURRENCY.name(), CurrencyEditor.class);
            editorMapping.put(DEFAULT_EDITOR_NAME, StringEditor.class);
        }

        FieldTypeEnum typeEnum = FieldTypeEnum.valueOf(type);
        if (log.isDebugEnabled()) {
            log.debug("typeEnum of type " + type + " = " + typeEnum);
        }
        String editorName = typeEnum == null ? DEFAULT_EDITOR_NAME : typeEnum.name();
        if (log.isDebugEnabled()) {
            log.debug("editor name for type [" + type.getType() + "] : " + editorName);
        }

        Class<? extends VradiEditor<?, ?>> editorClass = editorMapping.get(editorName);

        if (log.isDebugEnabled()) {
            log.debug("editor class to use : " + editorClass);
        }
        try {
            Constructor[] constructors = editorClass.getConstructors();
            for (Constructor constructor : constructors) {
                if (constructor.getParameterTypes() != null
                        && constructor.getParameterTypes().length == 1
                        && constructor.getParameterTypes()[0].equals(JAXXContext.class)) {
                    VradiEditor editor = (VradiEditor) constructor.newInstance(context);
                    editor.init();
                    return editor;
                }
            }
            return null;
        } catch (InvocationTargetException eee) {
            // should never happens
            throw new IllegalStateException(eee);
        } catch (InstantiationException eee) {
            // should never happens
            throw new IllegalStateException(eee);
        } catch (IllegalAccessException eee) {
            // should never happens
            throw new IllegalStateException(eee);
        }
    }

    /**
     * Changes the status of the form and save it.
     *
     * @param context the actual context of the ui
     * @param form the form to modify and save
     * @param status the new status of the form
     */
    public void changeStatusAndSave(JAXXContext context, Form form,
                                    Status status) {
        form.setStatus(status.getWikittyId());
        save(context, form);
    }

}
