/*
 * *##% 
 * vradi-swing
 * Copyright (C) 2009 JurisMarches
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.ui.email;

import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.SendingStatus;
import com.jurismarches.vradi.services.dto.VradiClientDTO;
import com.jurismarches.vradi.services.dto.VradiFormDTO;
import com.jurismarches.vradi.services.dto.VradiSendingDTO;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import static org.nuiton.i18n.I18n._;

import javax.swing.*;
import javax.swing.table.AbstractTableModel;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.*;

/**
 * Le modèle du tableau pour afficher et éditer les bindings d'un flux XML.
 * <p/>
 * User: chemit
 * Date: 15 nov. 2009
 * Time: 16:56:08
 */
public class ClientEmailContentTableModel extends AbstractTableModel {

    private static final long serialVersionUID = 1L;

    /**
     * Logger
     */
    static private final Log log = LogFactory.getLog(ClientEmailContentTableModel.class);

    /**
     * Map des clients et leurs formulaires à exclure
     */
    protected Map<VradiSendingDTO, List<VradiFormDTO>> data;

    /**
     * Un drapeau pour bloquer le déclanchement des fireXXX)
     */
    protected boolean valueIsAdjusting;

    public ClientEmailContentTableModel() {
        data = new LinkedHashMap<VradiSendingDTO, List<VradiFormDTO>>();
    }

    @Override
    public int getRowCount() {
        return data.size();
    }

    @Override
    public int getColumnCount() {
        return 6;
    }

    @Override
    public boolean isCellEditable(int rowIndex, int columnIndex) {
        return getData().get(rowIndex).getStatus() == SendingStatus.TO_SEND.getStatus()
                && (columnIndex == 2 || columnIndex == 3 || columnIndex == 5);
    }

    /**
     * Pour vider le modèle.
     */
    public void clear() {
        data.clear();
        if (!valueIsAdjusting) {
            fireTableDataChanged();
        }
    }

    public boolean isValueIsAdjusting() {
        return valueIsAdjusting;
    }

    public void setValueIsAdjusting(boolean valueIsAdjusting) {
        this.valueIsAdjusting = valueIsAdjusting;
    }

    /**
     * @return une copie de la liste des bindings
     */
    public List<VradiSendingDTO> getData() {
        // always send a copy
        return new ArrayList<VradiSendingDTO>(data.keySet());
    }

    /**
     * @return une copie de la liste des formulaires a envoyer
     */
    public List<VradiFormDTO> getFormsToExclude(VradiSendingDTO sending) {
        // always send a copy
        return new ArrayList<VradiFormDTO>(data.get(sending));
    }

    public void setFormsToExclude(VradiSendingDTO sending, List<VradiFormDTO> formsToSend) {
        this.data.put(sending, formsToSend);
    }

    /**
     * Ajouter une entrée dans le modèle
     */
    public void addEntry(VradiSendingDTO formsByClientDTO) {
        if (log.isDebugEnabled()) {
            log.debug("adding client " + formsByClientDTO.getClientDTO().getName());
        }
//        String[] status = new String[forms.size()];
//        Arrays.fill(status, _("vradi.email.status.toSend"));
//        clientContent.status = Arrays.asList(status);
        data.put(formsByClientDTO, new ArrayList<VradiFormDTO>());
        if (isValueIsAdjusting()) {
            int newIndex = data.size() - 1;
            fireTableRowsInserted(newIndex, newIndex);
        }
    }

    @Override
    public Object getValueAt(int rowIndex, int columnIndex) {
        ensureColumnIndex(columnIndex);
        ensureRowIndex(rowIndex);
        VradiSendingDTO value = new ArrayList<VradiSendingDTO>(data.keySet()).get(rowIndex);
        Object result = null;
        switch (columnIndex) {
            case 0:
                result = value.getSentDate();
                break;
            case 1:
                result = value.getClientDTO();
                break;
            case 2:
                result = data.get(value);
                break;
            case 3:
                result = value.getParagraph();
                break;
            case 4:
                result = SendingStatus.get(value.getStatus());
                break;
            case 5:
                result = value.getReceptionProof();
                break;
        }
        return result;
    }


    @Override
    public void setValueAt(Object aValue, int rowIndex, int columnIndex) {
        ensureColumnIndex(columnIndex);
        ensureRowIndex(rowIndex);
        VradiSendingDTO value = new ArrayList<VradiSendingDTO>(data.keySet()).get(rowIndex);
        if (aValue != null) {
            switch (columnIndex) {
                case 0:
                    value.setSentDate((Date) aValue);
                case 1:
                    value.setClientDTO((VradiClientDTO) aValue);
                    break;
                case 2:
                    data.get(value).clear();
                    for (Object o : (Object[]) aValue) {
                        data.get(value).add((VradiFormDTO) o);
                    }
                    break;
                case 3:
                    value.setParagraph((String) aValue);
                    break;
                case 4:
                    value.setStatus((Integer) aValue);
                    break;
                case 5:
                    value.setReceptionProof((Boolean) aValue);
                    break;
            }
        }
        fireTableCellUpdated(rowIndex, columnIndex);
    }

    @Override
    public Class getColumnClass(int column) {
        switch (column) {
            case 0:
                return Date.class;
            case 1:
                return String.class;
            case 2:
                return List.class;
            case 3:
                return String.class;
            case 4:
                return int.class;
            case 5:
                return boolean.class;
            default:
                return String.class;
        }
    }


    protected void ensureRowIndex(int rowIndex) {
        if (rowIndex >= getRowCount()) {
            throw new ArrayIndexOutOfBoundsException("rowIndex (" + rowIndex + ") can not be greater than " + getRowCount());
        }
    }

    protected void ensureColumnIndex(int columnIndex) {
        if (columnIndex >= getColumnCount()) {
            throw new ArrayIndexOutOfBoundsException("columnIndex (" + columnIndex + ") can not be greater than " + getColumnCount());
        }
    }

}
