/*
 * *##% 
 * vradi-swing
 * Copyright (C) 2009 JurisMarches
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.ui;

import com.jurismarches.vradi.VradiHelper;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.services.ServiceHelper;
import com.jurismarches.vradi.services.dto.VradiFormPageDTO;
import jaxx.runtime.JAXXObject;
import jaxx.runtime.binding.DefaultJAXXBinding;
import jaxx.runtime.swing.ErrorDialogUI;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import static org.nuiton.i18n.I18n.n_;
import org.sharengo.exceptions.TechnicalException;
import org.sharengo.wikitty.TreeNode;

import javax.swing.*;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableCellRenderer;
import java.awt.*;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.text.DateFormat;
import java.util.*;
import java.util.List;

/**
 * @author letellier
 */
public class OfferListTableModel extends AbstractTableModel {

    static private Log log = LogFactory.getLog(OfferListTableModel.class);
    static final public String PAGE_TO_SHOW_PROPERTY = "pageToShow";
    static final public String NB_FORMS_PER_PAGE_PROPERTY = "nbFormsPerPage";
    static final public String FIELD_TO_SORT_PROPERTY = "fieldToSort";
    static final public String ASCENDING_PROPERTY = "ascending";
    static final public String NB_PAGES = "nbPages";
    static final public String LAST_PAGE = "lastPage";
    static public final int TOOLTIP_LINE_MAX_CHAR_NB = 100;
    static public final int TOOLTIP_ELEMENT_MAX_CHAR_NB = 500;

    protected VradiFormPageDTO datas;
    protected Map<String, TreeNode> thesaurus;
    protected List<String> columnNames;

    protected PropertyChangeSupport propertyChange = new PropertyChangeSupport(
            this);

    public OfferListTableModel(VradiFormPageDTO datas) {

        columnNames = new ArrayList<String>();
        columnNames.add(n_("vradi.offer.name"));
        columnNames.add(n_("vradi.offerEdit.datePub"));
        columnNames.add(n_("vradi.offer.statut"));


        // Adding thesaurus to display
        thesaurus = new HashMap<String, TreeNode>();
        List<String> columnsThesaurusId = VradiHelper.getVradiListColumns();
        for (String id : columnsThesaurusId) {
            try {
                TreeNode t = ServiceHelper.getVradiStorageService()
                        .getThesaurus(id);
                if (t != null) {
                    thesaurus.put(t.getName(), t);
                    columnNames.add(t.getName());
                    if (log.isDebugEnabled()) {
                        log.debug("Adding Thesaurus name : " + t.getName());
                    }
                } else {
                    log.error("Cant get thesaurus : " + id);
                }
            } catch (TechnicalException e) {
                log.error("Cant get thesaurus : " + id, e);
                ErrorDialogUI.showError(e);
            }
        }
        this.datas = datas != null ? datas : new VradiFormPageDTO();
        datas.addPropertyChangeListener(
                VradiFormPageDTO.TOTAL_FOUND_FORM_NB_PROPERTY,
                new PropertyChangeListener() {

                    @Override
                    public void propertyChange(PropertyChangeEvent evt) {
                        propertyChange.firePropertyChange(evt.getPropertyName(),
                                evt.getOldValue(), evt.getNewValue());
                        propertyChange.firePropertyChange(NB_PAGES, null,
                                getNbPages());
                        propertyChange.firePropertyChange(LAST_PAGE, null,
                                isLastPage());
                    }
                });
        fireTableDataChanged();
    }

    public Integer getNbFormsPerPage() {
        return datas.getNbFormsToShow();
    }

    public void setNbFormsPerPage(Integer nbFormsPerPage) {
        Integer oldValue = getNbFormsPerPage();
        datas.setNbFormsToShow(nbFormsPerPage);
        int i = ((Double) Math
                .ceil(((Integer) (oldValue * (getPageToShow() - 1)))
                        .doubleValue() / nbFormsPerPage.doubleValue()))
                .intValue();
        setPageToShow(i == 0 && getTotalFoundFormNb() > 0 ? 1 : i);
        fireTableDataChanged();
        propertyChange.firePropertyChange(NB_FORMS_PER_PAGE_PROPERTY, oldValue,
                nbFormsPerPage);
        propertyChange.firePropertyChange(NB_PAGES, null, getNbPages());
        propertyChange.firePropertyChange(LAST_PAGE, null, isLastPage());
    }

    public Integer getPageToShow() {
        return datas.getPageToShow();
    }

    public void setPageToShow(Integer pageToShow) {
        Integer oldValue = getPageToShow();
        datas.setPageToShow(pageToShow);
        fireTableDataChanged();
        propertyChange.firePropertyChange(PAGE_TO_SHOW_PROPERTY, oldValue,
                pageToShow);
        propertyChange.firePropertyChange(LAST_PAGE, null, isLastPage());
    }

    public String getFieldToSort() {
        return datas.getFieldToSort();
    }

    public void setFieldToSort(String fieldToSort) {
        Integer oldValue = getPageToShow();
        datas.setFieldToSort(fieldToSort);
    }

    public boolean isAscending() {
        return datas.isAscending();
    }

    public void setAscending(boolean ascending) {
        datas.setAscending(ascending);
    }

    public Integer getTotalFoundFormNb() {
        return datas.getTotalFoundFormNb();
    }


    @Override
    public boolean isCellEditable(int rowIndex, int columnIndex) {
        return false;
    }

    @Override
    public String getColumnName(int col) {
        if (col >= 0) {
            return columnNames.get(col);
        }
        return "";
    }

    @Override
    public int getRowCount() {
        return datas.getFormsToShow().size();
    }

    @Override
    public int getColumnCount() {
        return columnNames.size();
    }

    @Override
    public Object getValueAt(int row, int col) {
        if (datas.getFormsToShow().get(row) != null) {
            if (col < 3) {
                // Data
                switch (col) {
                    case 0:
                        return datas.getFormsToShow().get(row).getName();
                    case 1:
                        return datas.getFormsToShow().get(row).getDatePub();
                    case 2:
                        return datas.getFormsToShow().get(row).getStatut();
                }
            } else {
                // Thesaurus
                Set<String> thesaurusIdOfBean = datas.getFormsToShow().get(row)
                        .getThesaurus();

                String result = "";
                if (thesaurusIdOfBean != null) {
                    String name = columnNames.get(col);
                    if (log.isDebugEnabled()) {
                        log.debug("Want thesaurus name : " + name);
                    }
                    TreeNode thesaurusOfColumns = thesaurus.get(name);
                    for (String idOfBean : thesaurusIdOfBean) {
                        Set<String> children = thesaurusOfColumns.getChildren();
                        if (children != null) {
                            if (children.contains(idOfBean)) {
                                TreeNode thesaurusToDisplay = null;
                                try {
                                    thesaurusToDisplay = ServiceHelper
                                            .getVradiStorageService()
                                            .getThesaurus(idOfBean);
                                } catch (TechnicalException e) {
                                    log.error(
                                            "cant get thesaurus : " + idOfBean,
                                            e);
                                    ErrorDialogUI.showError(e);
                                }
                                if (thesaurusToDisplay != null) {
                                    result += thesaurusToDisplay.getName()
                                            + ", ";
                                }
                            }
                        }
                    }
                    if (result.length() > 0) {
                        result = result.substring(0, result.length() - 2);
                    }
                }
                return result;
            }
        }
        return null;
    }

    @Override
    public Class getColumnClass(int c) {
        switch (c) {
            case 0:
                return String.class;
            case 1:
                return Date.class;
            case 2:
                return int.class;
            default:
                return String.class;
        }
    }

    @Override
    public void setValueAt(Object value, int row, int col) {
        if (datas.getFormsToShow().get(row) != null) {
            if (col < 3) {
                // Data
                switch (col) {
                    case 0:
                        datas.getFormsToShow().get(row)
                                .setName(value.toString());
                    case 1:
                        datas.getFormsToShow().get(row)
                                .setDatePub((Date) value);
                    case 2:
                        datas.getFormsToShow().get(row)
                                .setStatut((Integer) value);
                }
            }
        }
        fireTableCellUpdated(row, col);
    }

    public VradiFormPageDTO getDatas() {
        //return datas.subList((pageShown-1) * nbFormsPerPage, pageShown * nbFormsPerPage);
        return datas;
    }

//    public void setDatas(VradiFormPageDTO datas) {
//        this.datas = datas != null ? datas : new VradiFormPageDTO();
//        datas.addPropertyChangeListener(VradiFormPageDTO.TOTAL_FOUND_FORM_NB_PROPERTY, new PropertyChangeListener() {
//
//            @Override
//            public void propertyChange(PropertyChangeEvent evt) {
//                propertyChange.firePropertyChange(evt.getPropertyName(), evt.getOldValue(), evt.getNewValue());
//            }
//        });
//        fireTableDataChanged();
//    }

    public String getFieldNameAt(int col) {
        if (col < 3) {
            // Data
            switch (col) {
                case 0:
                    return Form.FQ_FIELD_NAME;
                case 1:
                    return Form.FQ_FIELD_DATEPUB;
                case 2:
                    return Form.FQ_FIELD_STATUT;
            }
        } else {
            // Thesaurus
            return columnNames.get(col);
        }
        return null;
    }

    public boolean isLastPage() {
        boolean result = ((Double) Math
                .ceil(getTotalFoundFormNb().doubleValue() / getNbFormsPerPage()
                        .doubleValue())).intValue() <= getPageToShow();
        return result;
    }

    public String getNbPages() {
        int nbPages = ((Double) Math
                .ceil(getTotalFoundFormNb().doubleValue() / getNbFormsPerPage()
                        .doubleValue())).intValue();
        return " / " + String.valueOf(nbPages);
    }

    public String getToolTip(int row) {
        StringBuffer result = new StringBuffer();
        Map<String, Object> fieldValues = new HashMap<String, Object>();
        Form form = datas.getFormsToShow().get(row);
        Collection<String> extensionNames = form.getExtensionNames();
        for (String extensionName : extensionNames) {
            Collection<String> fieldNames = form
                    .getExtensionFields(extensionName);
            for (String fieldName : fieldNames) {
                fieldValues.put(fieldName,
                        form.getField(extensionName, fieldName));
            }
        }
        result.append("<html>");
        for (Map.Entry entry : fieldValues.entrySet()) {
            Object value = entry.getValue();
            if (value != null) {
                String stringValue =
                        value.getClass().equals(Date.class) ?
                                DateFormat.getDateTimeInstance(DateFormat.SHORT,
                                        DateFormat.MEDIUM, Locale.FRANCE)
                                        .format((Date) value)
                                : String.valueOf(value);
                result.append("<strong>").append(entry.getKey())
                        .append("</strong>").append(" : ");
                if (stringValue.length() > TOOLTIP_ELEMENT_MAX_CHAR_NB) {
                    stringValue = stringValue.substring(0,
                            stringValue.lastIndexOf(' ',
                                    TOOLTIP_ELEMENT_MAX_CHAR_NB)) + "...";
                }
                log.debug(stringValue);
                while (stringValue.length() > TOOLTIP_LINE_MAX_CHAR_NB) {
                    int spaceIndex = stringValue.lastIndexOf("\\s",
                            TOOLTIP_LINE_MAX_CHAR_NB);
                    String lineValue = stringValue.substring(0,
                            spaceIndex < 0 ? stringValue.length() : spaceIndex);
                    result.append(lineValue);
                    result.append("<br/>&nbsp;");
                    stringValue = stringValue.substring(spaceIndex < 0 ?
                            stringValue.length() : spaceIndex);
                }
                result.append(stringValue).append("<br/>");
            }
        }
        result.append("</html>");
        return result.toString();
    }

    public TreeNode getThesaurusByName(String name) {
        return thesaurus.get(name);
    }

    public void addPropertyChangeListener(PropertyChangeListener listener) {
        propertyChange.addPropertyChangeListener(listener);
    }

    public void removePropertyChangeListener(PropertyChangeListener listener) {
        propertyChange.removePropertyChangeListener(listener);
    }

    public void addPropertyChangeListener(String propertyName,
                                          PropertyChangeListener listener) {
        propertyChange.addPropertyChangeListener(propertyName, listener);
    }

    public void removePropertyChangeListener(String propertyName,
                                             PropertyChangeListener listener) {
        propertyChange.removePropertyChangeListener(propertyName, listener);
    }

    static public abstract class OfferListTableModelBinding
            extends DefaultJAXXBinding {

        protected OfferListTableModel model;

        public OfferListTableModelBinding(JAXXObject source, String id,
                                          OfferListTableModel model) {
            super(source, id, false);
            this.model = model;
        }

        @Override
        public void applyDataBinding() {
            if (model != null) {
                model.addPropertyChangeListener(PAGE_TO_SHOW_PROPERTY, this);
                model.addPropertyChangeListener(NB_FORMS_PER_PAGE_PROPERTY,
                        this);
            }
        }

        @Override
        public void removeDataBinding() {
            if (model != null) {
                model.removePropertyChangeListener(PAGE_TO_SHOW_PROPERTY, this);
                model.addPropertyChangeListener(NB_FORMS_PER_PAGE_PROPERTY,
                        this);
            }
        }
    }

    static public class OfferListTableCellRenderer
            implements TableCellRenderer {

        @Override
        public Component getTableCellRendererComponent(JTable table,
                                                       Object value,
                                                       boolean isSelected,
                                                       boolean hasFocus,
                                                       int row, int column) {
            String stringValue =
                    value != null && value.getClass().equals(Date.class) ?
                            DateFormat.getDateTimeInstance(DateFormat.SHORT,
                                    DateFormat.MEDIUM, Locale.FRANCE)
                                    .format((Date) value)
                            : String.valueOf(value);
            JLabel label = new JLabel(stringValue);
            OfferListTableModel model = (OfferListTableModel) table.getModel();
            label.setToolTipText(model.getToolTip(row));
            return label;
        }
    }
}
