package com.jurismarches.vradi;

import com.jurismarches.vradi.entities.*;
import com.jurismarches.vradi.ui.Thesaurus;
import com.jurismarches.vradi.ui.VradiMainUI;
import com.jurismarches.vradi.ui.VradiMainUIHandler;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.Util;
import jaxx.runtime.decorator.DecoratorProvider;
import jaxx.runtime.swing.ErrorDialogUI;
import jaxx.runtime.swing.renderer.DecoratorProviderListCellRenderer;
import org.apache.commons.beanutils.ConvertUtils;
import org.apache.commons.beanutils.Converter;
import org.apache.commons.beanutils.converters.DateConverter;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.i18n.I18n;
import static org.nuiton.i18n.I18n._;
import org.nuiton.util.ConverterUtil;
import org.nuiton.util.StringUtil;
import org.sharengo.wikitty.TreeNode;
import org.sharengo.wikitty.WikittyExtension;

import javax.swing.*;
import java.util.logging.Level;
import java.util.logging.Logger;

public class VradiMain {

    /**
     * Logger
     */
    private static Log log = LogFactory.getLog(VradiMain.class);

    public static void main(String[] args) {

        long startingTime = System.nanoTime();
        log.info("Vradi start at " + new java.util.Date() + " args: " + java.util.Arrays.toString(args));

        // make solr quiet (must be done before any instanciation
        Logger.getLogger("org.apache.solr").setLevel(Level.FINEST);

        try {

            // init root context
            final VradiContext context = init(args);

            log.info(_("vradi.init.context.done", StringUtil.convertTime(startingTime, System.nanoTime())));

            Util.checkJAXXContextEntry(context, Util.newContextEntryDef(VradiConfig.class));
            Util.checkJAXXContextEntry(context, Util.newContextEntryDef(VradiAction.class));

            VradiConfig config = context.getContextValue(VradiConfig.class);

            config.doAction(VradiConfig.Step.AfterInit.ordinal());

            if (config.isDisplayMainUI()) {
                // on affiche l'ui principale

                VradiMainUIHandler uiHandler = context.getContextValue(VradiMainUIHandler.class);

                // do init ui
                final VradiMainUI ui = uiHandler.initUI(context, config.isFullScreen());

                log.info(_("vradi.init.ui.done"));

                // show ui
                SwingUtilities.invokeLater(new Runnable() {

                    @Override
                    public void run() {
                        ui.setVisible(true);
                    }
                });

            }
        } catch (Exception e) {
            ErrorDialogUI.showError(e);
            log.error(e.getMessage(), e);
            System.exit(1);
        }
    }

    public static VradiContext init(String... args) throws Exception {
        // to enable javassist on webstart, must remove any securityManager,
        // see if this can be dangerous (should not be since jnlp is signed ?)
        // moreover it speeds up the loading :)
        System.setSecurityManager(null);

        long t0 = System.nanoTime();

        I18n.setUniqueBundleName("vradi-swing-i18n");

        Runtime.getRuntime().addShutdownHook(new ShutdownHook());

        // init root context
        VradiContext context = VradiContext.init();


        // register decorator one for all
        DecoratorProvider p = new DecoratorProvider() {

            @Override
            protected void loadDecorators() {
                registerMultiJXPathDecorator(Client.class, "${name}$s", "", "");
                registerMultiJXPathDecorator(User.class, "${name}$s", "", "");
                registerMultiJXPathDecorator(Group.class, "${name}$s", "", "");
                registerMultiJXPathDecorator(WikittyExtension.class, "${name}$s", "", "");
                registerMultiJXPathDecorator(XmlStream.class, "${name}$s", "", "");
                registerMultiJXPathDecorator(Thesaurus.class, "${name}$s", "", "");
                registerMultiJXPathDecorator(TreeNode.class, "${name}$s", "", "");
            }
        };
        context.setContextValue(p);
        context.setContextValue(new DecoratorProviderListCellRenderer(p));

        // init config
        VradiConfig config = context.getContextValue(VradiConfig.class);
        config.parse(args);

        long t00 = System.nanoTime();

        // init i18n
        I18n.init(config.getLocale());
        log.info("language : " + config.getLocale());

        if (log.isDebugEnabled()) {
            log.debug("i18n loading time : " + (StringUtil.convertTime(t00, System.nanoTime())));
        }

        log.info(_("vradi.message.config.loaded", config.getVersion()));

        // prepare ui look&feel and load ui properties
        try {
            SwingUtil.initNimbusLoookAndFeel();
        } catch (Exception e) {
            // could not find nimbus look-and-feel
            log.warn(_("vradi.warning.nimbus.landf"));
        } catch (Throwable e) {
            log.warn(_("vradi.warning.no.ui"));
            // pas d'environnement d'ui
            config.setCanUseUI(false);
        }

        // init date converter
        Converter converter = ConverterUtil.getConverter(java.util.Date.class);
        if (converter != null) {
            ConvertUtils.deregister(java.util.Date.class);

            DateConverter dateConverter = new DateConverter();
            dateConverter.setUseLocaleFormat(true);
            ConvertUtils.register(dateConverter, java.util.Date.class);
        }

        if (config.isCanUseUI()) {
            // chargement de la configuration des uis
            SwingUtil.loadUIConfig(VradiConfig.APPLICATION_PROPERTIES, null);
        }

        if (log.isDebugEnabled()) {
            log.debug("init done in " + (StringUtil.convertTime(t0, System.nanoTime())));
        }
        return context;
    }

    public static class ShutdownHook extends Thread {

        public ShutdownHook() {
            super("shutdown vradi");
        }

        @Override
        public void run() {
            try {
                super.run();
                VradiContext.get().close();
                // force to kill main thread

                log.info(_("vradi.init.closed", new java.util.Date()));
                Runtime.getRuntime().halt(0);
            } catch (Exception ex) {
                log.error("error while closing " + ex.getMessage(), ex);
                Runtime.getRuntime().halt(1);
            }
        }
    }
}
