/* *##%
 * Copyright (c) 2009 Sharengo, Guillaume Dufrene, Benjamin POUSSIN.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *##%*/

package org.sharengo.wikitty;


import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 *
 * @author poussin
 * @version $Revision$
 *
 * Last update: $Date$
 * by : $Author$
 */
public class WikittyExtension implements Serializable {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(WikittyExtension.class);

    private static final long serialVersionUID = 1L;

    /**
     * name of this extension
     */
    protected String name;
    /**
     * name of other extension needed to put this extension to object
     */
    protected String requires;

    /**
     * use to know version objet, when you change field number, type or other
     * you must change version number.
     */
    protected String version = WikittyUtil.DEFAULT_VERSION;

    /**
     * fields use ordered map, to keep order insertion of field
     * key: field name
     * value: field type
     */
	protected LinkedHashMap<String, FieldType> fields = new LinkedHashMap<String, FieldType>();

	public WikittyExtension(String name) {
		if (name == null) {
            throw new IllegalArgumentException("Name must not be null");
        }
		this.name = name;
	}
	
	public WikittyExtension(String name, String version,
            String requires, LinkedHashMap<String, FieldType> fields) {
        if (name == null) {
            throw new IllegalArgumentException("Name must not be null");
        }
        if (version == null) {
            throw new IllegalArgumentException("Version must not be null");
        }
		this.name = name;
        this.version = version;
		this.requires = requires;
        if (fields != null) {
            this.fields = fields;
        }
	}

    /**
     * compute id for extension name and version in argument
     * @param name
     * @param version
     * @return
     */
    static public String computeId(String name, String version) {
        String result = name + "[" + version + "]";
        return result;
    }

    static public String computeName(String id) {
        int i = id.lastIndexOf("[");
        String result = id.substring(0, i);
        return result;
    }
    
    static public String computeVersion(String id) {
        int b = id.lastIndexOf("[");
        int e = id.lastIndexOf("]");
        String result = id.substring(b+1, e);
        return result;
    }

    public String getId() {
        String result = computeId(getName(), getVersion());
        return result;
    }

    public String getName() {
        return name;
    }

    public String getVersion() {
        return version;
    }

    public String getRequires() {
        return requires;
    }
    
	public FieldType getFieldType(String fieldName) {
		return fields.get(fieldName);
	}

    public List<String> getFieldNames() {
        List<String> result = new ArrayList<String>(fields.keySet());
        return result;
    }

	public void addField(String fieldName, FieldType type) {
		fields.put(fieldName, type);
	}

    @Override
    public int hashCode() {
        return getId().hashCode();
    }


    @Override
    public boolean equals(Object obj) {
        boolean result = false;
        if (obj instanceof WikittyExtension) {
            WikittyExtension other = (WikittyExtension)obj;
            result = this.getId().equals(other.getId());
        }
        return result;
    }

    @Override
    public String toString() {
        return getId();
    }
    
    public String toDefinition() {
    	String res = "Extension " + getId() + "{\n";
    	for ( String fieldName : fields.keySet() ) {
    		res += fields.get(fieldName).toDefinition(fieldName) + "\n";
    	}
    	res += "}";
    	return res;
    }


}
