/* *##%
 * Copyright (c) 2009 poussin. All rights reserved.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *##%*/

package org.sharengo.wikitty;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.Writer;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.FutureTask;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.sharengo.wikitty.search.Search;
import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserFactory;

/**
 * Abstract class that new implementation must extends.
 * New implementation only have three method to implement:
 * <li>getSearchEngine
 * <li>getExtensionStorage
 * <li>getWikittyStorage
 *
 * @author poussin
 * @version $Revision$
 *
 * Last update: $Date$
 * by : $Author$
 */
public abstract class AbstractWikittyService implements WikittyService {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(AbstractWikittyService.class);

    // FIXME poussin 20090902 next 3 variables must be read from configuration file
    /** number of thread used to import/export task */
    protected int MAX_IMPORT_EXPORT_THREAD = 1;
    /** directory path where export asynchronous file are stored */
    protected String EXPORT_DIRECTORY = "/tmp/";
    /** url used by client to retrieve export file when job is ended */
    protected String EXPORT_URL = "file:///tmp/";

    /** Executor that do import export task */
    protected ExecutorService importExportExecutor =
            // TODO poussin 20090902 do thread number configurable
            Executors.newFixedThreadPool(MAX_IMPORT_EXPORT_THREAD);
    /** contains all import or export task, key is job id send to client */
    protected Map<String, Future<String>> importExportTask =
            new HashMap<String, Future<String>>();

    abstract protected WikittySearchEngine getSearchEngine();
    abstract protected WikittyExtensionStorage getExtensionStorage();
    abstract protected WikittyStorage getWikittyStorage();

    /**
     * Store and index wikitty object
     * @param wikitty
     */
    public UpdateResponse store(Wikitty wikitty) {
        if (wikitty != null) {
            List<Wikitty> wikitties = Arrays.asList(wikitty);
            UpdateResponse result = store( wikitties );
            return result;
        } else {
            throw new WikittyException("You can't store null wikitty object");
        }
    }

    /**
     * Store and index wikitties object
     * @param wikitties
     */
    public UpdateResponse store(Collection<Wikitty> wikitties) {
    	return store(wikitties, false);
    }

    /**
     * Store and index wikitties object
     * @param wikitties
     * @param disableAutoVersionIncrement
     */
    public UpdateResponse store(Collection<Wikitty> wikitties, boolean disableAutoVersionIncrement) {
        try {
            // update/store extension if necessary
            Set<WikittyExtension> allExtensions = new HashSet<WikittyExtension>();
            for (Wikitty w : wikitties) {
                // collect all extensions used by all wikitties
                allExtensions.addAll(w.getExtensions());
            }

            WikittyTransaction transaction = new WikittyTransaction();

            // create extensions update statement
            List<WikittyExtensionStorage.Command> extensionStorageCommandList =
                    getExtensionStorage().prepare(transaction, allExtensions);

            // create wikitties update statement
            List<WikittyStorage.Command> wikittyStorageCommandList =
                    getWikittyStorage().prepare(transaction, wikitties, disableAutoVersionIncrement);

            // create solr wikitty document for solr
            List<WikittySearchEngine.Command> wikittyIndexationCommandList =
                    getSearchEngine().prepare(transaction, wikitties);

            // All prepare (store and index) are good
            // try to commit command
            UpdateResponse extUpdate = getExtensionStorage().commit(transaction,
                    extensionStorageCommandList);
            UpdateResponse wikUpdate = getWikittyStorage().commit(transaction,
                    wikittyStorageCommandList);
            UpdateResponse indexUpdate = getSearchEngine().commit(transaction,
                    wikittyIndexationCommandList);

            UpdateResponse result = new UpdateResponse();

            // prepare update client response
            result.add(extUpdate);
            result.add(wikUpdate);
            result.add(indexUpdate);

            return result;
        } catch (Exception eee) {
            throw new WikittyException(eee);
        }
    }

    public List<String> getAllExtensionIds() {
        List<String> result = getExtensionStorage().getAllExtensionIds();
        return result;
    }

    public List<WikittyExtension> getAllExtensions(boolean lastVersion) {
        List<WikittyExtension> result = getExtensionStorage().getAllExtensions(lastVersion);
        return result;
    }

    /**
     * Save just one extension
     * @param exts
     * @throws java.io.IOException
     */
    public UpdateResponse storeExtension(Collection<WikittyExtension> exts) {
        WikittyTransaction transaction = new WikittyTransaction();
        List<WikittyExtensionStorage.Command> commands =
                getExtensionStorage().prepare(transaction, exts);
        UpdateResponse result = getExtensionStorage().commit(transaction,
                commands);

        return result;
    }

    /**
     * Load extension from id. Id is 'name[version]'
     * @param id
     * @return
     */
    public WikittyExtension restoreExtension(String id) {
        WikittyExtension result = getExtensionStorage().restore(id);
        return result;
    }

    public Wikitty restore(String id) {
        if (!getWikittyStorage().exists(id)) {
            // object doesn't exist, we return null
            return null;
        }

        if (getWikittyStorage().isDeleted(id)) {
            // object deleted, we return null
            return null;
        }

        Wikitty result = getWikittyStorage().restore(id);
        return result;
    }

    public List<Wikitty> restore(List<String> id) {
        List<Wikitty> result = new ArrayList<Wikitty>();
        for(String k : id) {
            Wikitty w = restore(k);
            if ( w != null ) {
            	result.add(w);
            }
        }
        return result;
    }

    public void delete(String id){
        delete(Arrays.asList(id));
    }

    public void delete(Collection<String> ids){
        // work only on valid id
        List<String> idList = new LinkedList<String>(ids);
        for (Iterator<String> i = idList.iterator(); i.hasNext();) {
            String id = i.next();
            // test if wikitty exists
            if (!getWikittyStorage().exists(id)) {
                // don't exist, remove this id in id list
                i.remove();
            }
            if (getWikittyStorage().isDeleted(id)) {
                // already deleted, remove this id in id list
                i.remove();
            }
        }

        WikittyTransaction transaction = new WikittyTransaction();
        List<WikittyStorage.Command> storeCommands =
                getWikittyStorage().delete(idList);
        List<WikittySearchEngine.Command> indexCommands =
                getSearchEngine().delete(transaction, idList);

        getWikittyStorage().commit(transaction, storeCommands);
        getSearchEngine().commit(transaction, indexCommands);
    }


    public PagedResult<Wikitty> findAllByCriteria(Criteria criteria) {
        PagedResult<String> resultId = getSearchEngine().findAllByCriteria(criteria);
        PagedResult<Wikitty> result = resultId.cast(getWikittyStorage());
        return result;
    }

    public Wikitty findByCriteria(Criteria criteria) {
        criteria.setFirstIndex(0).setEndIndex(1);
        PagedResult<Wikitty> pages = findAllByCriteria(criteria);
        if (pages.size() > 0) {
            Wikitty result = pages.getFirst();
            return result;
        } else {
            return null;
        }
    }


    public void addLabel(String wikittyId, String label) {
        Wikitty w = restore(wikittyId);
        w.addExtension(LabelImpl.extensions);
        LabelImpl l = new LabelImpl(w);
        l.addLabels(label);
        store(l.getWikitty());
    }


    public PagedResult<Wikitty> findAllByLabel(String label, int firstIndex, int endIndex) {
        LabelImpl l = new LabelImpl();
        l.addLabels(label);
        Criteria criteria = Search.query(l.getWikitty()).criteria()
                .setFirstIndex(firstIndex).setEndIndex(endIndex);
        PagedResult<Wikitty> result = findAllByCriteria(criteria);
        return result;
    }


    public Wikitty findByLabel(String label) {
        LabelImpl l = new LabelImpl();
        l.addLabels(label);
        Criteria criteria = Search.query(l.getWikitty()).criteria();
        Wikitty result = findByCriteria(criteria);
        return result;
    }


    public Set<String> findAllAppliedLabels(String wikittyId) {
        Wikitty w = restore(wikittyId);
        Label l = new LabelImpl(w);
        Set<String> result = l.getLabels();
        return result;
    }


    public Tree restoreTree(String wikittyId) {
    	Wikitty w = restore(wikittyId);
        if(w == null) {
            return null;
        }

        if ( !w.hasExtension(TreeNode.EXT_TREENODE) ) {
    		throw new WikittyException(String.format(
                    "Wikitty '%s' do not handle extension %s",
                    wikittyId, TreeNode.EXT_TREENODE ));
    	}
    	Tree tree = new Tree();
    	TreeNode node = new TreeNodeImpl( w );
    	tree.setNode( node );

    	TreeNodeImpl exempleNode = new TreeNodeImpl();
    	exempleNode.setParent(wikittyId);

        Criteria criteria = Search.query(exempleNode.getWikitty()).criteria()
                .setFirstIndex(0).setEndIndex(ALL_ELEMENTS);
        PagedResult<Wikitty> childNodes = findAllByCriteria(criteria);
    	for( Wikitty childNode : childNodes.getAll() ) {
    		tree.addChild( restoreTree(childNode.getId()) );
    	}
    	Set<String> children = node.getChildren();
        if (children != null) {
            for (String childrenId : children) {
                tree.addElement(childrenId);
            }
    	}

    	return tree;
    }

    public Map<TreeNode, Integer> restoreChildren(String wikittyId) {
        Wikitty w = restore(wikittyId);
        if(w == null) {
            return null;
        }

    	if ( !w.hasExtension(TreeNode.EXT_TREENODE) ) {
    		throw new WikittyException(String.format(
                    "Wikitty '%s' do not handle extension %s",
                    wikittyId, TreeNode.EXT_TREENODE ));
    	}

        Map<TreeNode, Integer> result = new LinkedHashMap<TreeNode, Integer>();

        Map<String, Integer> search = getSearchEngine().findAllChildren(w);
        Set<Entry<String, Integer>> children = search.entrySet();
        for (Entry<String, Integer> child : children) {
            Integer count = child.getValue();

            String id = child.getKey();
            Wikitty wikitty = restore(id);
            TreeNode node = new TreeNodeImpl(wikitty);

            result.put(node, count);
        }

        return result;
    }

    public Wikitty restoreVersion(String wikittyId, String version) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * Class used for import process, this class retain numberForCommit object
     * before to send it to storage.
     */
    static protected class WikittyBatchUpdate {
        // TODO poussin 20090902 do configurable numberForCommit
        protected int numberForCommit = 1000;
        int currentAdded = 0;
        protected Map<String, WikittyExtension> exts =
                new HashMap<String, WikittyExtension>();
        protected List<Wikitty> wikitties = new LinkedList<Wikitty>();

        WikittyService ws;

        public WikittyBatchUpdate(WikittyService ws) {
            this.ws = ws;
        }

        public void addExtension(WikittyExtension ext) {
            exts.put(ext.getId(), ext);
            inc();
        }

        public void addWikitty(Wikitty w) {
            wikitties.add(w);
            inc();
        }

        /**
         * search extension in local extension list and if missed restore
         * extension from internal WikittyService
         * @param id
         * @return
         */
        public WikittyExtension getExtension(String id) {
            WikittyExtension result = exts.get(id);
            if (result == null) {
                result = ws.restoreExtension(id);
            }
            return result;
        }

        public void flush() {
            ws.storeExtension(exts.values());
            ws.store(wikitties, true);
            exts.clear();
            wikitties.clear();
            currentAdded = 0;
        }


        protected void inc() {
            currentAdded++;
            if (currentAdded >= numberForCommit) {
                flush();
            }
        }
    }


    public void syncImportFromXml(String xml) {
        Reader reader = new StringReader(xml);
        ImportTask task = new ImportTask(this, reader);
        task.run();
    }


    public void syncImportFromUri(String uri) {
        try {
            URL url = new URL(uri);
            Reader reader = new InputStreamReader(url.openStream());
            ImportTask task = new ImportTask(this, reader);
            task.run();
        } catch (Exception eee) {
            throw new WikittyException(eee);
        }
    }


    public String asyncImportFromUri(String uri) {
        try {
            URL url = new URL(uri);
            Reader reader = new InputStreamReader(url.openStream());
            ImportTask task = new ImportTask(this, reader);
            FutureTask<String> future = new FutureTask<String>(task, null);
            importExportExecutor.submit(future);

            String jobId = UUID.randomUUID().toString();
            importExportTask.put(jobId, future);
            return jobId;
        } catch (Exception eee) {
            throw new WikittyException(eee);
        }
    }


    static public class ImportTask implements Runnable {
        protected WikittyService ws;
        protected Reader reader;

        public ImportTask(WikittyService ws, Reader reader) {
            this.ws = ws;
            this.reader = reader;
        }

        public void run() {
            try {
                XmlPullParserFactory factory = XmlPullParserFactory.newInstance(
                        System.getProperty(XmlPullParserFactory.PROPERTY_NAME), null);
                factory.setNamespaceAware(true);
                XmlPullParser xpp = factory.newPullParser();
                xpp.setInput(reader);

                WikittyExtension ext = null;
                Wikitty w = null;
                String CDATA = null;

                WikittyBatchUpdate batchUpdate = new WikittyBatchUpdate(ws);

                long time = System.currentTimeMillis();

                int eventType = xpp.getEventType();
                do {
                    String objectVersion = null;
                    if (eventType == xpp.START_DOCUMENT) {
                        log.info("start XML import at " + new Date());
                    } else if (eventType == xpp.END_DOCUMENT) {
                        time = System.currentTimeMillis() - time;
                        log.info("XML import in (ms)" + time);
                    } else if (eventType == xpp.START_TAG) {
                        String name = xpp.getName();
                        if ("extension".equals(name)) {
                            String extName = xpp.getAttributeValue(null, "name");
                            String version = xpp.getAttributeValue(null, "version");
                            String requires = xpp.getAttributeValue(null, "requires");
                            ext = new WikittyExtension(extName, version, requires, new LinkedHashMap<String, FieldType>());
                        } else if ("object".equals(name)) {
                            String id = xpp.getAttributeValue(null, "id");
                            objectVersion = xpp.getAttributeValue(null, "version");
                            String extensions = xpp.getAttributeValue(null, "extensions");
                            w = new Wikitty(id);
                            String[] extensionList = extensions.split(",");
                            for (String extId : extensionList) {
                                WikittyExtension e = batchUpdate.getExtension(extId);
                                w.addExtension(e);
                            }
                        }
                    } else if (eventType == xpp.END_TAG) {
                        String name = xpp.getName();
                        if ("extension".equals(name)) {
                            batchUpdate.addExtension(ext);
                            ext = null;
                        } else if ("object".equals(name)) {
                            w.setVersion(objectVersion);
                            batchUpdate.addWikitty(w);
                            w = null;
                        } else if (ext != null && "field".equals(name)) {
                            FieldType type = new FieldType();
                            String fieldName = WikittyUtil.parseField(CDATA, type);
                            ext.addField(fieldName, type);
                        } else if (w != null) {
                            String[] fq = name.split("\\.");
                            String extensionName = fq[0];
                            String fieldName = fq[1];
                            FieldType fieldType = w.getFieldType(name);
                            if(fieldType.isCollection()) {
                                w.addToField(extensionName, fieldName, CDATA);
                            } else {
                                w.setField(extensionName, fieldName, CDATA);
                            }
                        }
                    } else if (eventType == xpp.TEXT) {
                        CDATA = xpp.getText();
                    }
                    eventType = xpp.next();
                } while (eventType != xpp.END_DOCUMENT);

                // don't forget to flush batchUpdate :)
                batchUpdate.flush();
            } catch (Exception eee) {
                throw new WikittyException(eee);
            }
        }
    } // end ImportTask

    public String asyncExportAllByCriteria(Criteria criteria) {
        try {
            criteria.setFirstIndex(0).setEndIndex(ALL_ELEMENTS);
            PagedResult<Wikitty> pageResult = findAllByCriteria(criteria);

            String jobId = UUID.randomUUID().toString();

            File file = new File(EXPORT_DIRECTORY, jobId);
            String url = EXPORT_URL + jobId;
            Writer result = new FileWriter(file);
            ExportTask task = new ExportTask(pageResult, result);
            FutureTask<String> future = new FutureTask<String>(task, url);
            importExportExecutor.submit(future);

            importExportTask.put(jobId, future);
            return jobId;
        } catch (Exception eee) {
            throw new WikittyException(eee);
        }
    }


    public String syncExportAllByCriteria(Criteria criteria) {
        criteria.setFirstIndex(0).setEndIndex(ALL_ELEMENTS);
        PagedResult<Wikitty> pageResult = findAllByCriteria(criteria);
        StringWriter result = new StringWriter();
        ExportTask task = new ExportTask(pageResult, result);
        task.run();
        return result.toString();
    }


    static public class ExportTask implements Runnable {
        protected PagedResult<Wikitty> pageResult;
        protected Writer result;

        public ExportTask(PagedResult<Wikitty> pageResult, Writer result) {
            this.pageResult = pageResult;
            this.result = result;
        }

        public void run() {
            try {
                // keep extension already done
                Set<String> extDone = new HashSet<String>();
                result.write("<wikengo>\n");
                for (Wikitty w : pageResult.getAll()) {
                    String extensionList = "";
                    for (WikittyExtension ext : w.getExtensions()) {
                        String id = ext.getId();
                        extensionList += "," + id;
                        if (!extDone.contains(id)) {
                            extDone.add(id);
                            result.write("  <extension name='" + ext.getName() + "' version='" + ext.getVersion() + "' requires='" + ext.getRequires() + "'>\n");
                            for (String fieldName : ext.getFieldNames()) {
                                String def = ext.getFieldType(fieldName).toDefinition(fieldName);
                                result.write("    <field>" + def + "</field>\n");
                            }
                            result.write("  </extension>\n");
                        }
                    }
                    if (!"".equals(extensionList)) {
                        // delete first ','
                        extensionList = extensionList.substring(1);
                    }
                    result.write("  <object id='" + w.getId() + "' version='" + w.getVersion() + "' extensions='" + extensionList + "'>\n");
                    for (String fieldName : w.fieldNames()) {
                        FieldType type = w.getFieldType(fieldName);
                        if (type.isCollection()) {
                            for (Object o : (List) w.getFqField(fieldName)) {
                                result.write("    <" + fieldName + ">" + WikittyUtil.toString(type, o) + "</" + fieldName + ">\n");
                            }
                        } else {
                            result.write("    <" + fieldName + ">" + WikittyUtil.toString(type, w.getFqField(fieldName)) + "</" + fieldName + ">\n");
                        }
                    }
                    result.write("  </object>\n");
                }
                result.write("</wikengo>\n");
            } catch (IOException eee) {
                throw new WikittyException(eee);
            }
        }
    } // end ExportTask


    public JobState infoJob(String jobId) {
        try {
            Future<String> future = importExportTask.get(jobId);
            JobState result = new JobState();
            if (future.isDone()) {
                result.status = "done";
                result.resourceUri = future.get();
            } else if (future.isCancelled()) {
                result.status = "cancelled";
            } else {
                result.status = "inProgress";
            }
            return result;
        } catch (Exception eee) {
            throw new WikittyException(eee);
        }
    }


    public void cancelJob(String jobId) {
        Future future = importExportTask.get(jobId);
        future.cancel(true); // true to kill process, perhaps to strong ?
    }


    public void freeJobResource(String jobId) {
        Future<String> future = importExportTask.remove(jobId);
        if (future != null) {
            File file = new File(EXPORT_DIRECTORY, jobId);
            file.delete();
        }
    }

    /**
     * Changes the data directory
     * @param newDataDir the new data directory path
     * @param oldDataDir the old data directory path.
     *          If null, the data in the old directory will not be copied.
     */
    public void changeDataDir(String newDataDir, String oldDataDir) {
        getSearchEngine().changeDataDir(newDataDir, oldDataDir);
    }

}
