/*
 * #%L
 * Vradi :: Services
 * 
 * $Id: VradiStorageService.java 1767 2010-11-12 12:05:53Z sletellier $
 * $HeadURL: svn+ssh://sletellier@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.4.0/vradi-services/src/main/java/com/jurismarches/vradi/services/VradiStorageService.java $
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package com.jurismarches.vradi.services;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import com.jurismarches.vradi.entities.Status;
import org.nuiton.wikitty.Criteria;
import org.nuiton.wikitty.WikittyExtension;

import com.jurismarches.vradi.beans.XmlStreamImportResult;
import com.jurismarches.vradi.entities.Group;
import com.jurismarches.vradi.entities.QueryMaker;
import com.jurismarches.vradi.entities.Sending;
import com.jurismarches.vradi.entities.Session;
import com.jurismarches.vradi.entities.User;
import com.jurismarches.vradi.entities.VradiUser;
import com.jurismarches.vradi.entities.WebHarvestStream;
import com.jurismarches.vradi.entities.XmlStream;

/**
 * Vradi storage service.
 * 
 * This service is intended to be used in remote mode.
 * 
 * Typical usage is code that can be run only on server :
 * <ul>
 *  <li>Openoffice template generation
 *  <li>Mail sending
 * </ul>
 *
 * @author morin
 * @version $Revision: 1767 $ $Date: 2010-11-12 13:05:53 +0100 (ven., 12 nov. 2010) $
 */
public interface VradiStorageService {


    /**
     * Create all default status.
     *
     * @return all status
     * @throws VradiException
     */
    List<Status> createDefaultStatuses() throws VradiException;

    /**
     * Binds all queryMaker with the forms found by their queries
     * and not already sent.
     */
    Session bindForms(Session session) throws VradiException;

    /**
     * Create new Sending
     *
     * @param session
     * @param group
     * @param formsIdsToBind
     * @return
     * @throws VradiException
     */
    List<Sending> createAllSending(String sessionId, Group group, List<String> formsIdsToBind) throws VradiException;

    /**
     * Create new Sending
     *
     * @param session
     * @param user
     * @param formsIdsToBind
     * @return
     * @throws VradiException
     */
    Sending createSending(String sessionId, User user, List<String> formsIdsToBind) throws VradiException;

    Session createNewSession() throws VradiException;

    /**
     * Creates and store forms from an {@code WebHarvestXmlStream} by using the XmlStreamBinding
     * to link xml stream field values with form fields.
     *
     * Input xmlStream is modified by this method (wikitty obselete).
     * 
     * @param webHarvestStream stream to import
     * @return a structure containing :
     * - the number of created forms
     * - the number of already existing forms
     * - the number of forms created with date parsing error
     * - the number of forms created with number parsing error
     *
     * @throws VradiException for various possible errors
     */
    XmlStreamImportResult importFormsFromWebHarvestStream(WebHarvestStream webHarvestStream) throws VradiException;

    /**
     * Get association between template fields and extension fields.
     *
     * @param extensionName
     * @param templateName
     * @return association map or {@code null} is there is no association yet
     * @throws VradiException
     */
    Map<String, String> getAssociatedFields(String extensionName,
            String templateName) throws VradiException;

    /**
     * Lists the template filename (simple name) associated with the specified
     * extension.
     *
     * This need to be called by a WS because an {@link WikittyExtension}
     * doesn't have a files fields.
     *
     * @param extension
     * @return
     */
    Collection<String> getTemplateFilenames(WikittyExtension extension);
                                                                 
    /**
     * Creates and store forms from an {@code XmlStream} by using the XmlStreamBinding
     * to link xml stream field values with form fields.
     *
     * Input xmlStream is modified by this method (wikitty obselete).
     * 
     * @param xmlStream stream to import
     * @return a structure containing :
     * - the number of created forms
     * - the number of already existing forms
     * - the number of forms created with date parsing error
     * - the number of forms created with number parsing error
     *
     * @throws VradiException for various possible errors
     */
    XmlStreamImportResult importFormsFromXmlStream(XmlStream xmlStream) throws VradiException;

    /**
     * Create a {@code VradiUser}.
     *
     * @param user the user to create
     * @return the vradiUser updated or {@code null} if user can't be created (already exists)
     * @throws VradiException
     */
    VradiUser createUser(VradiUser user) throws VradiException;

    /**
     * Checks if {@code userName} and {@code userPassword} matches a valid {@code VradiUser}.
     *
     * @param login the login of a WikittyUser
     * @param md5Password the password of a WikittyUser (md5 encoded)
     * @return a {@code WikittyUser} or null if either userName or userPassword don't match
     * @throws VradiException
     */
    VradiUser loginUser(String login, String md5Password) throws VradiException;

    /*
     * Import data from an XML file.
     *
     * @param file the XML file containing datas
     * @throws VradiException
     *
    void importData(File file) throws VradiException;

    /*
     * Export the datas into an XML format.
     *
     * @return the datas as an XML String
     * @throws VradiException
     *
    String exportData() throws VradiException;*/

    /*
     * Create or Update a list of <code>Forms</code>.
     *
     * @param forms the list of forms to insert or update
     * @return an updated list of <code>Form</code>
     *
    List<Form> updateForms(List<Form> forms) throws VradiException;*/

    /**
     * Import all data as CSV.
     * 
     * FIXME EC-20100415 will not work in remote mode (or uri maybe be
     * valid for remote server too)
     * 
     * @param uri uri used to read input stream
     * @throws VradiException if any error happen during import
     */
    void importAsCSV(String uri) throws VradiException;

    /**
     * Get csv export.
     * 
     * @param criteria export criteria
     * @return csv export
     * @throws VradiException if any error happen during export
     */
    String exportAsCSV(Criteria criteria) throws VradiException;

    /**
     * Reindex the datas.
     */
    void reindexData() throws VradiException;

    /**
     * Archives the queries of a user, client or group in a RSS file.
     * 
     * @param queryMaker the query maker whose queries are archived
     * @throws VradiException
     */
    void archiveQueries(QueryMaker queryMaker) throws VradiException;

    /**
     * Save association between template fields and extensions fields.
     * 
     * @param extensionName
     * @param templateName
     * @param fieldMap
     * @throws VradiException
     */
    void setAssociatedFields(String extensionName, String templateName,
            Map<String, String> fieldMap) throws VradiException;

    /**
     * Generate pdf file for specified form.
     * 
     * @param formId form id to generate pdf
     * @param force force regeneration
     * @return uri to download file
     */
    String generatePDF(String formId, boolean force) throws VradiException;

    /**
     * Remove all sending corresponding with form and group from session
     *
     * @param session concerned
     * @param formsIds concerned
     * @param queryMaker concerned
     * @return List of sending after deletion
     */
    List<Sending> removeAllSending(String sessionId, List<String> formsIds, Group group) throws VradiException;

    /**
     * Remove sending corresponding with form and user from session
     *
     * @param session concerned
     * @param formsIds concerned
     * @param queryMaker concerned
     * @return sending after deletion
     */
    Sending removeSending(String sessionId, List<String> formsIds, User user) throws VradiException;
    
    /**
     * Get template document fields names.
     * 
     * @param extension extension
     * @param templateName template name
     * @return field names
     * @throws VradiException
     */
    List<String> getDocumentFields(WikittyExtension extension, String templateName) throws VradiException;

    /**
     * Send session pass in param
     *
     * @param sessionId session id concerned
     * @return Logged messages
     * @throws VradiException exception is occured
     */
    String sendMessages(String sessionId) throws VradiException;

    /**
     * Send sending pass in param to sending user email
     *
     * @param sendingId sending id concerned
     * @param sessionParagraph paragraph to send
     * @return Logged messages
     * @throws VradiException exception is occured
     */
    String sendMessage(String sendingId, String sessionParagraph) throws VradiException;

    /**
     * Send sending pass in param to specific email
     *
     * @param sendingId sending id concerned
     * @param sessionParagraph paragraph to send
     * @param email specific
     * @return Logged messages
     * @throws VradiException exception is occured
     */
    String sendMessage(String sendingId, String sessionParagraph, String email) throws VradiException;

    void receiveMessages() throws VradiException;

    Session stopSentMail(Session session) throws VradiException;

    /**
     * Generate all the PDFs needed to send a session.
     *
     * @param sessionId the session wikitty id
     * @throws VradiException
     */
    void generatePDFForSession(String sessionId) throws VradiException;

    /**
     * Export all the clients, users and groups in database
     *
     * @return the csv export as a String
     * @throws VradiException if any error happen during export
     */
    String exportClientDB() throws VradiException;

    /**
     * Delete session and sending associated
     * This method return a {@link VradiException} if session to delete is not active
     *
     * @param sessionId to delete
     * @throws VradiException if an error occur during session deletion
     */
    void deleteSession(String sessionId) throws VradiException;

    /**
     * Check if template file form extension name in param
     *
     * @param extensionName to check
     * @return true if file exist
     * @throws VradiException if an error occur during checking template
     */
    boolean checkTemplateExist(String extensionName) throws VradiException;

    /**
     * Check if template file form extension name in param
     *
     * @param formsIds to update
     * @return false if no session opened found
     * @throws VradiException if an error occur during checking template
     */
    boolean removeFormsFromSession(String session, List<String> formsIds) throws VradiException;

    /**
     * Method call to revert froms deleted in sending
     *
     * @param sendingId to update
     * @param formsIdsToUpdate deleted to revert
     * @return updated sending
     */
    Sending revertDeletion(String sendingId, List<String> formsIdsToUpdate);

    /**
     * Revert forms deleted for user
     *
     * @param sessionId session concerned
     * @param userId user concerned
     * @param formId form to revert
     * @return sending impacted
     */
    Sending revertUserDeletion(String sessionId, String userId, String formId);

    /**
     * Revert forms deleted for group
     *
     * @param sessionId session concerned
     * @param groupId user concerned
     * @param formId form to revert
     * @return sending impacted
     */
    List<Sending> revertGroupDeletion(String sessionId, String groupId, String formId);


}
