/*
 * *##%
 * Vradi :: Services
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.services.managers;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.sharengo.wikitty.TreeNodeImpl;
import org.sharengo.wikitty.Criteria;
import org.sharengo.wikitty.FacetTopic;
import org.sharengo.wikitty.PagedResult;
import org.sharengo.wikitty.WikittyProxy;
import org.sharengo.wikitty.search.Element;
import org.sharengo.wikitty.search.Like;
import org.sharengo.wikitty.search.Search;

import com.jurismarches.vradi.VradiConstants;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.QueryMaker;
import com.jurismarches.vradi.entities.QueryMakerImpl;
import com.jurismarches.vradi.entities.Thesaurus;
import com.jurismarches.vradi.services.VradiException;
import com.jurismarches.vradi.services.dto.VradiQueryBean;
import com.jurismarches.vradi.services.search.CompareFilter;
import com.jurismarches.vradi.services.search.Filter;
import com.jurismarches.vradi.services.search.FilterList;
import com.jurismarches.vradi.services.search.VradiQueryParser;

/**
 * Class containing the methods to manage the thesaurus :
 * - node creation, update, retrieving, deletion
 * - get child node number
 * - get children
 *
 * @author schorlet
 * @version $Revision: 783 $ $Date: 2010-04-28 12:34:13 +0200 (mer., 28 avril 2010) $
 * @date 2010-01-22 20:18:29
 */
public class ThesaurusManager {
    private static final Log log = LogFactory.getLog(ThesaurusManager.class);

    private final WikittyProxy proxy;
    
    public ThesaurusManager(WikittyProxy proxy) {
        this.proxy = proxy;
    }

    public TreeNodeImpl getRootThesaurus() throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getRootThesaurus()");
        }

        Criteria criteria = Search.query()
                .eq(Element.ELT_EXTENSION, TreeNodeImpl.EXT_TREENODE)
                .eq(TreeNodeImpl.FQ_FIELD_NAME, VradiConstants.ROOT_THESAURUS_NAME)
                .criteria();

        TreeNodeImpl rootThesaurus = proxy
                .findByCriteria(TreeNodeImpl.class, criteria);

        if (rootThesaurus == null) {
            log.warn("root thesaurus not found, creating ...");

            rootThesaurus = new TreeNodeImpl();
            rootThesaurus.setName(VradiConstants.ROOT_THESAURUS_NAME);

            rootThesaurus = proxy.store(rootThesaurus);
        }

        log.info("root thesaurus: " + rootThesaurus);
        return rootThesaurus;
    }


    public List<TreeNodeImpl> getAllThesaurus() throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getAllThesaurus()");
        }

        Criteria criteria = Search.query().eq(Element.ELT_EXTENSION,
                Thesaurus.EXT_THESAURUS).criteria();

        PagedResult<TreeNodeImpl> findAllByCriteria = proxy.findAllByCriteria(
                TreeNodeImpl.class, criteria);
        List<TreeNodeImpl> nodes = findAllByCriteria.getAll();
        
        log.info(String.format("found %s thesaurus entries", nodes.size()));
        return nodes;
    }

    public TreeNodeImpl getThesaurus(String thesaurusId) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getThesaurus(" + thesaurusId + ")");
        }
        TreeNodeImpl node = proxy.restore(TreeNodeImpl.class, thesaurusId);
        return node;
    }

    public List<TreeNodeImpl> getThesaurus(List<String> thesaurusIds)
            throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getThesaurus(" + thesaurusIds + ")");
        }
        List<TreeNodeImpl> nodes = proxy.restore(TreeNodeImpl.class, thesaurusIds);

        return nodes;
    }

    public List<TreeNodeImpl> getChildrenThesaurus(String thesaurusId)
            throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getChildrenThesaurus(" + thesaurusId + ")");
        }

        Search query = Search.query();
        query.eq(Element.ELT_EXTENSION, Thesaurus.EXT_THESAURUS);
        query.eq(TreeNodeImpl.FQ_FIELD_PARENT, thesaurusId);

        Criteria criteria = query.criteria();
        PagedResult<TreeNodeImpl> nodes = proxy
                .findAllByCriteria(TreeNodeImpl.class, criteria);
        List<TreeNodeImpl> all = nodes.getAll();

        return all;
    }

    public int getNbFormsForThesaurus(String thesaurusId)
            throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getNbFormsForThesaurus(" + thesaurusId + ")");
        }

        Map.Entry<TreeNodeImpl, Integer> entry =
                proxy.restoreNode(TreeNodeImpl.class, thesaurusId, null);

        if (entry == null) {
            return 0;
        }

        return entry.getValue();
    }
    
    /**
     * Propose thesaurus nodes that might be in relation with a specified form
     *
     * @param form      the <code>Form</code> containing the information needed
     *                  to search the thesaurus nodes
     * @param thesaurus the list of all the tree nodes of the thesaurus
     * @return a list of <code>TreeNodeImpl</code>
     * @throws VradiException
     */
    public List<TreeNodeImpl> proposeThesaurus(Form form,
            List<TreeNodeImpl> thesaurus) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("proposeThesaurus(form)");
        }
        
        if (form == null) {
            return null;
        }
        
        // search for the specified form
        Criteria criteria = Search.query()
            .eq(Element.ELT_EXTENSION, Form.EXT_FORM)
            .eq(Form.FQ_FIELD_ID, form.getId())
            .criteria();

        // search within the specified thesaurus nodes or all
        List<TreeNodeImpl> nodes = thesaurus;
        if (nodes == null) {
            nodes = getAllThesaurus();
        }

        Set<String> formNodes = form.getThesaurus();
        
        // add a facet criteria for each thesaurus node id,
        // searching for tags of each thesaurus nodes
        for (TreeNodeImpl node : nodes) {
            if (formNodes != null && formNodes.contains(node.getWikittyId())) {
                continue;
            }

            List<String> tags = node.getWikitty().getFieldAsList(Thesaurus.EXT_THESAURUS,
                    Thesaurus.TAGS_FIELD, String.class);

            if (tags == null || tags.isEmpty()) {
                continue;
            }
            
            Search tagSearch = Search.query(Search.KIND.OR);
            for (String tag : tags) {
                if (!tag.isEmpty()) {
                    tagSearch.keyword(tag);
                }
            }

            Criteria facetCriteria = tagSearch.criteria(node.getWikittyId());
            criteria.addFacetCriteria(facetCriteria);
        }

        // execute the search
        PagedResult<Form> forms = proxy.findAllByCriteria(Form.class, criteria);
        List<TreeNodeImpl> result = new ArrayList<TreeNodeImpl>();
        
        // collects topic names (which are thesaurus node ids)
        if (forms != null && forms.getNumFound() > 0) {
            List<String> thesaurusIds = new ArrayList<String>();
            
            Map<String, List<FacetTopic>> facetsMap = forms.getFacets();
            for (Map.Entry<String, List<FacetTopic>> entry : facetsMap.entrySet()) {
                List<FacetTopic> facetTopics = entry.getValue();
                for (FacetTopic facetTopic : facetTopics) {
                    if (facetTopic.getCount() > 0) {
                        thesaurusIds.add(facetTopic.getTopicName());
                        break;
                    }
                }
            }
            
            if (!thesaurusIds.isEmpty()) {
                result = getThesaurus(thesaurusIds);
            }
        }
        
        return result;
    }

    /**
     * Gets the query makers whose queries are potentially to modify
     * after a thesaurus node modification
     *
     * @param thesaurusName the modified thesaurus node
     * @return a map containing the query makers and their queries which contains
     *         the thesaurus node name
     */
    public Map<QueryMaker, List<VradiQueryBean>> getQueriesToModifyAfterThesaurusModification(
            String thesaurusName) {
        if (log.isDebugEnabled()) {
            log.debug("getQueriesToModifyAfterThesaurusModification(" + thesaurusName + ")");
        }
        Map<QueryMaker, List<VradiQueryBean>> results = new HashMap<QueryMaker, List<VradiQueryBean>>();
        
        if (thesaurusName == null) {
            return results;
        }

        Criteria criteria = Search.query()
            .eq(Element.ELT_EXTENSION, QueryMaker.EXT_QUERYMAKER)
            .like(QueryMaker.FQ_FIELD_QUERIES, thesaurusName, Like.SearchAs.AsText)
            .criteria();
        
        PagedResult<QueryMaker> pagedResult = proxy.findAllByCriteria(QueryMaker.class, criteria);
        List<QueryMaker> queryMakers = pagedResult.getAll();

        for (QueryMaker queryMaker : queryMakers) {
            QueryMaker realQueryMaker = ClientManager.castAsRealQueryMaker((QueryMakerImpl) queryMaker);
            if (realQueryMaker == null) {
                continue;
            }
            
            Set<String> queries = queryMaker.getQueries();
            List<VradiQueryBean> queriesToModify = new ArrayList<VradiQueryBean>();
            
            for (String query : queries) {
                try {
                    VradiQueryBean queryBean = new VradiQueryBean(query, queryMaker.getWikittyId());
                    FilterList filter = VradiQueryParser.parse(queryBean.getQuery());
                    
                    if (isThesaurusInQuery(filter, thesaurusName)) {
                        queriesToModify.add(queryBean);
                    }
                    
                } catch (Exception e) {
                    // ignored exception
                    log.warn(e.getMessage(), e);
                }
            }
            
            if (!queriesToModify.isEmpty()) {
                results.put(realQueryMaker, queriesToModify);
            }
        }

        return results;
    }

    boolean isThesaurusInQuery(FilterList list, String thesaurusName) {
        boolean insideQuery = false;
        List<Filter> filters = list.getFilters();
        
        for (Filter filter : filters) {
            if (filter instanceof FilterList) {
                insideQuery = isThesaurusInQuery((FilterList) filter, thesaurusName);

            } else if (filter instanceof CompareFilter) {
                insideQuery = isThesaurusInQuery((CompareFilter) filter, thesaurusName);
            }
            
            if (insideQuery) {
                break;
            }
        }
        return insideQuery;
    }

    boolean isThesaurusInQuery(CompareFilter compareFilter, String thesaurusName) {
        String name = compareFilter.getName();
        String value = compareFilter.getValue();
        return (SearchManager.ALIAS_THESAURUS_1.equals(name)
                || SearchManager.ALIAS_THESAURUS_2.equals(name))
                && thesaurusName.equals(value);
    }

    void replaceThesaurusInQuery(FilterList list, String oldThesaurusName,
            String newThesaurusName) {
        List<Filter> filters = list.getFilters();
        
        for (Filter filter : filters) {
            if (filter instanceof FilterList) {
                replaceThesaurusInQuery((FilterList) filter, oldThesaurusName,
                        newThesaurusName);

            } else if (filter instanceof CompareFilter) {
                replaceThesaurusInQuery((CompareFilter) filter,
                        oldThesaurusName, newThesaurusName);
            }
        }
    }

    void replaceThesaurusInQuery(CompareFilter compareFilter,
            String oldThesaurusName, String newThesaurusName) {
        String value = compareFilter.getValue();
        
        if (value.equals(oldThesaurusName)) {
            String name = compareFilter.getName();

            if (SearchManager.ALIAS_THESAURUS_1.equals(name) ||
                SearchManager.ALIAS_THESAURUS_2.equals(name)) {
            
                compareFilter.setValue(newThesaurusName);
            }
        }
    }

}
