package com.jurismarches.vradi.services.dto;

import java.io.IOException;
import java.io.Serializable;
import java.io.StringWriter;

import au.com.bytecode.opencsv.CSVParser;
import au.com.bytecode.opencsv.CSVWriter;

/**
 * VradiQueryBean.
 *
 * @author schorlet
 * @version $Revision: 684 $ $Date: 2010-04-07 22:59:41 +0200 (mer., 07 avril 2010) $
 * @since 16 mars 2010 14:25:12
 */
public final class VradiQueryBean implements Serializable {
    private static final long serialVersionUID = 1L;
    
    private final String queryLine;
    private final String[] queryArray;
    private final String id;
    
    public VradiQueryBean() {
        this.id = null;
        this.queryLine = "";
        this.queryArray = new String[3];
    }
    
    /**
     * Decode the specified <code>queryLine</code>. The queryLine must be CSV formatted.
     * 
     * @param queryLine the formatted query.
     * @throws IOException if <code>queryLine</code> is badly formatted.
     */
    public VradiQueryBean(String queryLine) throws IOException {
        this(queryLine, null);
    }
    
    public VradiQueryBean(String queryLine, String id) throws IOException {
        this.id = id;
        this.queryLine = queryLine;
        this.queryArray = decode(queryLine, 3);
    }
    
    /**
     * Creates a CSV formatted query from the specified <code>queryArray</code>.
     * @param queryArray the string composing the query
     */
    private VradiQueryBean(String[] queryArray, String id) {
        if (queryArray.length < 3) {
            throw new IllegalArgumentException("queryArray length is less than 3");
        }
        
        this.id = id;
        this.queryArray = queryArray.clone();
        this.queryLine = encode(queryArray);
    }
    
    public String getQuery() {
        return queryArray[0];
    }
    
    public String getName() {
        return queryArray[1];
    }
    
    public String getDescription() {
        return queryArray[2];
    }
    
    @Override
    public String toString() {
        return queryLine;
    }
    
    public String[] toArray() {
        return queryArray;
    }
    
    public VradiQueryBean setQuery(String value) {
        return create(value, 0);
    }
    
    public VradiQueryBean setName(String value) {
        return create(value, 1);
    }
    
    public VradiQueryBean setDescription(String value) {
        return create(value, 2);
    }
    
    private VradiQueryBean create(String value, int pos) {
        String[] newArray = queryArray.clone();
        newArray[pos] = value;
        return new VradiQueryBean(newArray, id);
    }
    
    /**
     * Parse the specified line and returns a string array of the specified size.
     * 
     * @param line the string to parse
     * @return the comma-tokenized list of elements
     * @throws IOException
     */
    static String[] decode(String line, int size) throws IOException {
        CSVParser csvParser = new CSVParser();
        String[] parseLine = csvParser.parseLine(line);
        
        String[] result = new String[size];
        for (int i = 0; i < parseLine.length; i++) {
            result[i] = parseLine[i];
        }
        
        return result;
    }

    /**
     * Escape each columns of the specified array and return a valid CSV line.
     * 
     * @param columns columns composing the CVS string
     * @return a CSV formatted string
     */
    static String encode(String[] columns) {
        StringWriter writer = new StringWriter();
        // creates a CSVWriter with all default options but with no line end
        CSVWriter csvwriter = new CSVWriter(writer, CSVWriter.DEFAULT_SEPARATOR,
                CSVWriter.DEFAULT_QUOTE_CHARACTER, CSVWriter.DEFAULT_ESCAPE_CHARACTER, "");
        csvwriter.writeNext(columns);
        return writer.toString();
    }
    
    /**
     * Additionnal attribute. It does not belongs to the query definition.
     */
    public String getId() {
        return id;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((id == null) ? 0 : id.hashCode());
        result = prime * result
                + ((queryLine == null) ? 0 : queryLine.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        
        VradiQueryBean other = (VradiQueryBean) obj;
        if (id == null) {
            if (other.id != null) {
                return false;
            }
        } else if (!id.equals(other.id)) {
            return false;
        }
        
        if (queryLine == null) {
            if (other.queryLine != null) {
                return false;
            }
        } else if (!queryLine.equals(other.queryLine)) {
            return false;
        }
        
        return true;
    }
    
    
    
}
