package org.nuiton.topia.persistence.internal;

/*
 * #%L
 * ToPIA :: Persistence
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2004 - 2014 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.topia.persistence.TopiaEntityContextable;
import org.nuiton.topia.persistence.internal.support.TopiaFiresSupport;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.beans.VetoableChangeListener;
import java.beans.VetoableChangeSupport;
import java.lang.ref.WeakReference;
import java.util.Date;

import com.google.common.base.Objects;

/**
 * Base class of each entity. It contains the common attributes and a part of the entities event support.
 *
 * @author poussin <poussin@codelutin.com>
 * @author Arnaud Thimel (Code Lutin)
 */
public abstract class AbstractTopiaEntity implements TopiaEntity {

    private static final long serialVersionUID = -7458577454878852241L;

    /**
     * Default instance used as fallback by the entities if they are out of a ToPIA runtime scope.
     */
    protected static final TopiaFiresSupport DEFAULT_INSTANCE = new TopiaFiresSupport();

    protected String topiaId;

    protected long topiaVersion;

    protected Date topiaCreateDate = new Date();

    transient protected boolean deleted;

    /**
     * A potential instance that may be injected by the Dao or retrieved in a contextable entity. The TopiaFiresSupport
     * instance can be linked to the {@link org.nuiton.topia.persistence.TopiaPersistenceContext} instance, thus its
     * life cycle may be shorter than the entity. This is why it is kept as a {@link java.lang.ref.WeakReference}.
     *
     * If not present, the entity will use the <code>DEFAULT_INSTANCE</code> as a fallback TopiaFiresSupport.
     */
    transient protected WeakReference<TopiaFiresSupport> firesSupport;

    transient protected PropertyChangeSupport readListeners;

    transient protected PropertyChangeSupport writeListeners;

    transient protected VetoableChangeSupport readVetoables;

    transient protected VetoableChangeSupport writeVetoables;

    public void setFiresSupport(TopiaFiresSupport firesSupport) {
        this.firesSupport = new WeakReference<TopiaFiresSupport>(firesSupport);
    }

    protected TopiaFiresSupport getFiresSupportOrNull() {
        TopiaFiresSupport result = firesSupport == null ? null : firesSupport.get();
        if (result == null) { // First call or the weak reference has expired
            if (this instanceof TopiaEntityContextable) {
                TopiaEntityContextable contextable = (TopiaEntityContextable) this;
                AbstractTopiaDao entityDao = (AbstractTopiaDao) contextable.getGenericEntityDao();
                // Dao may be null if the entity isn't managed by ToPIA (ie. created via "new XxxImpl()")
                if (entityDao != null) {
                    result = entityDao.getTopiaFiresSupport();
                    setFiresSupport(result);
                }
            }
        }
        return result;
    }

    protected TopiaFiresSupport getFiresSupport() {
        TopiaFiresSupport result = Objects.firstNonNull(getFiresSupportOrNull(), DEFAULT_INSTANCE);
        return result;
    }

    /**
     * Initialize {@link #readListeners} at first use or after deserialisation.
     *
     * @param create indicates if the PropertyChangeSupport can be created if it does not exist
     * @return readListeners
     */
    protected PropertyChangeSupport getReadPropertyChangeSupport(boolean create) {
        if (readListeners == null && create) {
            readListeners = new PropertyChangeSupport(this);
        }
        return readListeners;
    }

    /**
     * Initialize {@link #writeListeners} at first use or after deserialisation.
     *
     * @param create indicates if the PropertyChangeSupport can be created if it does not exist
     * @return writeListeners
     */
    protected PropertyChangeSupport getWritePropertyChangeSupport(boolean create) {
        if (writeListeners == null && create) {
            writeListeners = new PropertyChangeSupport(this);
        }
        return writeListeners;
    }

    /**
     * Initialize {@link #readVetoables} at first use or after deserialisation.
     *
     * @param create indicates if the VetoableChangeSupport can be created if it does not exist
     * @return readVetoables
     */
    protected VetoableChangeSupport getReadVetoableChangeSupport(boolean create) {
        if (readVetoables == null && create) {
            readVetoables = new VetoableChangeSupport(this);
        }
        return readVetoables;
    }

    /**
     * Initialize {@link #writeVetoables} at first use or after deserialisation.
     *
     * @param create indicates if the VetoableChangeSupport can be created if it does not exist
     * @return writeVetoables
     */
    protected VetoableChangeSupport getWriteVetoableChangeSupport(boolean create) {
        if (writeVetoables == null && create) {
            writeVetoables = new VetoableChangeSupport(this);
        }
        return writeVetoables;
    }

    @Override
    public String getTopiaId() {
        return topiaId;
    }

    @Override
    public void setTopiaId(String v) {
        topiaId = v;
    }

    @Override
    public long getTopiaVersion() {
        return topiaVersion;
    }

    @Override
    public void setTopiaVersion(long v) {
        topiaVersion = v;
    }

    @Override
    public Date getTopiaCreateDate() {
        return topiaCreateDate;
    }

    @Override
    public void setTopiaCreateDate(Date topiaCreateDate) {
        this.topiaCreateDate = topiaCreateDate;
    }

    @Override
    public boolean isPersisted() {
        // Is or was the entity persisted ?
        boolean result = topiaId != null;
        // Is the entity deleted ?
        result &= !deleted;
        return result;
    }

    @Override
    public boolean isDeleted() {
        return deleted;
    }

    @Override
    public void notifyDeleted() {
        deleted = true;
    }

    /**
     * We are using the <code>topiaCreateDate</code> for the hashCode because it does not change through time.
     */
    @Override
    public int hashCode() {
        Date date = getTopiaCreateDate();
        //TC-20100220 : il se peut que la date de creation soit nulle
        // lorsque l'entite est utilise comme objet d'edition d'un formulaire
        // par exemple...
        int result = date == null ? 0 : date.hashCode();
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (!(obj instanceof TopiaEntity)) {
            return false;
        }
        TopiaEntity other = (TopiaEntity) obj;
        if (getTopiaId() == null || other.getTopiaId() == null) {
            return false;
        }
        boolean result = getTopiaId().equals(other.getTopiaId());
        return result;
    }

    protected void fireOnPreRead(String propertyName, Object value) {
        VetoableChangeSupport vetoableChangeSupport = getReadVetoableChangeSupport(false);
        if (vetoableChangeSupport != null) {
            TopiaFiresSupport topiaFiresSupport = getFiresSupport();
            topiaFiresSupport.fireOnPreRead(vetoableChangeSupport,
                    this, propertyName, value);
        }
    }

    protected void fireOnPostRead(String propertyName, Object value) {
        PropertyChangeSupport propertyChangeSupport = getReadPropertyChangeSupport(false);
        if (propertyChangeSupport != null) {
            TopiaFiresSupport topiaFiresSupport = getFiresSupport();
            topiaFiresSupport.fireOnPostRead(propertyChangeSupport,
                    this, propertyName, value);
        }
    }

    protected void fireOnPostRead(String propertyName, int index,
                                  Object value) {
        PropertyChangeSupport propertyChangeSupport = getReadPropertyChangeSupport(false);
        if (propertyChangeSupport != null) {
            TopiaFiresSupport topiaFiresSupport = getFiresSupport();
            topiaFiresSupport.fireOnPostRead(propertyChangeSupport,
                    this, propertyName, index, value);
        }
    }

    protected void fireOnPreWrite(String propertyName, Object oldValue,
                                  Object newValue) {
        VetoableChangeSupport vetoableChangeSupport = getWriteVetoableChangeSupport(false);
        if (vetoableChangeSupport != null) {
            TopiaFiresSupport topiaFiresSupport = getFiresSupport();
            topiaFiresSupport.fireOnPreWrite(vetoableChangeSupport,
                    this, propertyName, oldValue, newValue);
        }
    }

    protected void fireOnPostWrite(String propertyName, Object oldValue,
                                   Object newValue) {
        PropertyChangeSupport propertyChangeSupport = getWritePropertyChangeSupport(false);
        if (propertyChangeSupport != null) {
            TopiaFiresSupport topiaFiresSupport = getFiresSupport();
            topiaFiresSupport.fireOnPostWrite(
                    propertyChangeSupport, this, propertyName, oldValue, newValue);
        }
    }

    protected void fireOnPostWrite(String propertyName, int index,
                                   Object oldValue, Object newValue) {
        PropertyChangeSupport propertyChangeSupport = getWritePropertyChangeSupport(false);
        if (propertyChangeSupport != null) {
            TopiaFiresSupport topiaFiresSupport = getFiresSupport();
            topiaFiresSupport.fireOnPostWrite(
                    propertyChangeSupport, this, propertyName, index, oldValue,
                    newValue);
        }
    }

    @Override
    public void addPropertyChangeListener(String propertyName,
                                          PropertyChangeListener listener) {
        getWritePropertyChangeSupport(true).addPropertyChangeListener(propertyName, listener);
    }

    @Override
    public void addPropertyChangeListener(PropertyChangeListener listener) {
        getWritePropertyChangeSupport(true).addPropertyChangeListener(listener);
    }

    @Override
    public void addVetoableChangeListener(String propertyName,
                                          VetoableChangeListener vetoable) {
        getWriteVetoableChangeSupport(true).addVetoableChangeListener(propertyName, vetoable);
    }

    @Override
    public void addVetoableChangeListener(VetoableChangeListener vetoable) {
        getWriteVetoableChangeSupport(true).addVetoableChangeListener(vetoable);
    }

    @Override
    public void removePropertyChangeListener(String propertyName,
                                             PropertyChangeListener listener) {
        getWritePropertyChangeSupport(true).removePropertyChangeListener(propertyName, listener);
    }

    @Override
    public void removePropertyChangeListener(PropertyChangeListener listener) {
        getWritePropertyChangeSupport(true).removePropertyChangeListener(listener);
    }

    @Override
    public void removeVetoableChangeListener(String propertyName,
                                             VetoableChangeListener vetoable) {
        getWriteVetoableChangeSupport(true).removeVetoableChangeListener(propertyName, vetoable);
    }

    @Override
    public void removeVetoableChangeListener(VetoableChangeListener vetoable) {
        getWriteVetoableChangeSupport(true).removeVetoableChangeListener(vetoable);
    }

    @Override
    public void addPropertyListener(String propertyName,
                                    PropertyChangeListener listener) {
        getReadPropertyChangeSupport(true).addPropertyChangeListener(propertyName, listener);
    }

    @Override
    public void addPropertyListener(PropertyChangeListener listener) {
        getReadPropertyChangeSupport(true).addPropertyChangeListener(listener);
    }

    @Override
    public void addVetoableListener(String propertyName,
                                    VetoableChangeListener vetoable) {
        getReadVetoableChangeSupport(true).addVetoableChangeListener(propertyName, vetoable);
    }

    @Override
    public void addVetoableListener(VetoableChangeListener vetoable) {
        getReadVetoableChangeSupport(true).addVetoableChangeListener(vetoable);
    }

    @Override
    public void removePropertyListener(String propertyName,
                                       PropertyChangeListener listener) {
        getReadPropertyChangeSupport(true).removePropertyChangeListener(propertyName, listener);
    }

    @Override
    public void removePropertyListener(PropertyChangeListener listener) {
        getReadPropertyChangeSupport(true).removePropertyChangeListener(listener);
    }

    @Override
    public void removeVetoableListener(String propertyName,
                                       VetoableChangeListener vetoable) {
        getReadVetoableChangeSupport(true).removeVetoableChangeListener(propertyName, vetoable);
    }

    @Override
    public void removeVetoableListener(VetoableChangeListener vetoable) {
        getReadVetoableChangeSupport(true).removeVetoableChangeListener(vetoable);
    }

}
