/* *##% ToPIA - Persistence
 * Copyright (C) 2004 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*/
package org.nuiton.topia.framework;

import org.apache.commons.collections.set.MapBackedSet;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaContextFactory;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.TopiaNotFoundException;
import org.nuiton.topia.event.TopiaContextListener;
import org.nuiton.topia.event.TopiaEntitiesVetoable;
import org.nuiton.topia.event.TopiaEntityListener;
import org.nuiton.topia.event.TopiaEntityVetoable;
import org.nuiton.topia.event.TopiaTransactionListener;
import org.nuiton.topia.event.TopiaTransactionVetoable;
import org.nuiton.topia.persistence.TopiaDAO;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.topia.persistence.TopiaId;
import org.nuiton.util.ArrayUtil;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.DocumentFactory;
import org.dom4j.Element;
import org.dom4j.io.OutputFormat;
import org.dom4j.io.SAXReader;
import org.dom4j.io.XMLWriter;
import org.hibernate.EntityMode;
import org.hibernate.FlushMode;
import org.hibernate.HibernateException;
import org.hibernate.Query;
import org.hibernate.ReplicationMode;
import org.hibernate.SQLQuery;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.hibernate.cfg.Configuration;
import org.hibernate.event.PostDeleteEventListener;
import org.hibernate.event.PostInsertEventListener;
import org.hibernate.event.PostLoadEventListener;
import org.hibernate.event.PostUpdateEventListener;
import org.hibernate.event.PreDeleteEventListener;
import org.hibernate.event.PreInsertEventListener;
import org.hibernate.event.PreLoadEventListener;
import org.hibernate.event.PreUpdateEventListener;
import org.hibernate.tool.hbm2ddl.SchemaExport;
import org.hibernate.tool.hbm2ddl.SchemaUpdate;

import java.beans.PropertyChangeListener;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.io.Writer;
import java.lang.reflect.Field;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.WeakHashMap;
import java.util.zip.GZIPInputStream;
import org.apache.commons.collections.set.SynchronizedSet;
import org.nuiton.topia.persistence.TopiaDAOImpl;
import static org.nuiton.i18n.I18n._;

/**
 * Le TopiaContextImpl est le point d'entre pour acceder aux donnees. Il est
 * configurer par un fichier de propriete
 * <p>
 * List des proprietes disponible
 * <dl>
 * <dt> topia.persistence.properties.file
 * <dd> le fichier de propriété a utiliser pour configurer hibernate
 *
 * <dt> topia.persistence.directories
 * <dd> la liste des repertoires contenant les mappings hibernates (.hbm.xml) la
 * liste de repertoire est separer par des virgules ','
 *
 * <dt> topia.persistence.classes
 * <dd> la liste des classes que doit géré hibernate. On peut tres bien utiliser
 * topia.persistence.directories pour un ensemble d'entié du meme repertoire et
 * topia.persistence.classes pour d'autres classes
 * </dl>
 *
 * TopiaContextImpl.java
 *
 * Created: 23 déc. 2005 16:58:50
 *
 * @author poussin
 *
 * @version $Revision: 1646 $
 *
 * Last update: $Date: 2009-09-24 15:09:10 +0200 (jeu., 24 sept. 2009) $ by : $Author: sletellier $
 */
public class TopiaContextImpl implements TopiaContext, TopiaContextImplementor {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(TopiaContextImpl.class);
    static final public String TOPIA_PERSISTENCE_DIRECTORIES = "topia.persistence.directories";
    static final public String TOPIA_PERSISTENCE_CLASSES = "topia.persistence.classes";
    static final public String TOPIA_PERSISTENCE_PROPERTIES_FILE = "topia.persistence.properties.file";
    /**
     * Le pere de ce context, les contexts initaux n'ont pas de context pere
     */
    protected TopiaContextImplementor parentContext = null;
    /**
     * L'objet configuration utilisé pour la creation de la factory hibernate
     */
    protected Configuration hibernateConfiguration = null;
    /**
     * la factory permettant de recuperer la session hibernate. Seul les
     * TopiaContextImpl initiaux contiennent un hibernateFactory
     */
    protected SessionFactory hibernateFactory = null;
    /**
     * La session utilisé par le TopiaContextImpl
     */
    protected Session hibernate = null;
    /**
     * Indique si le contexte a ete ferme
     */
    protected boolean closed = false;
    /**
     * Propriete de configuration
     */
    protected Properties config = null;
    /**
     * cache des DAO deja chargé pour ce context
     */
    protected Map<Class<? extends TopiaEntity>, TopiaDAO<? extends TopiaEntity>> daoCache = new HashMap<Class<? extends TopiaEntity>, TopiaDAO<? extends TopiaEntity>>();
    /**
     * Set des sous context creer avec un beginTransaction et donc sur lequel
     * nous sommes listener
     */
    // by sletellier 24/09/09 : Fix concurent acces error
//    protected Set<TopiaContextImplementor> childContext = MapBackedSet.decorate(new WeakHashMap<TopiaContextImplementor, Object>());
    protected Set<TopiaContextImplementor> childContext = SynchronizedSet.decorate(MapBackedSet.decorate(new WeakHashMap<TopiaContextImplementor, Object>()));
    /** key: service name; value: service instance */
    protected Map<String, TopiaService> services = null;
    /** */
    protected TopiaFiresSupport firesSupport = new TopiaFiresSupport();
    /** Liste des classes perssitance */
    protected List<Class<?>> persistenceClasses = new ArrayList<Class<?>>();

    /**
     * constructeur utilisé par la factory pour creer les contexts initiaux
     *
     * @param config
     * @throws TopiaNotFoundException
     */
    public TopiaContextImpl(Properties config) throws TopiaNotFoundException {
        this.config = config;
        services = loadServices(config);
        preInitServices(services);
        getHibernateConfiguration(); // force mapping loading
        postInitServices(services);
    }

    protected Map<String, TopiaService> loadServices(Properties config) {
        Map<String, TopiaService> result = new HashMap<String, TopiaService>();
        // recherche des services present dans la config
        for (Enumeration<?> e = config.propertyNames(); e.hasMoreElements();) {
            String key = (String) e.nextElement();
            if (key.matches("^topia\\.service\\.\\w+$")) {
                String classService = config.getProperty(key);
                try {
                    Class<?> forName = Class.forName(classService);
                    Object newInstance = forName.newInstance();
                    TopiaService service = (TopiaService) newInstance;
                    if (key.equals("topia.service." + service.getServiceName())) {
                        result.put(service.getServiceName(), service);
                        log.info(_("topia.persistence.service.loaded", key, classService));
//                        log.info("Service " + key + " loaded by " + classService);
                    } else {
                        log.warn(_("topia.persistence.warn.service.not.loaded", key, service.getServiceName()));
//                        log.warn("Service config key doesn't match service name, disable it: " + key + " != " + service.getServiceName());
                    }
                } catch (Throwable eee) {
                    if (log.isErrorEnabled()) {
                        log.error(_("topia.persistence.error.service.unknown", key, classService));
//                        log.error("Service class unknow for " + key + ": " + classService);
                    }
                    if (log.isDebugEnabled()) {
                        log.debug(eee);
                    }
                }
            }
        }
        return result;
    }

    protected void preInitServices(Map<String, TopiaService> services) {
        for (TopiaService service : services.values()) {
            if (!service.preInit(this)) {
                log.warn(_("topia.persistence.warn.service.not.preInit", service.getServiceName()));
//                log.warn("Can't preInit service disable it: " + service.getServiceName());
            }
        }
    }

    protected void postInitServices(Map<String, TopiaService> services) {
        for (TopiaService service : services.values()) {
            if (!service.postInit(this)) {
                log.warn(_("topia.persistence.warn.service.not.postInit", service.getServiceName()));
//                log.warn("Can't postInit service disable it: " + service.getServiceName());
            }
        }
    }

    @Override
    public Map<String, TopiaService> getServices() {
        TopiaContextImplementor parent = getParentContext();

        Map<String, TopiaService> result = null;
        if (parent != null) {
            result = parent.getServices();
        } else {
            result = services;
        }
        return result;
    }

    protected boolean serviceEnabled(String name) {
        boolean result = getServices().containsKey(name);
        return result;
    }

    protected TopiaService getService(String name) {
        TopiaService result = getServices().get(name);
        return result;
    }

    @Override
    public <E extends TopiaService> boolean serviceEnabled(
            Class<E> interfaceService) {
        boolean result = false;
        try {
            Field f = interfaceService.getField("SERVICE_NAME");
            String name = (String) f.get(null);
            result = serviceEnabled(name);
        } catch (Exception eee) {
            if (log.isWarnEnabled()) {
                log.warn(_("topia.persistence.warn.service.not.found", interfaceService, eee.getMessage()), eee);
//                log.warn("Can't get service name for: " + interfaceService, eee);
            }
        }
        return result;
    }

    /**
     * Take one service, this service must be valid service interface with
     * public static final SERVICE_NAME declaration.
     * @param <E>
     * @throws TopiaNotFoundException
     */
    @Override
    public <E extends TopiaService> E getService(Class<E> interfaceService)
            throws TopiaNotFoundException {
        try {
            Field f = interfaceService.getField("SERVICE_NAME");
            String name = (String) f.get(null);
            E result = (E) getService(name);
            if (result == null) {
                throw new TopiaNotFoundException(_("topia.persistence.error.service.not.found", interfaceService));
//                throw new TopiaNotFoundException("Service not available: " + interfaceService);
            }
            return result;
        } catch (Exception eee) {

            throw new TopiaNotFoundException(_("topia.persistence.error.service.not.retreaved", interfaceService, eee.getMessage()), eee);
//            throw new TopiaNotFoundException("Can't get service: " + interfaceService, eee);
        }
    }

    @Override
    public Collection<TopiaService> getAllServices() {
        Collection<TopiaService> result = getServices().values();
        return result;
    }

    /**
     * Constructeur utilisé par le beginTransaction pour créer le context fils.
     *
     * @param parentContext
     * @throws HibernateException
     * @throws TopiaNotFoundException
     */
    protected TopiaContextImpl(TopiaContextImplementor parentContext)
            throws HibernateException, TopiaNotFoundException {
        this.parentContext = parentContext;
    }

    @Override
    public Set<TopiaContextImplementor> getChildContext() {
        return this.childContext;
    }

    protected void addChildContext(TopiaContextImplementor child) {
        childContext.add(child);
    }

    @Override
    public void removeChildContext(TopiaContextImplementor child) {
        //On ne retire les fils que si ce contexte n'est pas deja ferme. Permet d'eviter les acces concurrentiels
        if (!closed) {
            childContext.remove(child);
        }
    }

    @Override
    public TopiaContextImplementor getParentContext() {
        return parentContext;
    }

    @Override
    public TopiaContextImplementor getRootContext() {
        TopiaContextImplementor result = this;
        if (getParentContext() != null) {
            result = getParentContext().getRootContext();
        }
        return result;
    }

    @Override
    public Properties getConfig() {
        if (config == null && getParentContext() != null) {
            config = getParentContext().getConfig();
        }
        return config;
    }

    @Override
    public void createSchema() throws TopiaException {
        try {
            boolean showSchema = false;
            if (log.isDebugEnabled()) {
                showSchema = true;
            }
            getFiresSupport().firePreCreateSchema(this);
            new SchemaExport(getHibernateConfiguration()).create(showSchema,
                    true);
            getFiresSupport().firePostCreateSchema(this);
        } catch (HibernateException eee) {
            throw new TopiaException(_("topia.persistence.error.create.schema", eee.getMessage()), eee);
//            throw new TopiaException("Can't create database schema", eee);
        }
    }

    @Override
    public void showCreateSchema() throws TopiaException {
        try {
            new SchemaExport(getHibernateConfiguration()).execute(true, false, false, true);
        } catch (HibernateException eee) {
            throw new TopiaException(_("topia.persistence.error.create.schema", eee.getMessage()), eee);
//            throw new TopiaException("Can't create database schema", eee);
        }
    }

    /* (non-Javadoc)
     * @see org.nuiton.topia.TopiaContext#createSchema()
     */
    @Override
    public void updateSchema() throws TopiaException {
        try {
            boolean showSchema = false;
            if (log.isDebugEnabled()) {
                showSchema = true;
            }
            getFiresSupport().firePreUpdateSchema(this);
            new SchemaUpdate(getHibernateConfiguration()).execute(showSchema,
                    true);
            getFiresSupport().firePostUpdateSchema(this);
        } catch (HibernateException eee) {
            throw new TopiaException(_("topia.persistence.error.update.schema", eee.getMessage()), eee);
//            throw new TopiaException("Can't create database schema", eee);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.nuiton.topia.framework.TopiaContextImplementor#getHibernate()
     */
    @Override
    public Session getHibernate() throws TopiaException {
        if (hibernate == null) {
            throw new TopiaException(_("topia.persistence.error.no.hibernate.session"));
//            throw new TopiaException("No hibernate session available, you must start a transaction with beginTransaction");
        }
        return hibernate;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.nuiton.topia.framework.TopiaContextImplementor#getHibernateFactory()
     */
    @Override
    public SessionFactory getHibernateFactory() throws TopiaNotFoundException {
        if (hibernateFactory == null) {
            if (getParentContext() != null) {
                hibernateFactory = getParentContext().getHibernateFactory();
            } else {
                hibernateFactory = getHibernateConfiguration().buildSessionFactory();
            }
        }
        return hibernateFactory;
    }

    @Override
    public Configuration getHibernateConfiguration()
            throws TopiaNotFoundException {
        if (hibernateConfiguration == null) {
            if (getParentContext() != null) {
                hibernateConfiguration = getParentContext().getHibernateConfiguration();
            } else {
                hibernateConfiguration = new Configuration();

                // Ajout du listeners pour les events
                TopiaFiresSupport.TopiaHibernateEvent listener = new TopiaFiresSupport.TopiaHibernateEvent(
                        this);

                PreInsertEventListener[] preInsertEventListeners = hibernateConfiguration.getEventListeners().getPreInsertEventListeners();
                preInsertEventListeners = ArrayUtil.concatElems(
                        preInsertEventListeners, listener);
                PreLoadEventListener[] preLoadEventListeners = hibernateConfiguration.getEventListeners().getPreLoadEventListeners();
                preLoadEventListeners = ArrayUtil.concatElems(
                        preLoadEventListeners, listener);
                PreUpdateEventListener[] preUpdateEventListeners = hibernateConfiguration.getEventListeners().getPreUpdateEventListeners();
                preUpdateEventListeners = ArrayUtil.concatElems(
                        preUpdateEventListeners, listener);
                PreDeleteEventListener[] preDeleteEventListeners = hibernateConfiguration.getEventListeners().getPreDeleteEventListeners();
                preDeleteEventListeners = ArrayUtil.concatElems(
                        preDeleteEventListeners, listener);

                PostInsertEventListener[] postInsertEventListeners = hibernateConfiguration.getEventListeners().getPostInsertEventListeners();
                postInsertEventListeners = ArrayUtil.concatElems(
                        postInsertEventListeners, listener);
                PostLoadEventListener[] postLoadEventListeners = hibernateConfiguration.getEventListeners().getPostLoadEventListeners();
                postLoadEventListeners = ArrayUtil.concatElems(
                        postLoadEventListeners, listener);
                PostUpdateEventListener[] postUpdateEventListeners = hibernateConfiguration.getEventListeners().getPostUpdateEventListeners();
                postUpdateEventListeners = ArrayUtil.concatElems(
                        postUpdateEventListeners, listener);
                PostDeleteEventListener[] postDeleteEventListeners = hibernateConfiguration.getEventListeners().getPostDeleteEventListeners();
                postDeleteEventListeners = ArrayUtil.concatElems(
                        postDeleteEventListeners, listener);

                hibernateConfiguration.getEventListeners().setPreInsertEventListeners(preInsertEventListeners);
                hibernateConfiguration.getEventListeners().setPreLoadEventListeners(preLoadEventListeners);
                hibernateConfiguration.getEventListeners().setPreUpdateEventListeners(preUpdateEventListeners);
                hibernateConfiguration.getEventListeners().setPreDeleteEventListeners(preDeleteEventListeners);

                hibernateConfiguration.getEventListeners().setPostInsertEventListeners(postInsertEventListeners);
                hibernateConfiguration.getEventListeners().setPostLoadEventListeners(postLoadEventListeners);
                hibernateConfiguration.getEventListeners().setPostUpdateEventListeners(postUpdateEventListeners);
                hibernateConfiguration.getEventListeners().setPostDeleteEventListeners(postDeleteEventListeners);

                // ajout des repertoires contenant les mappings hibernate
                String[] dirs = getConfig().getProperty(
                        TOPIA_PERSISTENCE_DIRECTORIES, "").split(",");
                for (String dir : dirs) {
                    dir = dir.trim();
                    if (!"".equals(dir)) {
                        hibernateConfiguration.addDirectory(new File(dir));
                    }
                }

                // ajout des classes dites persistentes
                String listPersistenceClasses = getConfig().getProperty(
                        TOPIA_PERSISTENCE_CLASSES, "");

                for (TopiaService service : getAllServices()) {
                    Class<?>[] classes = service.getPersistenceClasses();

                    // certains service n'ont pas de classe persistantes
                    if (classes != null) {
                        for (Class<?> clazz : classes) {
                            hibernateConfiguration.addClass(clazz);
                        }
                    }
                }

                String[] classes = listPersistenceClasses.split(",");
                for (String classname : classes) {
                    classname = classname.trim();
                    if (!"".equals(classname)) {
                        Class<?> clazz;
                        try {
                            clazz = Class.forName(classname);
                        } catch (ClassNotFoundException eee) {
                            throw new TopiaNotFoundException(_("topia.persistence.error.class.not.found", classname));
//                            throw new TopiaNotFoundException("Persistent class " + classname + " not found");
                        }
                        persistenceClasses.add(clazz);
                        hibernateConfiguration.addClass(clazz);
                    }
                }

                Properties prop = new Properties();
                prop.putAll(hibernateConfiguration.getProperties());
                prop.putAll(getConfig());
                prop.putAll(TopiaUtil.getProperties(getConfig().getProperty(
                        TOPIA_PERSISTENCE_PROPERTIES_FILE)));
                hibernateConfiguration.setProperties(prop);
            }
        }
        return hibernateConfiguration;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.nuiton.topia.framework.TopiaContextImplementor#getDAO(java.lang.Class)
     */
    @Override
    public <E extends TopiaEntity> TopiaDAO<E> getDAO(Class<E> entityClass)
            throws TopiaException {
        if (entityClass == null) {
            throw new IllegalArgumentException(_("topia.persistence.error.null.parameter", "getDAO", "entityClass"));
//            throw new IllegalArgumentException("null is not valid entity class");
        }
        if (getRootContext() == this) {
            throw new TopiaException(_("topia.persistence.error.rootContext.access"));
//            throw new TopiaException("Vous êtes sur le root context vous" + " devez ouvrir une transaction pour pouvoir accèder" + " aux données");
        }
        if (getHibernateFactory().getClassMetadata(entityClass) == null && getHibernateFactory().getClassMetadata(
                entityClass.getName() + "Impl") == null && getHibernateFactory().getClassMetadata(
                entityClass.getName() + "Abstract") == null) {
            log.info(_("topia.persistence.supported.classes.for.context", getHibernateFactory().getAllClassMetadata().keySet()));
//            log.info("Classes supportées par ce TopiaContext: " + getHibernateFactory().getAllClassMetadata().keySet());
            throw new TopiaException(_("topia.persistence.error.unsupported.class", entityClass.getName()));
//            throw new TopiaException("La classe " + entityClass.getName() + " n'est pas supportée par ce TopiaContext. Vous avez sans" + " doute oublié d'ajouter son mapping");
        }

        TopiaDAO<E> result = (TopiaDAO<E>) daoCache.get(entityClass);
        if (result == null) {

            // looking for specialized DAO
            // normalement il en existe un car il est généré automatiquement
            // si on utilise la génération
            String daoClassname = entityClass.getName() + "DAO";
            try {
                Class<TopiaDAO<E>> daoClass = (Class<TopiaDAO<E>>) Class.forName(daoClassname);
                TopiaDAO<E> spe = daoClass.newInstance();
                result = spe;
            } catch (Exception eee) {
                log.warn("specialized DAO " + daoClassname + " not found, use default TopiaDAOHibernate");
                result = new TopiaDAOImpl<E>();
            }

            result.init(this, entityClass);
            daoCache.put(entityClass, result);
        }
        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.nuiton.topia.TopiaContext#beginTransaction()
     */
    @Override
    public TopiaContext beginTransaction() throws TopiaException {
        checkClosed(_("topia.persistence.error.context.is.closed"));
//        checkClosed("Ce contexte a ete ferme, impossible de commencer une transaction");
        TopiaContextImpl result = new TopiaContextImpl(this);

        SessionFactory factory = getHibernateFactory();
        result.hibernate = factory.openSession();
        // new TopiaInterceptor(result));
        // on ne synchronise jamais les données avec la base tant que
        // l'utilisateur n'a pas fait de commit du context
        result.hibernate.setFlushMode(FlushMode.MANUAL);

        // 20060926 poussin ajouter pour voir si ca regle les problemes de deadlock h2
        // Conclusion, il faut bien ouvrir une transaction maintenant, sinon 
        // lorsque l'on fait des acces a la base, une transaction par defaut est
        // utilisé mais elle n'est jamais vraiment fermé ce qui pose des problemes
        // de lock sur les tables.
        result.hibernate.beginTransaction();

        // 20081217 : add child AFTER hibernate session is opened
        addChildContext(result);

        // fire event
        getFiresSupport().fireOnBeginTransaction(result);

        return result;
    }

    //    Transaction tx = null;

    /*
     * (non-Javadoc)
     * 
     * @see org.nuiton.topia.TopiaContext#commitTransaction()
     */
    @Override
    public void commitTransaction() throws TopiaException {
        if (getRootContext() == this) {
            throw new TopiaException(_("topia.persistence.error.unsupported.operation.on.root.context", "commit"));
//            throw new TopiaException("Vous êtes sur le root context le commit est impossible");
        }
        checkClosed(_("topia.persistence.error.unsupported.operation.on.closed.context", "commit"));
//        checkClosed("Ce contexte a ete ferme, impossible de faire un commit");
        try {
            for (TopiaDAO<? extends TopiaEntity> dao : daoCache.values()) {
                dao.commitTransaction();
            }
            Transaction tx = hibernate.getTransaction();
            //            Transaction tx = hibernate.beginTransaction();
            hibernate.flush();
            tx.commit();

            getFiresSupport().fireOnPostCommit(this);
            TopiaContextImplementor parent = getParentContext();
            if (parent != null) {
                parent.getFiresSupport().fireOnPostCommit(this);
            }

            hibernate.beginTransaction();
            // it's seem necessary to change session after commit
            // NON, NON, NON, il ne faut surtout pas le faire, ca pose plein de probleme 
            //            hibernate = getHibernateFactory().openSession();
            //            hibernate.setFlushMode(FlushMode.NEVER);
        } catch (Exception eee) {
            throw new TopiaException(_("topia.persistence.error.on.commit", eee.getMessage()), eee);
//            throw new TopiaException("Error during commit", eee);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.nuiton.topia.TopiaContext#rollbackTransaction()
     */
    @Override
    public void rollbackTransaction() throws TopiaException {
        if (getRootContext() == this) {
            throw new TopiaException(_("topia.persistence.error.unsupported.operation.on.root.context", "rollback"));
//            throw new TopiaException("Vous êtes sur le root context le rollback est impossible");
        }
        checkClosed(_("topia.persistence.error.unsupported.operation.on.closed.context", "rollback"));
//        checkClosed("Ce contexte a ete ferme, impossible de faire un rollback");
        try {
            for (TopiaDAO<? extends TopiaEntity> dao : daoCache.values()) {
                dao.rollbackTransaction();
            }
            Transaction tx = hibernate.getTransaction();
            //            Transaction tx = hibernate.beginTransaction();
            hibernate.clear();
            tx.rollback();
            hibernate.close();
            // it's very important to change the session after rollback
            // otherwize there are many error during next Entity's modification
            hibernate = getHibernateFactory().openSession();
            hibernate.setFlushMode(FlushMode.MANUAL);

            hibernate.beginTransaction();

            getFiresSupport().fireOnPostRollback(this);
            TopiaContextImplementor parent = getParentContext();
            if (parent != null) {
                parent.getFiresSupport().fireOnPostRollback(this);
            }

        } catch (HibernateException eee) {
            throw new TopiaException(_("topia.persistence.error.on.rollback", eee.getMessage()), eee);
//            throw new TopiaException(eee);
        }
    }

    @Override
    public void closeContext() throws TopiaException {
        // 20060926 poussin: Si si on peut, ca ferme en fait tous les enfants et c tout
        //        if (getRootContext() == this) {
        //            throw new TopiaException(
        //                    "Vous ne pouvez pas fermer le root context");
        //        }
        checkClosed(_("topia.persistence.error.context.already.closed"));
//        checkClosed("Ce contexte a deja ete ferme");

        // suppression des contexts fils
        TopiaContextImplementor[] childs = childContext.toArray(new TopiaContextImplementor[childContext.size()]);
        for (TopiaContextImplementor child : childs) {
            if (!child.isClosed()) {
                child.closeContext();
            }
        }
        //        for (Iterator iter = childContext.iterator(); iter.hasNext();) {
        //            TopiaContextImplementor child = (TopiaContextImplementor) iter.next();
        //            child.closeContext();
        //            iter.remove();
        //        }

        // on se desenregistre du context pere et on ferme les connexions si 
        // on est pas le root context
        if (getRootContext() != this) {
            this.closed = true;
            hibernate.close();
            getParentContext().removeChildContext(this);
        } else {
            if (hibernateFactory != null) {
                hibernateFactory.close();
                this.closed = true;
                TopiaContextFactory.removeContext(this);
                log.debug("TopiaContext finalized");
            }
        }
    }

    /**
     * Pour le context root on ferme tous les fils, et la factory hibernate
     * @see java.lang.Object#finalize()
     */
    @Override
    protected void finalize() throws Throwable {
        if (hibernateFactory != null) {
            closeContext();
            hibernateFactory.close();
            this.closed = true;
            log.debug("TopiaContext finalized");
        }
    }

    @Override
    public boolean isClosed() {
        return closed;
    }

    protected void checkClosed(String message) throws TopiaException {
        if (closed) {
            throw new TopiaException(message);
        }
    }

    /* (non-Javadoc)
     * @see org.nuiton.topia.TopiaContext#findByTopiaId(java.lang.String)
     */
    @Override
    public TopiaEntity findByTopiaId(String topiaId) throws TopiaException {
        checkClosed(_("topia.persistence.error.unsupported.operation.on.closed.context", "findByTopiaId"));
//        checkClosed("Ce contexte a ete ferme, impossible de faire une recherche");
        TopiaEntity result;
        Class<TopiaEntity> entityClass = TopiaId.getClassName(topiaId);
        TopiaDAO<TopiaEntity> dao = getDAO(entityClass);
        result = dao.findByTopiaId(topiaId);
        return result;
    }

    /*
     * (non-Javadoc)
     * @see org.nuiton.topia.TopiaContext#find(java.lang.String, java.lang.Object[])
     */
    @Override
    public List find(String hql, Object... args) throws TopiaException {
        checkClosed(_("topia.persistence.error.unsupported.operation.on.closed.context", "find"));
//        checkClosed("Ce contexte a ete ferme, impossible de faire une recherche");
        try {
            Query query = getHibernate().createQuery(hql);
            for (int j = 0; j < args.length; j += 2) {
                String name = (String) args[j];
                Object value = args[j + 1];
                if (value.getClass().isArray()) {
                    query.setParameterList(name, (Object[]) value);
                } else {
                    query.setParameter(name, value);
                }
            }
            List result = query.list();
            result = firesSupport.fireEntitiesLoad(this, result);
            return result;
        } catch (HibernateException eee) {
            throw new TopiaException(_("topia.persistence.error.on.query", hql, eee.getMessage()), eee);
//            throw new TopiaException("Error during query execution: " + hql,eee);
        }
    }

    @Override
    public List<?> find(String hql, int startIndex, int endIndex, Object... args)
            throws TopiaException {
        checkClosed(_("topia.persistence.error.unsupported.operation.on.closed.context", "find"));
//        checkClosed("Ce contexte a ete ferme, impossible de faire une recherche");
        try {
            Query query = getHibernate().createQuery(hql);
            for (int j = 0; j < args.length; j += 2) {
                String name = (String) args[j];
                Object value = args[j + 1];
                if (value.getClass().isArray()) {
                    query.setParameterList(name, (Object[]) value);
                } else {
                    query.setParameter(name, value);
                }
            }
            query.setFirstResult(startIndex);
            query.setMaxResults(endIndex - startIndex + 1);
            List result = query.list();
            result = firesSupport.fireEntitiesLoad(this,result);
            return result;
        } catch (HibernateException eee) {
            throw new TopiaException(_("topia.persistence.error.on.query", hql, eee.getMessage()), eee);
//            throw new TopiaException("Error during query execution: " + hql,eee);
        }
    }

    /**
     * Execute HQL operation on data (Update, Delete)
     * @param hql
     * @param args
     * @return The number of entities updated or deleted.
     * @throws TopiaException
     */
    @Override
    public int execute(String hql, Object... args) throws TopiaException {
        checkClosed(_("topia.persistence.error.unsupported.operation.on.closed.context", "find"));
//        checkClosed("Ce contexte a ete ferme, impossible de faire une recherche");
        try {
            Query query = getHibernate().createQuery(hql);
            for (int j = 0; j < args.length; j += 2) {
                query.setParameter((String) args[j], args[j + 1]);
            }
            int result = query.executeUpdate();
            return result;
        } catch (HibernateException eee) {
            throw new TopiaException(_("topia.persistence.error.on.query", hql, eee.getMessage()), eee);
//            throw new TopiaException("Error during query execution: " + hql,eee);
        }
    }

    /* (non-Javadoc)
     * @see org.nuiton.topia.TopiaContext#add(org.nuiton.topia.persistence.TopiaEntity)
     */
    @Override
    public void add(TopiaEntity e) throws TopiaException {
        checkClosed(_("topia.persistence.error.unsupported.operation.on.closed.context", "add"));
//        checkClosed("Ce contexte a ete ferme, impossible d'ajouter une entite");
        String id = e.getTopiaId();
        Class<TopiaEntity> entityClass = TopiaId.getClassName(id);
        TopiaDAO<TopiaEntity> dao = getDAO(entityClass);
        dao.update(e);
    }

    @Override
    public void importXML(Reader xml) throws TopiaException {
        checkClosed(_("topia.persistence.error.unsupported.operation.on.closed.context", "importXML"));
//        checkClosed("Ce contexte a ete ferme, impossible d'effectuer l'import");
        Document doc;

        SAXReader xmlReader = new SAXReader();
        try {
            doc = xmlReader.read(xml);
            if (log.isDebugEnabled()) {
                log.debug("Lecture du document terminee");
            }
        } catch (DocumentException de) {
            throw new TopiaException(_("topia.persistence.error.on.loding.xml.doc", de.getMessage()), de);
//            throw new TopiaException("Lecture du document impossible", de);
        }

        if (doc != null) {
            Session sessionDom4j = getHibernate().getSession(EntityMode.DOM4J);
            Element rootElement = doc.getRootElement();
            Iterator<?> it = rootElement.elementIterator();
            while (it.hasNext()) {
                Element entity = (Element) it.next();
                try {
                    sessionDom4j.replicate(entity, ReplicationMode.EXCEPTION);
                } catch (HibernateException he) {
                    log.warn(_("topia.persistence.error.replicate.entity", entity, he.getMessage()), he);
//                    log.warn("Echec de replication sur " + entity, he);
                }
            }
            // must commit data, otherwise : no effects...
            sessionDom4j.flush();
        } else {
            throw new TopiaException(_("topia.persistence.error.empty.doc"));
//            throw new TopiaException("Document vide");
        }
    }

    @Override
    public void exportXML(Writer xml, Object... entityAndcondition)
            throws TopiaException {
        checkClosed(_("topia.persistence.error.unsupported.operation.on.closed.context", "exportXML"));
//        checkClosed("Ce contexte a ete ferme, impossible d'effectuer l'export");

        String[] queries = buildQueries(entityAndcondition);

        // performs queries
        try {
            Session sessionDom4j = getHibernate().getSession(EntityMode.DOM4J);

            Document doc = DocumentFactory.getInstance().createDocument();
            Element rootElement = doc.addElement("topiaExport");
            DateFormat format = new SimpleDateFormat("dd/MM/yyyy HH:mm:ss");
            String date = format.format(new Date(System.currentTimeMillis()));
            rootElement.addAttribute("date", date);

            for (String query : queries) {
                List<?> list = sessionDom4j.createQuery(query).list();
                for (Object o : list) {
                    rootElement.add((Element) o);
                }
            }

            /*for (int i = 0; i < entityAndcondition.length;) {
            try {
            entityClass = (Class) entityAndcondition[i++];
            condition = (String) entityAndcondition[i++];

            String query = "from " + entityClass.getName();
            if (condition != null && !condition.isEmpty()) {
            query += " where " + condition;
            }

            for (Object o : sessionDom4j.createQuery(query).list()) {
            rootElement.add((Element) o);
            }
            } catch (ClassCastException eee) {
            if (i % 2 == 0) {
            throw new IllegalArgumentException(
            "Others arguement must be String not "
            + entityAndcondition[i - 1], eee);
            } else {
            throw new IllegalArgumentException(
            "Others arguement must be Class not "
            + entityAndcondition[i - 1], eee);
            }
            } catch (IndexOutOfBoundsException eee) {
            throw new IllegalArgumentException(
            "Others arguement must be couple of (Class, String)",
            eee);
            }
            }*/

            XMLWriter result = new XMLWriter(xml, OutputFormat.createPrettyPrint());
            result.write(doc);
            result.close();

        } catch (HibernateException eee) {
            throw new TopiaException(_("topia.persistence.error.on.export", eee.getMessage()), eee);
//            throw new TopiaException("Can't export XML", eee);
        } catch (IOException eee) {
            throw new TopiaException(_("topia.persistence.error.on.export", eee.getMessage()), eee);
//            throw new TopiaException("Can't export XML", eee);
        }
    }

    @Override
    public void replicate(TopiaContext dstCtxt, Object... entityAndCondition) throws TopiaException, IllegalArgumentException {
        checkClosed(_("topia.persistence.error.unsupported.operation.on.closed.context", "replicate"));
//        checkClosed("Ce contexte a ete ferme, impossible d'effectuer l'export");
        TopiaContextImpl dstContextImpl = (TopiaContextImpl) dstCtxt;
        dstContextImpl.checkClosed(_("topia.persistence.error.unsupported.operation.on.closed.context", "replicate"));
//        dstContextImpl.checkClosed("Ce contexte a ete ferme, impossible d'effectuer l'export");
        if (getRootContext().equals(dstContextImpl.getRootContext())) {
            throw new IllegalArgumentException(_("topia.persistence.error.replicate.on.same.context"));
//            throw new IllegalArgumentException("Impossible de dupliquer dans la même base");
        }

        String[] queries = buildQueries(entityAndCondition);
        try {
            for (String query : queries) {
                if (log.isDebugEnabled()) {
                    log.debug("acquire entities " + query);
                }
                // acquire data to replicate
                List<?> entities = find(query);
                replicate0(dstContextImpl, entities.toArray());
                if (log.isDebugEnabled()) {
                    log.debug("replication of entities " + query + " was sucessfully done.");
                }
            }
        } catch (HibernateException e) {
            throw new TopiaException(_("topia.persistence.error.on.replicate", e.getMessage()), e);
//            throw new TopiaException("Could not duplicate data for reason " + e.getMessage(), e);
        }
    }

    @Override
    public <T extends TopiaEntity> void replicateEntity(TopiaContext dstCtxt, T entity) throws TopiaException, IllegalArgumentException {
        checkClosed(_("topia.persistence.error.unsupported.operation.on.closed.context", "replicateEntity"));
//        checkClosed("Ce contexte a ete ferme, impossible d'effectuer l'export");
        TopiaContextImpl dstContextImpl = (TopiaContextImpl) dstCtxt;
        dstContextImpl.checkClosed(_("topia.persistence.error.unsupported.operation.on.closed.context", "replicateEntity"));
//        dstContextImpl.checkClosed("Ce contexte a ete ferme, impossible d'effectuer l'export");
        if (getRootContext().equals(dstContextImpl.getRootContext())) {
            throw new IllegalArgumentException(_("topia.persistence.error.replicate.on.same.context"));
//            throw new IllegalArgumentException("Impossible de dupliquer dans la même base");
        }
        replicate0(dstContextImpl, entity);
    }

    @Override
    public <T extends TopiaEntity> void replicateEntities(TopiaContext dstCtxt, List<T> entities) throws TopiaException, IllegalArgumentException {
        checkClosed(_("topia.persistence.error.unsupported.operation.on.closed.context", "replicateEntities"));
//        checkClosed("Ce contexte a ete ferme, impossible d'effectuer l'export");
        TopiaContextImpl dstContextImpl = (TopiaContextImpl) dstCtxt;
        dstContextImpl.checkClosed(_("topia.persistence.error.unsupported.operation.on.closed.context", "replicateEntities"));
//        dstContextImpl.checkClosed("Ce contexte a ete ferme, impossible d'effectuer l'export");
        if (getRootContext().equals(dstContextImpl.getRootContext())) {
            throw new IllegalArgumentException(_("topia.persistence.error.replicate.on.same.context"));
//            throw new IllegalArgumentException("Impossible de dupliquer dans la même base");
        }
        replicate0(dstContextImpl, entities.toArray());
    }

    @Override
    public TopiaFiresSupport getFiresSupport() {
        return firesSupport;
    }

    /**
     * Backup database in gzip compressed file
     * Only work for h2 database
     *
     * @param file file to write backup
     * @param compress if true then use gzip to compress file
     *
     * @see TopiaContext#backup(java.io.File,boolean)
     */
    @Override
    public void backup(File file, boolean compress) throws TopiaException {
        checkClosed(_("topia.persistence.error.unsupported.operation.on.closed.context", "backup"));
//        checkClosed("Ce contexte a ete ferme, impossible d'effectuer le backup");
        try {
            //            Statement stat = getHibernate().connection().createStatement();
            //            ResultSet rs = stat.executeQuery("SCRIPT TO '" + file.getAbsolutePath() + "'");

            // Bug dans h2 v0.9, on ne peut pas directement passer le fichier dans le SQL
            // Il y a un ArrayOutBoundException -> org.h2.command.dml.Script.add:203
            // pour certaines lignes. C dommage, car on est obligé de rammener
            // tout en texte, ce qui peut-etre gros pour la memoire :(

            String options = "";
            if (compress) {
                options += " COMPRESSION GZIP";
            }

            SQLQuery query = getHibernate().createSQLQuery(
                    "SCRIPT TO '" + file.getAbsolutePath() + "'" + options);
            query.list();

            //            List<String> lines = query.list();

            //            // en fait on est un peu obligé d'exporter toute la base 
            //            // (creation du schema compris) car sinon lors de la restauration
            //            // si le schema a ete creer avant il contient aussi les 
            //            // contrainte et du coup les inserts se passent mal :(
            //            // Si on ne voulait que les inserts, lors de la resauration il 
            //            // faudrait desactiver les contraintes et les reactiver ensuite
            //            
            //            OutputStream os = new BufferedOutputStream (
            //                    new FileOutputStream(file));
            //            
            //            PrintStream out;
            //            if (compress) {
            //                out = new PrintStream(new GZIPOutputStream(os));
            //            } else {
            //                out = new PrintStream(os);
            //            }
            //                
            //            for (String line : lines) {
            //                out.println(line + ";");                
            //            }
            //            out.close();
        } catch (Exception eee) {
            throw new TopiaException(_("topia.persistence.error.on.backup", eee.getMessage()), eee);
//            throw new TopiaException("Can't backup", eee);
        }
    }

    /**
     * Read database from gzip compressed file
     *
     * Only work for h2 database
     *
     * @see org.nuiton.topia.TopiaContext#restore(java.io.File)
     */
    @Override
    public void restore(File file) throws TopiaException {
        // send event
        getFiresSupport().firePreRestoreSchema(this);
        checkClosed(_("topia.persistence.error.unsupported.operation.on.closed.context", "restore"));
//        checkClosed("Ce contexte a ete ferme, impossible d'effectuer le restore");
        String sql = null;
        try {
            // decompresse file in temporary file
            InputStream in = new BufferedInputStream(new FileInputStream(file));
            in.mark(2);

            // read header to see if is compressed file
            int b = in.read();
            // redundant cast : int magic = ((int) in.read() << 8) | b;
            int magic = (in.read() << 8) | b;
            in.reset();

            String options = "";

            if (magic == GZIPInputStream.GZIP_MAGIC) {
                //                in = new GZIPInputStream(in);
                options += " COMPRESSION GZIP";
            }
            in.close();

            SQLQuery query = getHibernate().createSQLQuery(
                    "RUNSCRIPT FROM '" + file.getAbsolutePath() + "'" + options);
            query.executeUpdate();

            // send event AFTER restore
            getFiresSupport().firePostRestoreSchema(this);

            //            BufferedReader reader = new BufferedReader(new InputStreamReader(in));
            //            Connection conn = getHibernate().connection();
            //            conn.setAutoCommit(false);
            //            Statement stat = conn.createStatement();
            //            
            //            while((sql = reader.readLine()) != null) {
            //                // FIXME remove next line when h2 will do the bug correction (normaly in october 2006)
            //                // this permit database restauration
            ////                if (sql.startsWith("ALTER TABLE")) {
            ////                    sql = sql.replaceAll("INDEX CONSTRAINT_INDEX_\\d*", "");
            ////                }
            //                stat.execute(sql);
            //            }
            //            stat.close();
            //            conn.commit();
            //            conn.close();

            //            // by default we supposed that file is not compressed
            //            File tmp = file;
            //            if (magic == GZIPInputStream.GZIP_MAGIC) {
            //                // in fact file is compressed, use temporaly file
            //                tmp = File.createTempFile("tmp-topia-restore", ".sql");
            //                tmp.deleteOnExit();
            //
            //                // decompresse file in temporary file
            //                InputStream gzin = new GZIPInputStream(in);
            //            
            //                OutputStream out = new BufferedOutputStream(
            //                        new FileOutputStream(tmp));
            //            
            //                byte [] buffer = new byte[64 * 1024];
            //                int len = 0;
            //                while ((len = gzin.read(buffer)) != -1) {
            //                    out.write(buffer, 0, len);
            //                }
            //                out.close();
            //                gzin.close();
            //            }
            //            
            //            // restore data
            //            Statement stat = getHibernate().connection().createStatement();
            //            stat.execute("RUNSCRIPT FROM '" + tmp.getAbsolutePath() + "'");
            //
            ////            SQLQuery query = getHibernate().createSQLQuery("RUNSCRIPT FROM '" + file.getAbsolutePath() + "'");
            ////            List result = query.list();
        } catch (Exception eee) {
            throw new TopiaException(_("topia.persistence.error.on.restore", sql, eee.getMessage()), eee);
//            throw new TopiaException("Can't restore. Last sql instruction was:" + sql, eee);
        }
    }

    /**
     * Only h2 supported for now
     * @see org.nuiton.topia.TopiaContext#clear(boolean)
     */
    @Override
    public void clear(boolean dropDatabase) throws TopiaException {
        try {
            TopiaContextImpl root = (TopiaContextImpl) getRootContext();
            TopiaContextImpl tx = (TopiaContextImpl) root.beginTransaction();

            String sql = "DROP ALL OBJECTS";
            if (dropDatabase) {
                sql += " DELETE FILES";
            }
            Query query = tx.getHibernate().createSQLQuery(sql);
            query.executeUpdate();
            tx.closeContext();
            root.finalize();
        } catch (Throwable eee) {
            throw new TopiaException(_("topia.persistence.error.on.clear", eee.getMessage()), eee);
//            throw new TopiaException("Can't clear topia context", eee);
        }
    }

    @Override
    public List<Class<?>> getPersistenceClasses() {
        return persistenceClasses;
    }

    @Override
    public boolean isSchemaExist(Class<? extends TopiaEntity> clazz) throws TopiaException {
        checkClosed(_("topia.persistence.error.unsupported.operation.on.closed.context", "replicateEntity"));
        boolean result = TopiaUtil.isSchemaExist(hibernateConfiguration, clazz.getName());
        return result;
    }

    /* Adders */
    @Override
    public void addTopiaEntityListener(TopiaEntityListener listener) {
        getFiresSupport().addTopiaEntityListener(listener);
    }

    @Override
    public void addTopiaEntityListener(
            Class<? extends TopiaEntity> entityClass,
            TopiaEntityListener listener) {
        getFiresSupport().addTopiaEntityListener(entityClass, listener);
    }

    @Override
    public void addTopiaEntityVetoable(TopiaEntityVetoable vetoable) {
        getFiresSupport().addTopiaEntityVetoable(TopiaEntity.class, vetoable);
    }

    @Override
    public void addTopiaEntityVetoable(
            Class<? extends TopiaEntity> entityClass,
            TopiaEntityVetoable vetoable) {
        getFiresSupport().addTopiaEntityVetoable(entityClass, vetoable);
    }

    @Override
    public void addTopiaTransactionListener(TopiaTransactionListener listener) {
        getFiresSupport().addTopiaTransactionListener(listener);
    }

    @Override
    public void addTopiaTransactionVetoable(TopiaTransactionVetoable vetoable) {
        getFiresSupport().addTopiaTransactionVetoable(vetoable);
    }

    @Override
    public void addPropertyChangeListener(PropertyChangeListener listener) {
        getFiresSupport().addPropertyChangeListener(listener);
    }

    @Override
    public void addTopiaContextListener(TopiaContextListener listener) {
        getFiresSupport().addTopiaContextListener(listener);
    }

    /* Removers */
    @Override
    public void removeTopiaEntityListener(TopiaEntityListener listener) {
        getFiresSupport().removeTopiaEntityListener(TopiaEntity.class, listener);
    }

    @Override
    public void removeTopiaEntityListener(
            Class<? extends TopiaEntity> entityClass,
            TopiaEntityListener listener) {
        getFiresSupport().removeTopiaEntityListener(entityClass, listener);
    }

    @Override
    public void removeTopiaEntityVetoable(TopiaEntityVetoable vetoable) {
        getFiresSupport().removeTopiaEntityVetoable(TopiaEntity.class, vetoable);
    }

    @Override
    public void removeTopiaEntityVetoable(
            Class<? extends TopiaEntity> entityClass,
            TopiaEntityVetoable vetoable) {
        getFiresSupport().removeTopiaEntityVetoable(entityClass, vetoable);
    }

    @Override
    public void removeTopiaTransactionListener(TopiaTransactionListener listener) {
        getFiresSupport().removeTopiaTransactionListener(listener);
    }

    @Override
    public void removeTopiaTransactionVetoable(TopiaTransactionVetoable vetoable) {
        getFiresSupport().removeTopiaTransactionVetoable(vetoable);
    }

    @Override
    public void removePropertyChangeListener(PropertyChangeListener listener) {
        getFiresSupport().removePropertyChangeListener(listener);
    }

    @Override
    public void removeTopiaContextListener(TopiaContextListener listener) {
        getFiresSupport().removeTopiaContextListener(listener);
    }

    @Override
    public void addTopiaEntitiesVetoable(TopiaEntitiesVetoable vetoable) {
        getFiresSupport().addTopiaEntitiesVetoable(vetoable);
    }

    @Override
    public void removeTopiaEntitiesVetoable(TopiaEntitiesVetoable vetoable) {
        getFiresSupport().removeTopiaEntitiesVetoable(vetoable);
    }

    /**
     * Build the list of queries from the given parameter <code>entityAndCondition>/code>.
     *
     * If no parameter is given, then build the queries for all entities is db, with no condition.
     *
     * @param entityAndCondition the list of tuples (Class,String)
     * @return the list of queries.
     * @throws TopiaException if any pb of db while getting entities classes.
     * @throws IllegalArgumentException if any pb with the given parameter (mainly ClassCastException).
     */
    protected String[] buildQueries(Object... entityAndCondition) throws TopiaException, IllegalArgumentException {
        Class<?> entityClass;
        String condition;

        // si entityAndcondition est vide alors il faut le remplir
        // avec toutes les entités du mapping (class, null)
        if (entityAndCondition.length == 0) {
            entityAndCondition = new Object[getHibernateFactory().getAllClassMetadata().size() * 2];
            int i = 0;
            for (Object className : getHibernateFactory().getAllClassMetadata().keySet()) {
                try {
                    entityAndCondition[i++] = Class.forName((String) className);
                } catch (ClassNotFoundException e) {
                    // should never happen!
                    throw new TopiaException("class cast exception for entity " + className);
                }
                entityAndCondition[i++] = null;

            }
        }

        // prepare queries to perform beofre opening any transaction
        if (entityAndCondition.length % 2 != 0) {
            throw new IllegalArgumentException("entityAndCondition must be a couple of (Class, String)");
        }
        String queries[] = new String[entityAndCondition.length / 2];
        for (int i = 0; i < entityAndCondition.length;) {
            try {
                entityClass = (Class<?>) entityAndCondition[i++];
                condition = (String) entityAndCondition[i++];
                String query = "from " + entityClass.getName();
                if (condition != null && !condition.isEmpty()) {
                    query += " where " + condition;
                }
                queries[(i - 1) / 2] = query;
            } catch (ClassCastException e) {
                if (i % 2 == 0) {
                    throw new IllegalArgumentException(
                            "Others arguement must be String not " + entityAndCondition[i - 1], e);
                } else {
                    throw new IllegalArgumentException(
                            "Others arguement must be Class not " + entityAndCondition[i - 1], e);
                }
            }
        }
        return queries;
    }

    protected void replicate0(TopiaContextImpl dstContextImpl, Object... entities) throws TopiaException {
        try {
            for (Object entity : entities) {
                // dettach entity to source session, to make possible copy of collection
                // without a hibernate exception (list opened in two session...)
                getHibernate().evict(entity);
                dstContextImpl.getHibernate().replicate(entity, ReplicationMode.EXCEPTION);
            }

        } catch (HibernateException eee) {
            throw new TopiaException(_("topia.persistence.error.on.replicate", eee.getMessage()), eee);
//            throw new TopiaException("Could not duplicate data for reason " + e.getMessage(), e);
        }
    }
} //TopiaContextImpl

