/**
 * *##% Plugin maven de changement de license
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*
 */
package org.nuiton.license.plugin;

import org.apache.maven.plugin.MojoExecutionException;


import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import org.apache.maven.project.MavenProject;
import org.nuiton.AbstractPlugin;
import org.nuiton.license.License;
import org.nuiton.license.LicenseFactory;
import org.nuiton.processor.LicenseProcessor;
import org.nuiton.util.PluginHelper;

/**
 * Le modele de goal a utiliser pour mettre a jour l'entete d'un fichier.
 *
 * Une implantation pour les fichiers java est disponible : {@link Jav
 *
 * @author chemit
 * @requiresProject true
 * 
 * @since 1.0.0
 */
public abstract class AbstractUpdateHeaderPlugin extends AbstractPlugin {

    /**
     * Dependance du projet.
     *
     * @parameter default-value="${project}"
     * @required
     * @since 1.0.0
     */
    protected MavenProject project;
    /**
     * Encoding a utiliser pour lire et ecrire les fichiers.
     *
     * @parameter expression="${license.encoding}" default-value="${project.build.sourceEncoding}"
     * @required
     * @since 1.0.0
     */
    protected String encoding;
    /**
     * Un flag pour forcer la generation.
     *
     * @parameter expression="${license.force}"  default-value="false"
     * @since 1.0.0
     */
    protected boolean force;
    /**
     * l'annee de creation du module (sera place dans le header)
     *
     * @parameter expression="${license.inceptionYear}" default-value="${project.inceptionYear}"
     * @required
     * @since 1.0.0
     */
    protected String inceptionYear;
    /**
     * le nom de l'organisation (sera place dans le header)
     *
     * @parameter expression="${license.organizationName}"  default-value="${project.organization.name}"
     * @required
     * @since 1.0.0
     */
    protected String organizationName;
    /**
     * le nom du projet (sera place dans le header)
     *
     * @parameter expression="${license.projectName}" default-value="${project.name}"
     * @required
     * @since 1.0.0
     */
    protected String projectName;
    /**
     * Le type de license a appliquer.
     *
     * @parameter expression="${license.licenseName}"
     * @required
     * @since 1.0.0
     */
    protected String licenseName;
    /**
     * Repertoires des fichiers sources a traiter.
     *
     * @parameter expression="${license.compileSourceRoots}" default-value="${project.compileSourceRoots}"
     * @required
     * @since 1.0.0
     */
    protected List<String> compileSourceRoots;
    /**
     * Repertoires des fichiers sources de test a traiter.
     *
     * @parameter expression="${license.testCompileSourceRoots}" default-value="${project.testCompileSourceRoots}"
     * @required
     * @since 1.0.0
     */
    protected List<String> testCompileSourceRoots;
    /**
     * Repertoire de sortie des classes (classpath).
     *
     * @parameter expression="${license.outputDirectory}" default-value="${project.build.outputDirectory}"
     * @required
     * @since 1.0.0
     */
    protected File outputDirectory;
    /**
     * Repertoire du build des tests
     *
     * @parameter expression="${license.testOutputDirectory}" default-value="${project.build.testOutputDirectory}"
     * @required
     * @since 1.0.0
     */
    protected File testOutputDirectory;
    /**
     * Un resolver externe
     *
     * @parameter expression="${license.extraResolver}"
     * @since 1.0.0
     */
    protected String[] extraResolver;
    /**
     * Pour ajouter d'autres fichiers a traiter (separer par des virgules).
     *
     * Exemple : <code>**\\/*.css,**\\/*.properties</code>
     *
     * @parameter expression="${license.extraIncludes}"
     * @since 1.0.0
     */
    protected String extraIncludes;
    /**
     * Un flag pour conserver un backup des fichiers modifies.
     *
     * @parameter expression="${license.keepBackup}"  default-value="false"
     * @since 1.0.0
     */
    protected boolean keepBackup;
    /**
     * Un flag pour activer le mode verbeux.
     *
     * @parameter expression="${license.verbose}"  default-value="${maven.verbose}"
     * @since 1.0.0
     */
    protected boolean verbose;
    /** le header a ajouter dans chaque fichier source java */
    protected String licenseHeaderContent;
    /** la liste des chemin relatifs des sources java a traiter pour chaque repertoire contenant des sources */
    protected Map<File, String[]> filesToTreate;
    /** le generateur d'en-tete a utiliser */
    protected HeaderGenerator generator;
    protected long timestamp;

    public AbstractUpdateHeaderPlugin() {
        super("all files are up-to-date.");
    }

    protected abstract Map<File, String[]> getFilesToTreate();

    protected abstract HeaderGenerator newHeaderGenerator();

    public HeaderGenerator getGenerator() {
        return generator;
    }

    @Override
    public boolean ensurePackaging() {
        return "pom".equals(project.getPackaging()) || "site".equals(project.getPackaging());
    }

    @Override
    public boolean init() throws IOException {

        timestamp = System.nanoTime();

        // obtain all java source files to be treated
        filesToTreate = getFilesToTreate();

        boolean doGenerate = !filesToTreate.isEmpty();

        if (doGenerate) {

            LicenseFactory factory = LicenseFactory.newInstance(extraResolver);
            License license = factory.revolv(licenseName);

            // obtain content of license header
            licenseHeaderContent = computeHeader(license);

            generator = newHeaderGenerator();

            if (verbose) {
                getLog().info("config - header to write on files \n" + licenseHeaderContent);
                getLog().info("config - will use the generator " + generator);
            }
        } else {
            getLog().info("no jaxx file to treate.");
        }

        return doGenerate;
    }

    @Override
    protected void doAction() throws Exception {
        // create a licence processor with given header
        LicenseProcessor p = new LicenseProcessor(licenseHeaderContent);

        for (Entry<File, String[]> entry : filesToTreate.entrySet()) {
            File src = entry.getKey();
            for (String javaRelativePath : entry.getValue()) {
                File sourceFile = new File(src, javaRelativePath);
                try {
                    processFile(p, sourceFile);
                } catch (Exception e) {
                    throw new MojoExecutionException("could not treate java source file " + sourceFile + " for reason : " + e.getMessage(), e);
                }
            }
        }
    }

    protected String computeHeader(License license) throws IOException {


        String tmpHeader = license.getHeaderContent(encoding);

        // defined inceptionYear (if year is older than now suffix with a - thisYear)
        Calendar cal = Calendar.getInstance();
        cal.setTime(new Date());
        String thisYear = cal.get(Calendar.YEAR) + "";
        if (!thisYear.equals(inceptionYear)) {
            inceptionYear = inceptionYear + " - " + thisYear;
        }

        // format header with projet informations
        tmpHeader = String.format(tmpHeader, projectName, inceptionYear, organizationName);

        // add " * " before each line
        BufferedReader reader = new BufferedReader(new java.io.StringReader(tmpHeader));
        StringBuilder sb = new StringBuilder();

        String line = reader.readLine();
        sb.append(line).append('\n');
        while ((line = reader.readLine()) != null) {
            line = line.trim();
            if (line.isEmpty()) {
                sb.append(" *\n");
            } else {
                sb.append(" * ").append(line).append("\n");
            }
        }
        tmpHeader = sb.toString();
        return tmpHeader.substring(0, tmpHeader.length() - 1);
    }

    /**
     * @param p          license processor
     * @param file the file to process
     * @throws Exception if IO pb
     */
    protected void processFile(LicenseProcessor p, File file) throws Exception {

        if (verbose) {
            getLog().info("process file " + file);
        }

        // file where to writeFile result
        File processFile = new File(file.getAbsolutePath() + "_" + timestamp);

        try {
            p.process(file, processFile);
            boolean foundLicenseHeader = p.getLicenceFilter().wasTouched();

            if (!foundLicenseHeader) {

                // no license header found in file, add it
                addHeaderToFile(file, processFile);
            }

            if (keepBackup) {
                File backupFile = new File(file.getAbsolutePath() + "~");
                if (verbose) {
                    getLog().debug("backup original file " + file);
                }
                file.renameTo(backupFile);
            }
            processFile.renameTo(file);

        } catch (Exception e) {
            getLog().error("could not process file " + file + " for reason " + e.getMessage(), e);
            processFile.delete();
            throw e;
        } finally {
            p.getLicenceFilter().reset();
        }
    }

    protected void addHeaderToFile(File sourceFile, File processFile) throws IOException {
        getLog().warn("no license header was found on file " + processFile + ", adding one");
        String content = PluginHelper.readAsString(sourceFile, encoding);
        content = getGenerator().getHeader(licenseHeaderContent) + content;
        writeFile(processFile, content, encoding);
    }

    @Override
    public boolean isVerbose() {
        return verbose;
    }

    @Override
    public void setVerbose(boolean verbose) {
        this.verbose = verbose;
    }

    @Override
    public MavenProject getProject() {
        return project;
    }

    @Override
    public void setProject(MavenProject project) {
        this.project = project;
    }

    protected void getExtraFilesToTreate(Map<File, String[]> files) {
        if (extraIncludes != null && !extraIncludes.isEmpty()) {
            String[] includes = extraIncludes.split(",");
            getFilesToTreateForRoots(includes, null, compileSourceRoots, files, null);
            getFilesToTreateForRoots(includes, null, testCompileSourceRoots, files, null);
        }
    }
}
