package org.nuiton.jaxx.runtime.api.internal.binding;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.jaxx.runtime.api.JAXXObject;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeListenerProxy;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.WeakHashMap;

/**
 * Created on 4/5/15.
 *
 * @author Tony Chemit - chemit@codelutin.com
 * @since 3.0
 */
public class JAXXBindings {

    /** Logger. */
    private static final Log log = LogFactory.getLog(JAXXBindings.class);

    private static final PropertyChangeListener[] EMPTY_ARRAY_PROPERTY_CHANGE_LISTENERS = new PropertyChangeListener[0];

    private static final Map<JAXXObject, WeakReference<List<DataBindingUpdateListener>>>
            dataBindingUpdateListeners = new WeakHashMap<JAXXObject, WeakReference<List<DataBindingUpdateListener>>>();

    /**
     * To reload a binding, the method will invoke
     * <ul>
     * <li>{@link JAXXBinding#removeDataBinding()}</li>
     * <li>{@link JAXXBinding#applyDataBinding()}</li>
     * </ul>
     *
     * @param binding the binding to reload.
     * @since 2.4.2
     */
    public static void reloadBinding(JAXXBinding binding) {
        binding.removeDataBinding();
        binding.applyDataBinding();
    }

    /**
     * Convinient method to reload a given binding by his id on a JAXX ui.
     *
     * @param src       the ui to treate
     * @param bindingId the id of binding to reload.
     * @since 2.4.2
     */
    public static void reloadBinding(JAXXObject src, String bindingId) {
        JAXXBinding dataBinding = src.getDataBinding(bindingId);
        if (dataBinding != null) {
            reloadBinding(dataBinding);
        } else {
            if (log.isWarnEnabled()) {
                log.warn("Could not found binding[" + bindingId + "] on ui " + src);
            }
        }
    }

    /**
     * Convinient method to apply more than one binding on a JAXX ui.
     *
     * @param src      the ui to treate
     * @param bindings the list of binding to process.
     */
    public static void applyDataBinding(JAXXObject src, String... bindings) {
        for (String binding : bindings) {
            src.applyDataBinding(binding);
        }
    }

    /**
     * Convinient method to apply more than one binding on a JAXX ui.
     *
     * @param src      the ui to treate
     * @param bindings the list of binding to process.
     */
    public static void applyDataBinding(JAXXObject src,
                                        Collection<String> bindings) {
        for (String binding : bindings) {
            src.applyDataBinding(binding);
        }
    }

    /**
     * Convinient method to process more than one binding on a JAXX ui.
     *
     * @param src      the ui to treate
     * @param bindings the list of binding to process.
     */
    public static void processDataBinding(JAXXObject src, String... bindings) {
        for (String binding : bindings) {
            src.processDataBinding(binding);
        }
    }

    /**
     * Convinient method to remove more than one binding on a JAXX ui.
     *
     * @param src      the ui to treate
     * @param bindings the list of binding to process.
     */
    public static void removeDataBinding(JAXXObject src, String... bindings) {
        for (String binding : bindings) {
            src.removeDataBinding(binding);
        }
    }

    /**
     * Convinient method to remove all than one binding on a JAXX ui.
     *
     * @param src the ui to treate
     * @since 2.10
     */
    public static void removeAllDataBindings(JAXXObject src) {

        JAXXBinding[] bindings = src.getDataBindings();
        for (JAXXBinding binding : bindings) {
            removeDataBinding(src, binding.getId());
        }

    }

    public static DataBindingUpdateListener getDataBindingUpdateListener(
            JAXXObject object, String bindingName) {
        WeakReference<List<DataBindingUpdateListener>> ref =
                dataBindingUpdateListeners.get(object);
        List<DataBindingUpdateListener> listeners = ref == null ? null :
                                                    ref.get();
        if (listeners == null) {
            listeners = new ArrayList<DataBindingUpdateListener>();
            dataBindingUpdateListeners.put(
                    object,
                    new WeakReference<List<DataBindingUpdateListener>>(
                            listeners));
        } else {
            for (DataBindingUpdateListener listener : listeners) {
                if (bindingName.equals(listener.getBindingName())) {
                    return listener;
                }
            }
        }
        DataBindingUpdateListener listener =
                new DataBindingUpdateListener(object, bindingName);
        listeners.add(listener);
        return listener;
    }

    /**
     * detects all PropertychangedListener added by Jaxx uis (should be a {@link
     * DataBindingListener}
     *
     * @param propertyNames the array of property names to find
     * @param listeners     the array of listeners to filter
     * @return the filtered listeners
     */
    public static PropertyChangeListener[] findJaxxPropertyChangeListener(
            String[] propertyNames, PropertyChangeListener... listeners) {
        if (listeners == null || listeners.length == 0) {
            return EMPTY_ARRAY_PROPERTY_CHANGE_LISTENERS;
        }
        List<String> pNames = Arrays.asList(propertyNames);

        List<PropertyChangeListener> toRemove =
                new ArrayList<PropertyChangeListener>();

        for (PropertyChangeListener listener : listeners) {
            String pName = null;
            PropertyChangeListenerProxy plistener = null;
            if (listener instanceof PropertyChangeListenerProxy) {
                plistener = (PropertyChangeListenerProxy) listener;
                if (!pNames.contains(plistener.getPropertyName())) {
                    // not on the good property
                    continue;
                }
                listener = plistener.getListener();
                pName = plistener.getPropertyName();
            }
            if (plistener != null &&
                pName != null &&
                listener instanceof DataBindingListener) {
                if (log.isDebugEnabled()) {
                    log.debug("find config listener to remove  [" + pName +
                              "] : " + listener);
                }
                toRemove.add(plistener);
                //toRemove.add(listener);
            }
        }
        return toRemove.toArray(new PropertyChangeListener[toRemove.size()]);
    }
}
