package org.nuiton.jaxx.runtime.api.internal;

import java.lang.ref.WeakReference;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.EventListener;
import java.util.List;
import java.util.Map;
import java.util.WeakHashMap;

/**
 * Created on 4/5/15.
 *
 * @author Tony Chemit - chemit@codelutin.com
 * @since 3.0
 */
public class EventListeners {
    
    private static Map<Object, WeakReference<List<EventListenerDescriptor>>>
            eventListeners = new WeakHashMap<Object, WeakReference<List<EventListenerDescriptor>>>();

    public static <E extends EventListener> E getEventListener(
            Class<E> listenerClass,
            final String listenerMethodName,
            final Object methodContainer,
            String methodName) {

        WeakReference<List<EventListenerDescriptor>> ref =
                eventListeners.get(methodContainer);
        List<EventListenerDescriptor> descriptors = ref != null ?
                                                    ref.get() : null;
        if (descriptors == null) {
            descriptors = new ArrayList<EventListenerDescriptor>();
            eventListeners.put(
                    methodContainer,
                    new WeakReference<List<EventListenerDescriptor>>(
                            descriptors));
        } else {
            for (EventListenerDescriptor descriptor : descriptors) {
                if (descriptor.listenerClass.equals(listenerClass) &&
                    (listenerMethodName == null ?
                     descriptor.listenerMethodName == null :
                     listenerMethodName.equals(
                             descriptor.listenerMethodName)) &&
                    methodName.equals(descriptor.methodName)) {
                    return (E) descriptor.eventListener;
                }
            }
        }

        // else we need to create a new listener
        final EventListenerDescriptor descriptor = new EventListenerDescriptor();
        descriptor.listenerClass = listenerClass;
        descriptor.listenerMethodName = listenerMethodName;
        descriptor.methodName = methodName;
        try {
            final List<Method> listenerMethods =
                    Arrays.asList(listenerClass.getMethods());
            Method listenerMethod = null;
            if (listenerMethodName != null) {
                for (Method listenerMethod1 : listenerMethods) {
                    if (listenerMethod1.getName().equals(listenerMethodName)) {
                        listenerMethod = listenerMethod1;
                        break;
                    }
                }
            }
            if (listenerMethodName != null && listenerMethod == null) {
                throw new IllegalArgumentException(
                        "no method named " + listenerMethodName +
                        " found in class " + listenerClass.getName());
            }
            // tchemit 2010-12-01 : we must the exact method found, some none javaBeans
            // api does use different signature for some of them listener
            // an exemple is the TableColumnModelListener : http://download.oracle.com/javase/6/docs/api/javax/swing/event/TableColumnModelListener.html
            // This fix the bug https://forge.nuiton.org/issues/show/1124
            Class<?>[] parameterTypes;
            if (listenerMethodName != null) {

                // search an exact method, so must use the exact found listener method
                parameterTypes = listenerMethod.getParameterTypes();
            } else {

                // keep this horrible code which is not safe at all :
                // see previous comment
                parameterTypes = listenerMethods.get(0).getParameterTypes();
            }
            Class<?> methodContainerClass = methodContainer.getClass();
            final Method targetMethod =
                    methodContainerClass.getMethod(methodName, parameterTypes);
            descriptor.eventListener =
                    Proxy.newProxyInstance(listenerClass.getClassLoader(),
                                           new Class<?>[]{listenerClass},
                                           new InvocationHandler() {

                                               @Override
                                               public Object invoke(Object proxy,
                                                                    Method method,
                                                                    Object[] args) {
                                                   String methodName = method.getName();
                                                   if (listenerMethodName == null &&
                                                       listenerMethods.contains(method) ||
                                                       methodName.equals(listenerMethodName)) {
                                                       try {
                                                           targetMethod.setAccessible(true);
                                                           return targetMethod.invoke(
                                                                   methodContainer, args);
                                                       } catch (IllegalAccessException e) {
                                                           throw new RuntimeException(
                                                                   "could not invoke on container " +
                                                                   methodContainer, e);
                                                       } catch (InvocationTargetException e) {
                                                           throw new RuntimeException(e);
                                                       }
                                                   }
                                                   if (methodName.equals("toString")) {
                                                       return toString();
                                                   }
                                                   if (methodName.equals("equals")) {
                                                       return descriptor.eventListener == args[0];
                                                   }
                                                   if (methodName.equals("hashCode")) {
                                                       return hashCode();
                                                   }
                                                   return null;
                                               }
                                           });
            descriptors.add(descriptor);
            return (E) descriptor.eventListener;
        } catch (NoSuchMethodException e) {
            throw new RuntimeException(e);
        }
    }

    public static <E extends EventListener> E getEventListener(
            Class<E> listenerClass,
            Object methodContainer,
            String methodName) {
        return getEventListener(listenerClass, null, methodContainer,
                                methodName);
    }

    private static class EventListenerDescriptor {

        Class<?> listenerClass;

        String listenerMethodName;

        String methodName;

        Object eventListener;
    }
}
