/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.jose.jwk;

import com.fasterxml.jackson.databind.JsonNode;
import java.math.BigInteger;
import java.security.KeyFactory;
import java.security.PublicKey;
import java.security.spec.ECParameterSpec;
import java.security.spec.ECPoint;
import java.security.spec.ECPublicKeySpec;
import java.security.spec.RSAPublicKeySpec;
import org.keycloak.common.crypto.CryptoIntegration;
import org.keycloak.common.util.Base64Url;
import org.keycloak.jose.jwk.AKPUtils;
import org.keycloak.jose.jwk.JWK;
import org.keycloak.jose.jwk.JWKBuilder;
import org.keycloak.util.JsonSerialization;

public class JWKParser {
    protected JWK jwk;

    private JWKParser() {
    }

    public static JWKParser create() {
        return new JWKParser();
    }

    public JWKParser(JWK jwk) {
        this.jwk = jwk;
    }

    public static JWKParser create(JWK jwk) {
        return new JWKParser(jwk);
    }

    public JWKParser parse(String jwk) {
        try {
            this.jwk = (JWK)JsonSerialization.mapper.readValue(jwk, JWK.class);
            return this;
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public JWK getJwk() {
        return this.jwk;
    }

    public PublicKey toPublicKey() {
        if (this.jwk == null) {
            throw new IllegalStateException("Not possible to convert to the publicKey. The jwk is not set");
        }
        String keyType = this.jwk.getKeyType();
        JsonNode normalizedJwkNode = JsonSerialization.writeValueAsNode(this.jwk);
        if ("RSA".equals(keyType)) {
            return JWKParser.createRSAPublicKey(normalizedJwkNode);
        }
        if ("EC".equals(keyType)) {
            return JWKParser.createECPublicKey(normalizedJwkNode);
        }
        if ("OKP".equals(keyType)) {
            return JWKBuilder.EdEC_UTILS.createOKPPublicKey(this.jwk);
        }
        if ("AKP".equals(keyType)) {
            return JWKParser.createAPKPublicKey(normalizedJwkNode);
        }
        throw new RuntimeException("Unsupported keyType " + keyType);
    }

    private static PublicKey createECPublicKey(JsonNode jwk) {
        String name;
        String crv = jwk.path("crv").asText(null);
        String xStr = jwk.get("x").asText(null);
        String yStr = jwk.get("y").asText(null);
        if (crv == null || xStr == null || yStr == null) {
            throw new RuntimeException("Fail to retrieve ECPublicJWK.CRV, ECPublicJWK.X or ECPublicJWK.Y field.");
        }
        BigInteger x = new BigInteger(1, Base64Url.decode(xStr));
        BigInteger y = new BigInteger(1, Base64Url.decode(yStr));
        switch (crv) {
            case "P-256": {
                name = "secp256r1";
                break;
            }
            case "P-384": {
                name = "secp384r1";
                break;
            }
            case "P-521": {
                name = "secp521r1";
                break;
            }
            default: {
                throw new RuntimeException("Unsupported curve");
            }
        }
        try {
            ECPoint point = new ECPoint(x, y);
            ECParameterSpec params = CryptoIntegration.getProvider().createECParams(name);
            ECPublicKeySpec pubKeySpec = new ECPublicKeySpec(point, params);
            KeyFactory kf = CryptoIntegration.getProvider().getKeyFactory("ECDSA");
            return kf.generatePublic(pubKeySpec);
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private static PublicKey createRSAPublicKey(JsonNode jwk) {
        BigInteger modulus = new BigInteger(1, Base64Url.decode(jwk.path("n").asText(null)));
        BigInteger publicExponent = new BigInteger(1, Base64Url.decode(jwk.path("e").asText(null)));
        try {
            KeyFactory kf = KeyFactory.getInstance("RSA");
            return kf.generatePublic(new RSAPublicKeySpec(modulus, publicExponent));
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private static PublicKey createAPKPublicKey(JsonNode jwk) {
        String algorithm = jwk.path("alg").asText();
        String publicKey = jwk.path("pub").asText();
        return AKPUtils.fromEncodedPub(publicKey, algorithm);
    }

    public boolean isKeyTypeSupported(String keyType) {
        return "RSA".equals(keyType) || "EC".equals(keyType) || JWKBuilder.EdEC_UTILS.isEdECSupported() && "OKP".equals(keyType) || "AKP".equals(keyType);
    }
}

