package org.hsqldb.persist;

import java.io.File;
import java.io.IOException;
import org.hsqldb.Database;
import org.hsqldb.HsqlException;
import org.hsqldb.NumberSequence;
import org.hsqldb.Session;
import org.hsqldb.SessionManager;
import org.hsqldb.Table;
import org.hsqldb.Trace;
import org.hsqldb.lib.Collection;
import org.hsqldb.lib.FileAccess;
import org.hsqldb.lib.HashMap;
import org.hsqldb.lib.Iterator;
import org.hsqldb.lib.SimpleLog;
import org.hsqldb.lib.ZipUnzipFile;
import org.hsqldb.scriptio.ScriptReaderBase;
import org.hsqldb.scriptio.ScriptWriterBase;

public class Log
{
  private HsqlDatabaseProperties properties;
  private String fileName;
  private Database database;
  private FileAccess fa;
  private ScriptWriterBase dbLogWriter;
  private String scriptFileName;
  private String logFileName;
  private boolean filesReadOnly;
  private long maxLogSize;
  private int writeDelay;
  private int scriptFormat;
  private DataFileCache cache;
  private HashMap textCacheList = new HashMap();

  Log(Database paramDatabase)
    throws HsqlException
  {
    this.database = paramDatabase;
    this.fa = paramDatabase.getFileAccess();
    this.fileName = paramDatabase.getPath();
    this.properties = paramDatabase.getProperties();
  }

  void initParams()
  {
    int i = this.properties.getIntegerProperty("hsqldb.log_size", 0);
    this.maxLogSize = (i * 1024 * 1024);
    this.scriptFormat = this.properties.getIntegerProperty("hsqldb.script_format", 0);
    this.writeDelay = this.properties.getDefaultWriteDelay();
    this.filesReadOnly = this.database.isFilesReadOnly();
    this.scriptFileName = (this.fileName + ".script");
    this.logFileName = (this.fileName + ".log");
  }

  void open()
    throws HsqlException
  {
    initParams();
    int i = this.properties.getDBModified();
    switch (i)
    {
    case 1:
      deleteNewAndOldFiles();
      restoreBackup();
      processScript();
      processDataFile();
      processLog();
      close(false);
      if (this.cache != null)
        this.cache.open(this.filesReadOnly);
      reopenAllTextCaches();
      break;
    case 2:
      try
      {
        deleteBackup();
        backupData();
        renameNewBackup();
        renameNewScript();
        deleteLog();
        this.properties.setDBModified(0);
      }
      catch (IOException localIOException)
      {
        this.database.logger.appLog.logContext(localIOException, null);
      }
    case 0:
      processScript();
      if (isAnyCacheModified())
      {
        this.properties.setDBModified(1);
        close(false);
        if (this.cache != null)
          this.cache.open(this.filesReadOnly);
        reopenAllTextCaches();
      }
    }
    openLog();
    if (!this.filesReadOnly)
      this.properties.setDBModified(1);
  }

  void close(boolean paramBoolean)
    throws HsqlException
  {
    closeLog();
    deleteNewAndOldFiles();
    writeScript(paramBoolean);
    closeAllTextCaches(paramBoolean);
    if (this.cache != null)
      this.cache.close(true);
    this.properties.setProperty("version", "1.8.0");
    this.properties.setProperty("hsqldb.compatible_version", "1.8.0");
    this.properties.setDBModified(2);
    deleteLog();
    if (paramBoolean)
    {
      deleteBackup();
      deleteData();
    }
    else
    {
      try
      {
        backupData();
        renameNewBackup();
      }
      catch (IOException localIOException)
      {
      }
    }
    renameNewScript();
    this.properties.setProperty("hsqldb.cache_version", "1.7.0");
    this.properties.setDBModified(0);
  }

  void shutdown()
    throws HsqlException
  {
    synchLog();
    if (this.cache != null)
      this.cache.close(false);
    closeAllTextCaches(false);
    closeLog();
  }

  void deleteNewAndOldFiles()
  {
    this.fa.removeElement(this.fileName + ".data" + ".old");
    this.fa.removeElement(this.fileName + ".data" + ".new");
    this.fa.removeElement(this.fileName + ".backup" + ".new");
    this.fa.removeElement(this.scriptFileName + ".new");
  }

  void deleteBackup()
  {
    this.fa.removeElement(this.fileName + ".backup");
  }

  void deleteData()
  {
    this.fa.removeElement(this.fileName + ".data");
  }

  void backupData()
    throws IOException
  {
    if (this.fa.isStreamElement(this.fileName + ".data"))
      ZipUnzipFile.compressFile(this.fileName + ".data", this.fileName + ".backup.new", this.database.getFileAccess());
  }

  void renameNewBackup()
  {
    if (this.fa.isStreamElement(this.fileName + ".backup.new"))
      this.fa.renameElement(this.fileName + ".backup.new", this.fileName + ".backup");
  }

  void renameNewScript()
  {
    if (this.fa.isStreamElement(this.scriptFileName + ".new"))
      this.fa.renameElement(this.scriptFileName + ".new", this.scriptFileName);
  }

  void deleteNewScript()
  {
    this.fa.removeElement(this.scriptFileName + ".new");
  }

  void deleteNewBackup()
  {
    this.fa.removeElement(this.scriptFileName + ".backup.new");
  }

  void deleteLog()
  {
    this.fa.removeElement(this.logFileName);
  }

  boolean isAnyCacheModified()
  {
    if ((this.cache != null) && (this.cache.isFileModified()))
      return true;
    return isAnyTextCacheModified();
  }

  void checkpoint(boolean paramBoolean)
    throws HsqlException
  {
    if (this.filesReadOnly)
      return;
    this.database.logger.appLog.logContext(SimpleLog.LOG_NORMAL, "start");
    deleteNewAndOldFiles();
    if (this.cache != null)
    {
      if (forceDefrag())
        paramBoolean = true;
      if (paramBoolean)
      {
        try
        {
          this.cache.defrag();
        }
        catch (Exception localException)
        {
        }
      }
      else
      {
        this.cache.close(true);
        try
        {
          this.cache.backupFile();
        }
        catch (IOException localIOException1)
        {
          deleteNewBackup();
          this.cache.open(false);
          return;
        }
        this.cache.open(false);
      }
    }
    writeScript(false);
    this.properties.setDBModified(2);
    closeLog();
    deleteLog();
    renameNewScript();
    renameNewBackup();
    this.properties.setDBModified(1);
    if (this.dbLogWriter == null)
      return;
    openLog();
    Session[] arrayOfSession = this.database.sessionManager.getAllSessions();
    try
    {
      for (int i = 0; i < arrayOfSession.length; i++)
      {
        Session localSession = arrayOfSession[i];
        if (localSession.isAutoCommit())
          continue;
        this.dbLogWriter.writeLogStatement(localSession, localSession.getAutoCommitStatement());
      }
    }
    catch (IOException localIOException2)
    {
      throw Trace.error(29, this.logFileName);
    }
    this.database.logger.appLog.logContext(SimpleLog.LOG_NORMAL, "end");
  }

  boolean forceDefrag()
  {
    long l1 = this.properties.getIntegerProperty("hsqldb.defrag_limit", 200);
    long l2 = l1 * 1024L * 1024L;
    long l3 = this.cache.freeBlocks.getLostBlocksSize();
    return l3 > l2;
  }

  boolean hasCache()
  {
    return this.cache != null;
  }

  DataFileCache getCache()
    throws HsqlException
  {
    if (this.cache == null)
    {
      this.cache = new DataFileCache(this.database, this.fileName);
      this.cache.open(this.filesReadOnly);
    }
    return this.cache;
  }

  int getLogSize()
  {
    return (int)(this.maxLogSize / 11288576L);
  }

  void setLogSize(int paramInt)
  {
    this.properties.setProperty("hsqldb.log_size", String.valueOf(paramInt));
    this.maxLogSize = (paramInt * 1024 * 1024);
  }

  int getScriptType()
  {
    return this.scriptFormat;
  }

  void setScriptType(int paramInt)
    throws HsqlException
  {
    if (this.database.isStoredFileAccess())
      return;
    int i = this.scriptFormat != paramInt ? 1 : 0;
    this.scriptFormat = paramInt;
    this.properties.setProperty("hsqldb.script_format", String.valueOf(this.scriptFormat));
    if (i != 0)
      this.database.logger.needsCheckpoint = true;
  }

  int getWriteDelay()
  {
    return this.writeDelay;
  }

  void setWriteDelay(int paramInt)
  {
    this.writeDelay = paramInt;
    if (this.dbLogWriter != null)
    {
      synchLog();
      this.dbLogWriter.setWriteDelay(paramInt);
    }
  }

  void writeStatement(Session paramSession, String paramString)
    throws HsqlException
  {
    if ((paramString == null) || (paramString.length() == 0))
      return;
    try
    {
      this.dbLogWriter.writeLogStatement(paramSession, paramString);
    }
    catch (IOException localIOException)
    {
      throw Trace.error(29, this.logFileName);
    }
    if ((this.maxLogSize > 0L) && (this.dbLogWriter.size() > this.maxLogSize))
      this.database.logger.needsCheckpoint = true;
  }

  void writeInsertStatement(Session paramSession, Table paramTable, Object[] paramArrayOfObject)
    throws HsqlException
  {
    try
    {
      this.dbLogWriter.writeInsertStatement(paramSession, paramTable, paramArrayOfObject);
    }
    catch (IOException localIOException)
    {
      throw Trace.error(29, this.logFileName);
    }
    if ((this.maxLogSize > 0L) && (this.dbLogWriter.size() > this.maxLogSize))
      this.database.logger.needsCheckpoint = true;
  }

  void writeDeleteStatement(Session paramSession, Table paramTable, Object[] paramArrayOfObject)
    throws HsqlException
  {
    try
    {
      this.dbLogWriter.writeDeleteStatement(paramSession, paramTable, paramArrayOfObject);
    }
    catch (IOException localIOException)
    {
      throw Trace.error(29, this.logFileName);
    }
    if ((this.maxLogSize > 0L) && (this.dbLogWriter.size() > this.maxLogSize))
      this.database.logger.needsCheckpoint = true;
  }

  void writeSequenceStatement(Session paramSession, NumberSequence paramNumberSequence)
    throws HsqlException
  {
    try
    {
      this.dbLogWriter.writeSequenceStatement(paramSession, paramNumberSequence);
    }
    catch (IOException localIOException)
    {
      throw Trace.error(29, this.logFileName);
    }
    if ((this.maxLogSize > 0L) && (this.dbLogWriter.size() > this.maxLogSize))
      this.database.logger.needsCheckpoint = true;
  }

  void writeCommitStatement(Session paramSession)
    throws HsqlException
  {
    try
    {
      this.dbLogWriter.writeCommitStatement(paramSession);
    }
    catch (IOException localIOException)
    {
      throw Trace.error(29, this.logFileName);
    }
    if ((this.maxLogSize > 0L) && (this.dbLogWriter.size() > this.maxLogSize))
      this.database.logger.needsCheckpoint = true;
  }

  void synchLog()
  {
    if (this.dbLogWriter != null)
      this.dbLogWriter.sync();
  }

  private void openLog()
    throws HsqlException
  {
    if (this.filesReadOnly)
      return;
    try
    {
      this.dbLogWriter = ScriptWriterBase.newScriptWriter(this.database, this.logFileName, false, false, 0);
      this.dbLogWriter.setWriteDelay(this.writeDelay);
      this.dbLogWriter.start();
    }
    catch (Exception localException)
    {
      throw Trace.error(29, this.logFileName);
    }
  }

  private synchronized void closeLog()
    throws HsqlException
  {
    if (this.dbLogWriter != null)
      this.dbLogWriter.close();
  }

  private void writeScript(boolean paramBoolean)
    throws HsqlException
  {
    deleteNewScript();
    ScriptWriterBase localScriptWriterBase = ScriptWriterBase.newScriptWriter(this.database, this.scriptFileName + ".new", paramBoolean, true, this.scriptFormat);
    localScriptWriterBase.writeAll();
    localScriptWriterBase.close();
  }

  private void processScript()
    throws HsqlException
  {
    ScriptReaderBase localScriptReaderBase = null;
    try
    {
      if ((this.database.isFilesInJar()) || (this.fa.isStreamElement(this.scriptFileName)))
      {
        localScriptReaderBase = ScriptReaderBase.newScriptReader(this.database, this.scriptFileName, this.scriptFormat);
        localScriptReaderBase.readAll(this.database.sessionManager.getSysSession(null, true));
        localScriptReaderBase.close();
      }
    }
    catch (Throwable localThrowable)
    {
      if (localScriptReaderBase != null)
      {
        localScriptReaderBase.close();
        if (this.cache != null)
          this.cache.close(false);
        closeAllTextCaches(false);
      }
      this.database.logger.appLog.logContext(localThrowable, null);
      if ((localThrowable instanceof HsqlException))
        throw ((HsqlException)localThrowable);
      if ((localThrowable instanceof IOException))
        throw Trace.error(29, localThrowable.toString());
      if ((localThrowable instanceof OutOfMemoryError))
        throw Trace.error(72);
      throw Trace.error(40, localThrowable.toString());
    }
  }

  private void processDataFile()
    throws HsqlException
  {
    if ((this.cache == null) || (this.filesReadOnly) || (this.database.isStoredFileAccess()) || (!this.fa.isStreamElement(this.logFileName)))
      return;
    File localFile = new File(this.logFileName);
    long l1 = localFile.length();
    long l2 = this.cache.getFileFreePos();
    if (l1 + l2 > this.cache.maxDataFileSize)
      this.database.logger.needsCheckpoint = true;
  }

  private void processLog()
    throws HsqlException
  {
    if ((!this.database.isFilesInJar()) && (this.fa.isStreamElement(this.logFileName)))
      ScriptRunner.runScript(this.database, this.logFileName, 0);
  }

  private void restoreBackup()
    throws HsqlException
  {
    DataFileCache.deleteOrResetFreePos(this.database, this.fileName + ".data");
    try
    {
      ZipUnzipFile.decompressFile(this.fileName + ".backup", this.fileName + ".data", this.database.getFileAccess());
    }
    catch (Exception localException)
    {
      throw Trace.error(29, 115, new Object[] { this.fileName + ".backup", localException.toString() });
    }
  }

  DataFileCache openTextCache(Table paramTable, String paramString, boolean paramBoolean1, boolean paramBoolean2)
    throws HsqlException
  {
    closeTextCache(paramTable);
    Object localObject;
    if (!this.properties.isPropertyTrue("textdb.allow_full_path"))
    {
      if (paramString.indexOf("..") != -1)
        throw Trace.error(33, paramString);
      localObject = new File(new File(this.database.getPath() + ".properties").getAbsolutePath()).getParent();
      if (localObject != null)
        paramString = (String)localObject + File.separator + paramString;
    }
    if (paramBoolean2)
      localObject = new TextCache(paramTable, paramString);
    else
      localObject = new TextCache(paramTable, paramString);
    ((TextCache)localObject).open((paramBoolean1) || (this.filesReadOnly));
    this.textCacheList.put(paramTable.getName(), localObject);
    return (DataFileCache)localObject;
  }

  void closeTextCache(Table paramTable)
    throws HsqlException
  {
    TextCache localTextCache = (TextCache)this.textCacheList.remove(paramTable.getName());
    if (localTextCache != null)
      localTextCache.close(true);
  }

  private void closeAllTextCaches(boolean paramBoolean)
    throws HsqlException
  {
    Iterator localIterator = this.textCacheList.values().iterator();
    while (localIterator.hasNext())
    {
      if (paramBoolean)
      {
        ((TextCache)localIterator.next()).purge();
        continue;
      }
      ((TextCache)localIterator.next()).close(true);
    }
  }

  private void reopenAllTextCaches()
    throws HsqlException
  {
    Iterator localIterator = this.textCacheList.values().iterator();
    while (localIterator.hasNext())
      ((TextCache)localIterator.next()).reopen();
  }

  private boolean isAnyTextCacheModified()
  {
    Iterator localIterator = this.textCacheList.values().iterator();
    while (localIterator.hasNext())
      if (((TextCache)localIterator.next()).isFileModified())
        return true;
    return false;
  }
}

/* Location:           /home/mnovotny/projects/EMBEDDED_JBOSS_BETA3_COMMUNITY/embedded/output/lib/embedded-jboss/lib/thirdparty-all.jar
 * Qualified Name:     org.hsqldb.persist.Log
 * JD-Core Version:    0.6.0
 */