/*
 * Hibernate, Relational Persistence for Idiomatic Java
 *
 * Copyright (c) 2010, Red Hat Inc. or third-party contributors as
 * indicated by the @author tags or express copyright attribution
 * statements applied by the authors.  All third-party contributions are
 * distributed under license by Red Hat Inc.
 *
 * This copyrighted material is made available to anyone wishing to use, modify,
 * copy, or redistribute it subject to the terms and conditions of the GNU
 * Lesser General Public License, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
 * for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this distribution; if not, write to:
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301  USA
 */
package org.hibernate.type.descriptor.java;

import java.util.Calendar;
import java.util.Comparator;
import java.util.Date;
import java.util.GregorianCalendar;

import org.hibernate.cfg.Environment;
import org.hibernate.internal.util.compare.CalendarComparator;
import org.hibernate.type.descriptor.WrapperOptions;

/**
 * Descriptor for {@link java.util.Calendar} handling, but just for the time portion.
 *
 * @author Steve Ebersole
 */
public class CalendarTimeTypeDescriptor extends AbstractTypeDescriptor<Calendar> {
	public static final CalendarTimeTypeDescriptor INSTANCE = new CalendarTimeTypeDescriptor();

	protected CalendarTimeTypeDescriptor() {
		super( Calendar.class, CalendarTypeDescriptor.CalendarMutabilityPlan.INSTANCE );
	}

	public String toString(Calendar value) {
		return DateTypeDescriptor.INSTANCE.toString( value.getTime() );
	}

	public Calendar fromString(String string) {
		Calendar result = new GregorianCalendar();
		result.setTime( DateTypeDescriptor.INSTANCE.fromString( string ) );
		return result;
	}

	@Override
	public boolean areEqual(Calendar one, Calendar another) {
		if ( one == another ) {
			return true;
		}
		if ( one == null || another == null ) {
			return false;
		}

		return one.get(Calendar.DAY_OF_MONTH) == another.get(Calendar.DAY_OF_MONTH)
			&& one.get(Calendar.MONTH) == another.get(Calendar.MONTH)
			&& one.get(Calendar.YEAR) == another.get(Calendar.YEAR);
	}

	@Override
	public int extractHashCode(Calendar value) {
		int hashCode = 1;
		hashCode = 31 * hashCode + value.get(Calendar.DAY_OF_MONTH);
		hashCode = 31 * hashCode + value.get(Calendar.MONTH);
		hashCode = 31 * hashCode + value.get(Calendar.YEAR);
		return hashCode;
	}

	@Override
	public Comparator<Calendar> getComparator() {
		return CalendarComparator.INSTANCE;
	}

	@SuppressWarnings({ "unchecked" })
	public <X> X unwrap(Calendar value, Class<X> type, WrapperOptions options) {
		if ( value == null ) {
			return null;
		}
		if ( Calendar.class.isAssignableFrom( type ) ) {
			return (X) value;
		}
		if ( java.sql.Date.class.isAssignableFrom( type ) ) {
			return (X) new java.sql.Date( value.getTimeInMillis() );
		}
		if ( java.sql.Time.class.isAssignableFrom( type ) ) {
			return (X) new java.sql.Time( value.getTimeInMillis() );
		}
		if ( java.sql.Timestamp.class.isAssignableFrom( type ) ) {
			return (X) new java.sql.Timestamp( value.getTimeInMillis() );
		}
		if ( Date.class.isAssignableFrom( type ) ) {
			return (X) new  Date( value.getTimeInMillis() );
		}
		throw unknownUnwrap( type );
	}

	public <X> Calendar wrap(X value, WrapperOptions options) {
		Calendar cal = null;
		
		if ( value == null ) {
			return null;
		}
		if ( Calendar.class.isInstance( value ) ) {
			cal = (Calendar) value;
		}

		if ( ! Date.class.isInstance( value ) ) {
			throw unknownWrap( value.getClass() );
		}
		
		if ( cal == null ) {
			cal = new GregorianCalendar();
			if ( Environment.jvmHasTimestampBug() ) {
				final long milliseconds = ( (Date) value ).getTime();
				final long nanoseconds = java.sql.Timestamp.class.isInstance( value )
						? ( (java.sql.Timestamp) value ).getNanos()
						: 0;
				cal.setTime( new Date( milliseconds + nanoseconds / 1000000 ) );
			}
			else {
				cal.setTime( (Date) value );
			}
		}
		
		// Ensure the year/month/date is actually stripped out.
		cal.set( Calendar.YEAR, 0 );
		cal.set( Calendar.MONTH, 0 );
		cal.set( Calendar.DATE, 0 );
		
		return cal;
	}
}
