/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.ui.pages.poll;

import java.io.File;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;

import org.apache.tapestry5.BindingConstants;
import org.apache.tapestry5.Block;
import org.apache.tapestry5.annotations.Component;
import org.apache.tapestry5.annotations.IncludeStylesheet;
import org.apache.tapestry5.annotations.InjectPage;
import org.apache.tapestry5.annotations.Parameter;
import org.apache.tapestry5.annotations.Persist;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.ioc.Messages;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.chorem.pollen.business.ServicePoll;
import org.chorem.pollen.business.ServicePollAccount;
import org.chorem.pollen.business.ServiceResults;
import org.chorem.pollen.business.dto.ChoiceDTO;
import org.chorem.pollen.business.dto.PollAccountDTO;
import org.chorem.pollen.business.dto.PollDTO;
import org.chorem.pollen.business.dto.ResultDTO;
import org.chorem.pollen.business.utils.MD5;
import org.chorem.pollen.common.ChoiceType;
import org.chorem.pollen.ui.components.Chart;
import org.chorem.pollen.ui.components.FeedBack;
import org.chorem.pollen.ui.data.Lien;
import org.chorem.pollen.ui.services.Configuration;

/**
 * Classe de la page de rendu des résultats.
 *
 * @author rannou
 * @version $Id: Results.java 2615 2009-07-01 13:47:45Z nrannou $
 */
@IncludeStylesheet("context:css/results.css")
public class Results {
    @Parameter(defaultPrefix = BindingConstants.MESSAGE, value = "title")
    @Property
    private String title;

    @SuppressWarnings("unused")
    @Property
    private Lien[] address;

    /** Affichage des messages pour l'utilisateur */
    @Component(id = "feedback")
    private FeedBack feedback;

    /** Affichage simplifié des résultats */
    @Inject
    private Block chartBlock;

    /** Affichage détaillé des résultats */
    @Inject
    private Block chartBlockFull;

    /** Affichage détaillé ? */
    @Persist
    @Property
    private boolean full;

    /** Indique si l'utilisateur est authorisé */
    @Property
    private boolean userAllowed = false;

    /** Type de représentation des résultats */
    @SuppressWarnings("unused")
    @Persist
    @Property
    private int type;

    /** Sondage concerné par les résultats */
    @Property
    @Persist
    private PollDTO poll;

    /** Résultats du sondage */
    @Persist
    @Property
    private List<ResultDTO> results;

    /** Résultat courant */
    @Property
    private ResultDTO result;

    /** Choix courant (diagramme) */
    @SuppressWarnings("unused")
    @Property
    private List<String> choice;

    /** Localisation */
    @Inject
    private Messages messages;

    /** Page d'affichage d'une image */
    @InjectPage
    private ImageDisplay imageDisplay;

    /**
     * Service contenant la configuration de l'application.
     */
    @Inject
    private Configuration conf;

    /** Injection des services */
    @Inject
    private ServicePoll servicePoll;
    @Inject
    private ServicePollAccount servicePollAccount;
    @Inject
    private ServiceResults serviceResults;

    /** Récupération de la liste des résultats pour le diagramme. */
    public ArrayList<ArrayList<String>> getChoices() {
        ArrayList<ArrayList<String>> choices = new ArrayList<ArrayList<String>>();
        ArrayList<String> choiceValues = new ArrayList<String>();

        for (ResultDTO result : results) {
            String name = String.valueOf(result.getName());
            String value = String.valueOf(result.getValue());

            if (poll.getChoiceType() == ChoiceType.DATE) { // mise en forme de la date
                Date date = new Date(Long.parseLong(result.getName()));
                SimpleDateFormat dateFormat = new SimpleDateFormat(messages
                        .get("date-pattern"));
                name = dateFormat.format(date);
            }

            choiceValues.add(name);
            choiceValues.add(value);
        }

        choices.add(choiceValues);
        return choices;
    }

    /** Récupération de la liste des résultats pour le classement. */
    public List<ResultDTO> getRanking() {
        List<ResultDTO> ranking = results;

        Collections.sort(ranking, new Comparator<ResultDTO>() {
            @Override
            public int compare(ResultDTO o1, ResultDTO o2) {
                Double result1 = Double.parseDouble(o1.getValue());
                Double result2 = Double.parseDouble(o2.getValue());
                int comp = 0; // résultat de la comparaison

                if (result1 > result2) {
                    comp = -1;
                } else if (result1 < result2) {
                    comp = 1;
                }
                if (comp == 0) {
                    comp = o1.getName().compareTo(o2.getName());
                }
                return comp;
            }
        });

        return ranking;
    }

    /** Récupération des résultats gagnants. */
    public List<ResultDTO> getTopRanking() {
        List<ResultDTO> ranking = getRanking();
        List<ResultDTO> winners = new ArrayList<ResultDTO>();

        String winValue = null;
        if (ranking.size() > 0) {
            winValue = ranking.get(0).getValue();
        } else {
            return winners;
        }

        for (ResultDTO r : ranking) {
            if (winValue.equals(r.getValue())) {
                winners.add(r);
            }
        }
        return winners;
    }

    /**
     * Changement de type de diagrammes
     *
     * @return le bloc actif
     */
    Block onType(int type) {
        this.type = type;
        return getActiveBlock();
    }

    /**
     * Changement de type d'affichage (full, simple)
     *
     * @return le bloc actif
     */
    Block onSuccessFromDisplayType() {
        return getActiveBlock();
    }

    /**
     * Sélection du bloc actif
     *
     * @return le bloc actif
     */
    public Block getActiveBlock() {
        return full ? chartBlockFull : chartBlock;
    }

    public boolean isTextType() {
        return poll.getChoiceType() == ChoiceType.TEXT;
    }

    public boolean isDateType() {
        return poll.getChoiceType() == ChoiceType.DATE;
    }

    public boolean isImageType() {
        return poll.getChoiceType() == ChoiceType.IMAGE;
    }

    /**
     * Récupération du créateur du sondage.
     *
     * @return le nom du créateur du sondage.
     */
    public String getCreatorName() {
        PollAccountDTO pollAccount = servicePollAccount
                .findPollAccountById(poll.getCreatorId());
        return pollAccount.getVotingId();
    }

    /** Retourne le message d'aide correspondant au type de sondage. */
    public String getHelpMessage() {
        switch (poll.getVoteCounting()) {
        case NORMAL:
            return messages.get("normalVote-help");
        case PERCENTAGE:
            return messages.get("percentageVote-help");
        case CONDORCET:
            return messages.get("condorcetVote-help");
        default:
            return "";
        }
    }

    /** Retourne le message de victoire indiquant le ou les gagnants. */
    public String getVictoryMessage() {
        List<ResultDTO> winners = getTopRanking();

        if (winners.size() == 0) {
            return "";
        } else if (winners.size() == 1) {
            return messages.format("victory");
        } else {
            return messages.format("victories");
        }
    }

    /** Retourne la valeur du résultat courant sans le .0 final */
    public String getTrimValue() {
        String value = result.getValue();

        // le résultat peut-être un double : 1,0 -> 1 et 1,2 -> 1,2
        if (value.endsWith(".0")) {
            value = value.substring(0, value.indexOf('.'));
        }
        return value;
    }

    /** Retourne la date correspondant au résultat courant */
    public Date getResultNameAsDate() {
        return new Date(Long.valueOf(result.getName()));
    }

    /** Retourne le chemin de la miniature correspondant au résultat courant */
    public String getResultThumbSrc() {
        File dir = new File(conf.getProperty("upImgDir"), poll.getPollId());
        File file = new File(dir, "thumb_" + result.getName());
        return file.getAbsolutePath();
    }

    /** Action réalisée lorsqu'on clique sur l'image */
    Object onDisplayImage(String resultName) {
        String choiceId = "";

        // Récupération du choix correspondant au résultat
        for (ChoiceDTO choice : poll.getChoiceDTOs()) {
            if (resultName.equals(choice.getName())) {
                choiceId = choice.getId();
            }
        }

        imageDisplay.setPoll(poll);
        imageDisplay.setChoiceId(choiceId);
        imageDisplay.setPageStyle("VoteCounting");
        return imageDisplay;
    }

    /** Initialisation des objets de session */
    void onActivate(String id) {
        String pollId = null;

        if (id != null && !"".equals(id)) {
            pollId = id.split(":", 2)[0];
            poll = servicePoll.findPollByPollId(pollId);

            if (poll != null) {

                // Identification de l'utilisateur
                if (poll.isPublicResults()) {
                    userAllowed = true;
                } else if (id.split(":", 2).length == 2) {
                    String creatorId = id.split(":", 2)[1];
                    if (creatorId.equals(MD5.encode(poll.getCreatorId()))) {
                        userAllowed = true;
                    }
                }

                // Initialisation des résultats
                if (userAllowed) {
                    type = Chart.PIE;
                    full = true;
                    results = serviceResults.getAllResults(pollId);
                }
            }
        }

        // Affichage des erreurs
        if (poll == null) {
            feedback.addError(messages.get("pollNotFound"));
        } else if (!userAllowed) {
            feedback.addError(messages.get("userNotAllowed"));
        } else if (!poll.isClosed()) {
            feedback.addError(messages.get("pollNotClosed"));
        }
    }

    /**
     * Initialisation de l'affichage
     */
    void setupRender() {
        address = new Lien[] { new Lien("Pollen", "Index"),
                new Lien(title, null) };
    }
}
