/*
 * #%L
 * Pollen :: UI (strust2)
 * 
 * $Id: AbstractVoteAction.java 3333 2012-04-30 15:52:41Z fdesbois $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.3/pollen-ui-struts2/src/main/java/org/chorem/pollen/ui/actions/poll/AbstractVoteAction.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.ui.actions.poll;

import com.google.common.base.Preconditions;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.struts2.interceptor.ParameterAware;
import org.chorem.pollen.bean.PollResult;
import org.chorem.pollen.bean.PollResultList;
import org.chorem.pollen.business.persistence.Choice;
import org.chorem.pollen.business.persistence.Comment;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.business.persistence.UserAccount;
import org.chorem.pollen.business.persistence.Vote;
import org.chorem.pollen.business.persistence.VoteToChoice;
import org.chorem.pollen.common.ChoiceType;
import org.chorem.pollen.common.PollType;
import org.chorem.pollen.common.VoteCountingType;
import org.chorem.pollen.services.exceptions.PollAccountNotFound;
import org.chorem.pollen.services.exceptions.PollNotFoundException;
import org.chorem.pollen.services.impl.PollCommentService;
import org.chorem.pollen.services.impl.PollFeedService;
import org.chorem.pollen.services.impl.PollResultsService;
import org.chorem.pollen.services.impl.PollService;
import org.chorem.pollen.services.impl.VoteService;
import org.chorem.pollen.ui.actions.PageSkin;

import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * Abstract action for actions on the vote poll page.
 *
 * @author tchemit <chemit@codelutin.com>
 * @author fdesbois <fdesbois@codelutin.com>
 * @since 1.3
 */
public abstract class AbstractVoteAction extends AbstractPollUriIdAction implements ParameterAware {

    private static final long serialVersionUID = 1L;

    /** Logger. */
    private static final Log log = LogFactory.getLog(AbstractVoteAction.class);

    private transient PollResultsService pollResultsService;
    
    private transient PollService pollService;

    private transient VoteService voteService;
    
    private transient PollCommentService pollCommentService;

    private Poll poll;

    private boolean feedFileExisting;

    private boolean creatorUser;

    private PollAccount pollAccount;

    private boolean alreadyVoted;

    private boolean voteAllowed;
    
    private List<Vote> votes;
    
    private Vote vote;
    
    private String commentAuthor;

    private String voteSizeMessage;

    private List<PollResult> results;

    private List<Comment> comments;
    
    private Map<String, String[]> parameters;

    @Override
    public PageSkin getSkin() {
        return PageSkin.VOTE;
    }

    protected PollResultsService getPollResultsService() {
        if (pollResultsService == null) {
            pollResultsService = newService(PollResultsService.class);
        }
        return pollResultsService;
    }
    
    protected PollService getPollService() {
        if (pollService == null) {
            pollService = newService(PollService.class);
        }
        return pollService;
    }

    protected VoteService getVoteService() {
        if (voteService == null) {
            voteService = newService(VoteService.class);
        }
        return voteService;
    }

    protected PollCommentService getPollCommentService() {
        if (pollCommentService == null) {
            pollCommentService = newService(PollCommentService.class);
        }
        return pollCommentService;
    }

    @Override
    public void setParameters(Map<String, String[]> parameters) {
        this.parameters = parameters;
    }

    public Poll getPoll() {
        return poll;
    }

    public boolean isFeedFileExisting() {
        return feedFileExisting;
    }

    public PollAccount getPollAccount() {
        return pollAccount;
    }

    public List<Vote> getVotes() {
        return votes;
    }

    public Vote getVote() {
        return vote;
    }

    public boolean isPollChoiceOrVoteStarted() {
        Date now = serviceContext.getCurrentTime();
        return poll.isAddChoiceStarted(now) || poll.isStarted(now);
    }

    public boolean isAlreadyVoted() {
        return alreadyVoted;
    }

    public boolean isAnonymousVote() {
        return poll.isAnonymous();
    }

    public List<PollResult> getResults() {
        return results;
    }

    public List<Comment> getComments() {
        return comments;
    }

    public boolean isCreatorUser() {
        return creatorUser;
    }

    public boolean isAccountFieldDisplayed() {
        return !poll.isAnonymous() || isRestrictedPoll() || isGroupPoll();
    }

    public boolean isPollChoiceRunning() {
        Date now = serviceContext.getCurrentTime();
        return poll.isAddChoiceRunning(now);
    }

    public String getCreatorName() {
        return poll.getCreator().getVotingId();
    }

    public String getVoteSizeMessage() {
        return _("pollen.common.voteNbVotes", getPoll().sizeVote());
    }

    public boolean isDescNull(Choice choice) {
        return StringUtils.isEmpty(choice.getDescription());
    }

    public boolean isFreePoll() {
        return poll.getPollType() == PollType.FREE;
    }

    public boolean isRestrictedPoll() {
        return poll.getPollType() == PollType.RESTRICTED;
    }

    public boolean isGroupPoll() {
        return poll.getPollType() == PollType.GROUP;
    }

    public boolean isTextType() {
        return poll.getChoiceType() == ChoiceType.TEXT;
    }

    public boolean isDateType() {
        return poll.getChoiceType() == ChoiceType.DATE;
    }

    public boolean isImageType() {
        return poll.getChoiceType() == ChoiceType.IMAGE;
    }

    public boolean isNormalVoteCounting() {
        return poll.getVoteCountingType() == VoteCountingType.NORMAL;
    }

    public boolean isPercentageVoteCounting() {
        return poll.getVoteCountingType() == VoteCountingType.PERCENTAGE;
    }

    public boolean isCondorcetVoteCounting() {
        return poll.getVoteCountingType() == VoteCountingType.CONDORCET;
    }

    public boolean isNumberVoteCounting() {
        return poll.getVoteCountingType() == VoteCountingType.NUMBER;
    }

    public boolean isPollStarted() {
        Date now = serviceContext.getCurrentTime();
        return poll.getBeginDate() == null || poll.getBeginDate().before(now);
    }

    public boolean isPollFinished() {
        Date now = serviceContext.getCurrentTime();
        return poll.getEndDate() != null && poll.getEndDate().before(now);
    }

    public String getCommentAuthor() {
        if (commentAuthor == null) {
            
            UserAccount user = getPollenUserAccount();
            if (user == null) {

                // Use current pollAccount name for comment
                commentAuthor = getPollAccount().getVotingId();

            } else {

                // Connecter user name
                commentAuthor = user.getDisplayName();
            }
        }
        return commentAuthor;
    }

    public void setCommentAuthor(String commentAuthor) {
        this.commentAuthor = commentAuthor;
    }

    public String prepareVotePage() throws Exception {

        loadPoll();

        // Current poll account
        loadPollAccount();

        // All votes
        // TODO no pagination for the moment, need to retrieve the correct page depends on current pollAccount
        votes = poll.getVote();

        voteAllowed = getVoteService().isVoteAllowed(poll, pollAccount);

        // Current vote
        if (voteAllowed) {
            vote = getVoteService().getVoteEditable(poll, pollAccount);
        }

        loadPollResults();
        loadPollComments();

        PollFeedService pollFeedService = newService(PollFeedService.class);
        feedFileExisting = pollFeedService.isFeedExists(poll);
        
        PollAccount pollCreator = poll.getCreator();
        if (getPollenUserAccount() == null) {

            // User is creator if loaded pollAccount is the creator
            creatorUser = pollCreator.equals(pollAccount);
            
        } else {

            // User is creator if the userAccount is the same
            creatorUser = getPollenUserAccount().equals(pollCreator.getUserAccount());
        }

        // Messages
        if (poll.isClosed()) {
            addFlashMessage(_("pollen.information.pollClosed"));
        } else if (!isPollStarted()) {
            addFlashMessage(_("pollen.information.pollNotStarted"));
        } else if (isPollFinished()) {
            addFlashMessage(_("pollen.information.pollFinished"));
        } else if (pollCreator.equals(pollAccount)) {
            addFlashMessage(_("pollen.information.vote.creatorUser"));
        }
        if (isPollChoiceRunning()) {
            addFlashMessage(_("pollen.information.pollChoiceRunning"));
        }

        if (log.isInfoEnabled()) {
            Date now = serviceContext.getCurrentTime();
            log.info("pollChoiceOrVoteStarted = " + isPollChoiceOrVoteStarted());
            log.info("pollChoiceRunning       = " + isPollChoiceRunning());
            log.info("pollRunning             = " + poll.isRunning(now));
            log.info("accountFieldDisplayed   = " + isAccountFieldDisplayed());
            log.info("creatorUser             = " + creatorUser);
        }
        return INPUT;
    }

    public String getResultValue(Choice choice) {

        String val = getPollResultsService().getResultValue(choice, results);
        return val;
    }

    /**
     * Retourne si le choix fait partie du vote (s'il a été renseigné par le
     * votant).
     *
     * @param choice le choix concerné
     * @return true si le choix est dans le vote
     */
    public boolean isChoiceInVote(VoteToChoice choice) {
        boolean result = false;
        if (choice != null) {
            switch (poll.getVoteCountingType()) {
                case NORMAL:
                    result = choice.getVoteValue() > 0;
                    break;
                case PERCENTAGE:
                    result = true;
                    break;
                case CONDORCET:
                    result = choice.getVoteValue() < 100;
                    break;
                case NUMBER:
                    result = choice.getVoteValue() >= 0;
            }
        }
        return result;
    }

    public boolean isModifAllowed(Vote vote) {
        return getVoteService().isUpdateAllowed(getPoll(), vote.getTopiaId(), getAccountId(), getPollenUserAccount());
    }

    public boolean isVoteAllowed() {
        return voteAllowed;
    }

    public boolean isDeleteCommentAllowed(Comment comment) {
        return getPollCommentService().isCanDeleteComment(comment, getPollAccount(), getPollenUserAccount());
    }
    
    public boolean isDeleteVoteAllowed(Vote vote) {
        return isUserAdmin();
    }

    public String escapeLineBreak(String text) {
        return text;
    }

    public Date getChoiceAsDate(Choice choice) {
        return new Date(Long.valueOf(choice.getName()));
    }

    protected void loadPollAccount() throws PollAccountNotFound {

        // Current poll account
        pollAccount = getPollService().getPollAccountEditable(getAccountId(), getPollenUserAccount(), poll);
    }

    protected void loadPoll() throws PollNotFoundException {

        // Ensure uri for poll and pollAccount loading
        preparePollUri(parameters);
        
        String pollId = getPollId();
        if (StringUtils.isNotEmpty(pollId)) {
            poll = getPollService().getPollByPollId(pollId);
        }
        Preconditions.checkNotNull(poll, "Can't load poll with id = [" + pollId + "]");
        
        log.debug("Poll TopiaId: " + poll.getTopiaId());
    }

    protected void loadPollComments() {

        PollCommentService service = newService(PollCommentService.class);

        comments = service.getAllComments(poll.getPollId());
    }

    protected void loadPollResults() throws PollNotFoundException {

        PollResultsService service = getPollResultsService();

        PollResultList pollResultList = service.getResults(poll.getPollId());

        results = pollResultList.getPollResults();

        if (log.isDebugEnabled()) {
            for (PollResult res : results) {
                log.debug(res.getName() + ": " + res.getValue()
                          + ", (voteCounting=" + res.getVoteCountingType()
                          + ", byGroup=" + res.isByGroup() + ")");
            }
        }
    }
}