/*
 * #%L
 * Pollen :: Services
 * $Id: VoteService.java 3719 2012-10-01 11:32:38Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.5.4/pollen-services/src/main/java/org/chorem/pollen/services/impl/VoteService.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.services.impl;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.PollenTechnicalException;
import org.chorem.pollen.business.persistence.Choice;
import org.chorem.pollen.business.persistence.PersonToList;
import org.chorem.pollen.business.persistence.PersonToListDAO;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.business.persistence.PollAccountDAO;
import org.chorem.pollen.business.persistence.PollType;
import org.chorem.pollen.business.persistence.UserAccount;
import org.chorem.pollen.business.persistence.Vote;
import org.chorem.pollen.business.persistence.VoteDAO;
import org.chorem.pollen.business.persistence.VoteToChoice;
import org.chorem.pollen.business.persistence.VoteToChoiceDAO;
import org.chorem.pollen.entities.PollenBinderHelper;
import org.chorem.pollen.services.PollenServiceSupport;
import org.chorem.pollen.services.exceptions.PollAccountNotFound;
import org.chorem.pollen.services.exceptions.VoteNotFoundException;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.persistence.pager.TopiaPagerBean;

import java.util.List;

public class VoteService extends PollenServiceSupport {

    /** Logger. */
    private static final Log log = LogFactory.getLog(VoteService.class);

    public Vote getVoteEditable(Poll poll, PollAccount accountEditable) {

        Preconditions.checkNotNull(poll);
        Preconditions.checkNotNull(accountEditable);

        VoteDAO dao = getDAO(Vote.class);
        Vote voteLoaded = null;
        if (StringUtils.isNotBlank(accountEditable.getAccountId())) {
            // if pollAccount has no accoutId, so not yet voted, no need to
            // seek for something is db.
            try {
                voteLoaded = dao.findByPollAndAccountId(
                        poll, accountEditable.getAccountId());
            } catch (TopiaException e) {
                throw new PollenTechnicalException(e);
            }
        }

        Vote result;
        if (voteLoaded != null) {

            result = newInstance(dao);
            PollenBinderHelper.simpleCopy(voteLoaded, result, true);

            // Attach the given accountEditable instead of the loaded one
            result.setPollAccount(accountEditable);

            // For an existing Vote, ensure that there is as many voteToChoice as poll choices
            VoteToChoiceDAO voteToChoiceDAO = getDAO(VoteToChoice.class);
            List<VoteToChoice> voteToChoices = Lists.newArrayList();

            // Add all VoteToChoice even they have empty value to match the size of Poll Choice List
            List<Choice> choices = poll.getChoice();
            for (int i = 0; i < choices.size(); i++) {

                Choice choice = choices.get(i);

                VoteToChoice voteToChoice = newInstance(voteToChoiceDAO);
                voteToChoice.setChoice(choice);

                // Retrieve existing value from loaded vote
                VoteToChoice voteToChoiceLoaded = voteLoaded.getChoiceVoteToChoice(choice);
                if (voteToChoiceLoaded != null) {
                    voteToChoice.setVoteValue(voteToChoiceLoaded.getVoteValue());
                }

                voteToChoices.add(i, voteToChoice);
            }
            result.setChoiceVoteToChoice(voteToChoices);

        } else {

            result = newInstance(dao);
            result.setPollAccount(accountEditable);
            result.setWeight(1.);

            VoteToChoiceDAO voteToChoiceDAO = getDAO(VoteToChoice.class);
            // Prepare the List of VoteToChoice with Poll's choices
            for (Choice choice : poll.getChoice()) {
                VoteToChoice element = newInstance(voteToChoiceDAO);
                element.setChoice(choice);
                result.addChoiceVoteToChoice(element);
            }
        }

        // Retrieve weight for Restricted Poll with existing account
        if (!poll.isPollFree() && accountEditable.getTopiaId() != null) {
            PersonToListDAO personToListDAO = getDAO(PersonToList.class);
            PersonToList personToList = personToListDAO.findByPollAndAccount(poll, accountEditable);
            result.setWeight(personToList.getWeight());
        }

        if (poll.isAnonymous()) {

            // force anonymous flag on vote
            result.setAnonymous(true);
        }
        return result;
    }

    public Vote createVote(Poll poll, Vote vote) throws PollAccountNotFound {

        VoteDAO voteDAO = getDAO(Vote.class);

        Vote result = create(voteDAO);
        result.setWeight(vote.getWeight());

        // make sure vote is anonymous if poll is
        result.setAnonymous(vote.isAnonymous() || poll.isAnonymous());

        // -- PollAccount -- //
        PollAccount pollAccount = vote.getPollAccount();
        String pollAccountId = pollAccount.getTopiaId();
        PollAccount pollAccountLoaded;

        if (pollAccountId == null) {

            // Create new account
            PollAccountDAO pollAccountDAO = getDAO(PollAccount.class);
            pollAccountLoaded = createWithProperties(
                    pollAccountDAO, PollAccount.PROPERTY_ACCOUNT_ID, serviceContext.generateId());
        } else {

            // Load existing account (restricted poll)
            pollAccountLoaded = getEntityById(PollAccount.class, pollAccountId);
        }

        // Update user data

        pollAccountLoaded.setVotingId(pollAccount.getVotingId());
        pollAccountLoaded.setEmail(pollAccount.getEmail());

        UserAccount userAccount = pollAccount.getUserAccount();
        if (userAccount != null) {
            UserAccount userAccountLoaded =
                    getEntityById(UserAccount.class, userAccount.getTopiaId());
            pollAccountLoaded.setUserAccount(userAccountLoaded);
        }
        result.setPollAccount(pollAccountLoaded);

        // -- List of VoteToChoice -- //
        for (VoteToChoice input : vote.getChoiceVoteToChoice()) {

            Integer value = input.getVoteValue();
            if (value != null) {
                createVoteToChoice(result, input);
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Entity created: " + result.getTopiaId());
        }

        Poll pollToUpdate = getEntityById(Poll.class, poll.getTopiaId());

        pollToUpdate.addVote(result);

        // Update hasVoted flag for RESTRICTED/GROUP poll
        if (!poll.isPollFree()) {

            PersonToList personToList = pollToUpdate.getPersonToListByVote(result);
            personToList.setHasVoted(true);
        }

        // feed notification

        PollFeedService pollFeedService = newService(PollFeedService.class);
        pollFeedService.onVoteAdded(pollToUpdate, result);

        commitTransaction("Could not create vote");

        // Send notification if necessary
        PreventRuleService preventRuleService = newService(PreventRuleService.class);
        preventRuleService.onVoteAdded(poll);

        return result;
    }

    public Vote updateVote(Poll poll, Vote vote) throws VoteNotFoundException {

        Vote result = getEntityById(Vote.class, vote.getTopiaId());

        if (result == null) {
            throw new VoteNotFoundException();
        }

        // -- PollAccount -- //
        PollAccount voteAccount = vote.getPollAccount();
        PollAccount pollAccountEntity = result.getPollAccount();
        pollAccountEntity.setVotingId(voteAccount.getVotingId());
        pollAccountEntity.setEmail(voteAccount.getEmail());
        pollAccountEntity.setUserAccount(voteAccount.getUserAccount());

        // make sure vote is anonymous if poll is
        result.setAnonymous(vote.isAnonymous() || poll.isAnonymous());

        VoteToChoiceDAO voteToChoiceDao = getDAO(VoteToChoice.class);

        // -- List of VoteToChoice -- //
        for (VoteToChoice input : vote.getChoiceVoteToChoice()) {

            Integer value = input.getVoteValue();

            VoteToChoice voteToChoiceEntity =
                    result.getChoiceVoteToChoice(input.getChoice());
            if (value != null) {

                if (voteToChoiceEntity != null) {
                    voteToChoiceEntity.setVoteValue(value);

                } else {
                    createVoteToChoice(result, input);
                }

            } else if (voteToChoiceEntity != null) {
                result.removeChoiceVoteToChoice(voteToChoiceEntity);
                delete(voteToChoiceDao, voteToChoiceEntity);
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Entity updated: " + result.getTopiaId());
        }

        // feed notification

        PollFeedService pollFeedService = newService(PollFeedService.class);
        pollFeedService.onVoteUpdated(poll, result);

        commitTransaction("Could not update vote");

        return result;
    }

    /**
     * Delete vote referenced by {@code voteId}. This will also delete all
     * associated {@link VoteToChoice}. The {@link PollAccount} could also be
     * deleted for a {@link PollType#FREE} poll where vote will be removed. This
     * remove implied an new execution of poll vote counting if continuous results
     * is activated.
     *
     * @param voteId TopiaId of the vote
     * @param reason reason to delete this vote
     * @throws VoteNotFoundException if vote doesn't exist
     */
    public void deleteVote(Poll poll,
                           String voteId,
                           String reason) throws VoteNotFoundException {

        Preconditions.checkNotNull(poll);
        Preconditions.checkNotNull(voteId);

        Vote entityToDelete = getEntityById(Vote.class, voteId);

        if (entityToDelete == null) {
            throw new VoteNotFoundException();
        }

        VoteDAO dao = getDAO(Vote.class);
        VoteToChoiceDAO voteToChoiceDao = getDAO(VoteToChoice.class);

        // Delete all VoteToChoice
        List<VoteToChoice> voteToChoices = ImmutableList.copyOf(entityToDelete.getChoiceVoteToChoice());
        for (VoteToChoice voteToChoice : voteToChoices) {
            entityToDelete.removeChoiceVoteToChoice(voteToChoice);
            delete(voteToChoiceDao, voteToChoice);
        }

        PollAccount voteAccount = entityToDelete.getPollAccount();

        if (poll.isPollFree()) {

            if (!voteAccount.equals(poll.getCreator())) {

                // Delete vote PollAccount if the Poll is free and account is not the creator
                PollAccountDAO accountDAO = getDAO(PollAccount.class);
                delete(accountDAO, voteAccount);
            }

        } else {

            // Update pollAccount hasVoted flag
            PersonToList personToList = poll.getPersonToListByVote(entityToDelete);
            personToList.setHasVoted(false);
        }

        poll.removeVote(entityToDelete);
        delete(dao, entityToDelete);

        if (log.isDebugEnabled()) {
            log.debug("Entity deleted: " + voteId);
        }

        // feed notification
        PollFeedService pollFeedService = newService(PollFeedService.class);
        pollFeedService.onVoteDeleted(poll, entityToDelete, voteAccount, reason);

        commitTransaction("Could not delete vote");

    }

    public List<Vote> getAllVotes(Poll poll) {
        try {
            VoteDAO dao = getDAO(Vote.class);
            List<Vote> results = dao.findAllVotes(poll);
            return results;

        } catch (TopiaException e) {
            throw new PollenTechnicalException("Could not obtain votes", e);
        }
    }

    public List<Vote> getVotesByPoll(Poll poll, TopiaPagerBean pager) {
        try {
            VoteDAO dao = getDAO(Vote.class);
            List<Vote> results = dao.findAllVotesByPoll(poll, pager);
            return results;

        } catch (TopiaException e) {
            throw new PollenTechnicalException("Could not obtain votes", e);
        }
    }

    public boolean isVotingIdFree(Poll poll,
                                  String pollAccountId,
                                  String votingId) {
        try {
            VoteDAO dao = getDAO(Vote.class);
            Vote existingVote = dao.findVoteByPollAndVotingId(poll, votingId);
            boolean result;
            if (existingVote == null) {
                // no such vote
                // votingId is free
                result = true;
            } else {

                // there is a vote using this votingId
                // votingId is free only if the pollAccount is the incoming one
                result = existingVote.getPollAccount().getTopiaId().equals(pollAccountId);
            }
            return result;
        } catch (Exception e) {
            throw new PollenTechnicalException(
                    "Unable test vote existing for account with votingId = " +
                    votingId + " and poll with uid = " + poll.getPollId(), e);
        }
    }

    protected VoteToChoice createVoteToChoice(Vote vote, VoteToChoice source) {

        VoteToChoiceDAO voteToChoiceDao = getDAO(VoteToChoice.class);
        VoteToChoice result = create(voteToChoiceDao);

        String choiceId = source.getChoice().getTopiaId();
        Choice choiceLoaded = getEntityById(Choice.class, choiceId);
        result.setChoice(choiceLoaded);

        result.setVote(vote);
        result.setVoteValue(source.getVoteValue());

        vote.addChoiceVoteToChoice(result);

        return result;
    }

}
