/*
 * #%L
 * Pollen :: Services
 * $Id: FavoriteService.java 3691 2012-09-14 16:33:06Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.5.3/pollen-services/src/main/java/org/chorem/pollen/services/impl/FavoriteService.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.services.impl;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.PollenTechnicalException;
import org.chorem.pollen.business.persistence.PersonList;
import org.chorem.pollen.business.persistence.PersonListDAO;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.business.persistence.PollAccountDAO;
import org.chorem.pollen.business.persistence.UserAccount;
import org.chorem.pollen.business.persistence.UserAccountDAO;
import org.chorem.pollen.services.PollenServiceSupport;
import org.chorem.pollen.services.exceptions.FavoriteListAlreadyExistException;
import org.chorem.pollen.services.exceptions.FavoriteListImportException;
import org.chorem.pollen.services.exceptions.FavoriteListNotFoundException;
import org.chorem.pollen.services.exceptions.FavoriteListNotOwnedByUserException;
import org.chorem.pollen.services.exceptions.ParticipantAlreadyFoundInListException;
import org.chorem.pollen.services.exceptions.ParticipantNotFoundException;
import org.chorem.pollen.services.exceptions.ParticipantNotFoundInListException;
import org.chorem.pollen.services.exceptions.UserNotFoundException;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.persistence.pager.TopiaPagerBean;
import org.nuiton.util.StringUtil;

import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.DirContext;
import javax.naming.directory.InitialDirContext;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.util.List;
import java.util.Locale;
import java.util.Properties;

import static org.nuiton.i18n.I18n.l_;

public class FavoriteService extends PollenServiceSupport {

    /** Logger. */
    private static final Log log = LogFactory.getLog(FavoriteService.class);

    public List<PersonList> getFavoriteLists(UserAccount user,
                                             TopiaPagerBean pager) {

        Preconditions.checkNotNull(user);
        Preconditions.checkNotNull(pager);

        try {
            PersonListDAO dao = getDAO(PersonList.class);
            List<PersonList> result = dao.findFavoriteLists(user, pager);
            return result;
        } catch (TopiaException e) {
            throw new PollenTechnicalException(e);
        }
    }

    public List<PersonList> getFavoriteLists(UserAccount user) {

        Preconditions.checkNotNull(user);

        try {
            PersonListDAO dao = getDAO(PersonList.class);
            List<PersonList> result = dao.findAllFavoriteLists(user);
            return result;
        } catch (TopiaException e) {
            throw new PollenTechnicalException(e);
        }
    }

    public PersonList getFavoriteList(UserAccount user,
                                      String favoriteListId) throws FavoriteListNotFoundException, FavoriteListNotOwnedByUserException {

        Preconditions.checkNotNull(user);
        Preconditions.checkNotNull(favoriteListId);

        try {
            PersonListDAO dao = getDAO(PersonList.class);

            PersonList favoriteList = dao.findByTopiaId(favoriteListId);

            if (favoriteList == null) {
                throw new FavoriteListNotFoundException();
            }

            // check this favorite belongs to the given user
            UserAccount favoriteListOwner = favoriteList.getOwner();
            if (!favoriteListOwner.equals(user)) {
                throw new FavoriteListNotOwnedByUserException();
            }

            return favoriteList;

        } catch (TopiaException e) {
            throw new PollenTechnicalException(e);
        }
    }

    public List<PollAccount> getFavoriteListUsers(PersonList favoriteList,
                                                  TopiaPagerBean pager) {

        Preconditions.checkNotNull(favoriteList);
        Preconditions.checkNotNull(pager);

        try {

            PollAccountDAO dao = getDAO(PollAccount.class);
            List<PollAccount> result = dao.findFavoriteListUsers(favoriteList, pager);
            return result;

        } catch (TopiaException e) {
            throw new PollenTechnicalException(e);
        }
    }

    public PersonList createFavoriteList(UserAccount user,
                                         String name)
            throws FavoriteListAlreadyExistException, UserNotFoundException {

        Preconditions.checkNotNull(user);
        Preconditions.checkNotNull(name);

        // can't accept favorite list without name
        Preconditions.checkArgument(StringUtils.isNotEmpty(name));

        UserAccountDAO userDAO = getDAO(UserAccount.class);
        PersonListDAO dao = getDAO(PersonList.class);

        PersonList result;
        try {

            UserAccount userToUse =
                    userDAO.findByTopiaId(user.getTopiaId());

            if (userToUse == null) {
                throw new UserNotFoundException();
            }

            // check list does not already exists

            boolean exist = dao.isPersonListExist(user, name);

            if (exist) {

                throw new FavoriteListAlreadyExistException();
            }

            result = dao.create(
                    PersonList.PROPERTY_OWNER, userToUse
            );
            result.setName(name);
        } catch (TopiaException e) {
            throw new PollenTechnicalException(e);
        }
        return result;
    }

    public PersonList deleteFavoriteList(UserAccount user,
                                         PersonList favoriteList) throws FavoriteListNotFoundException, FavoriteListNotOwnedByUserException, UserNotFoundException {

        Preconditions.checkNotNull(user);
        Preconditions.checkNotNull(favoriteList);

        UserAccountDAO userDAO = getDAO(UserAccount.class);
        PersonListDAO dao = getDAO(PersonList.class);

        try {

            UserAccount userToUse =
                    userDAO.findByTopiaId(user.getTopiaId());

            if (userToUse == null) {
                throw new UserNotFoundException();
            }

            PersonList favoriteListToDelete =
                    dao.findByTopiaId(favoriteList.getTopiaId());

            if (favoriteListToDelete == null) {
                throw new FavoriteListNotFoundException();
            }

            // check this favorite belongs to the given user
            UserAccount favoriteListOwner = favoriteListToDelete.getOwner();
            if (!favoriteListOwner.equals(user)) {
                throw new FavoriteListNotOwnedByUserException();
            }

            dao.delete(favoriteListToDelete);

            commitTransaction("Could not delete favorite list");
            return favoriteListToDelete;

        } catch (TopiaException e) {
            throw new PollenTechnicalException(e);
        }
    }

    public void addPollAccountToFavoriteList(PersonList favoriteList,
                                             PollAccount pollAccount) throws ParticipantAlreadyFoundInListException, FavoriteListNotFoundException {

        Preconditions.checkNotNull(favoriteList);
        Preconditions.checkNotNull(pollAccount);

        // can't accept poll account without name
        Preconditions.checkArgument(
                StringUtils.isNotEmpty(pollAccount.getVotingId()));

        // can't accept poll account without email
        Preconditions.checkArgument(
                StringUtils.isNotEmpty(pollAccount.getEmail()));

        PollAccountDAO dao = getDAO(PollAccount.class);
        PersonListDAO personListDAO = getDAO(PersonList.class);

        try {

            PersonList personListToUpdate =
                    personListDAO.findByTopiaId(favoriteList.getTopiaId());

            if (personListToUpdate == null) {
                throw new FavoriteListNotFoundException();
            }

            // check there is other poll account in this list with same email
            boolean pollAccountExists = dao.isPollAccountExists(personListToUpdate, pollAccount);
            if (pollAccountExists) {
                throw new ParticipantAlreadyFoundInListException();
            }

            PollAccount pollAccountCreated =
                    dao.create(PollAccount.PROPERTY_PERSON_LIST,
                               personListToUpdate);

            pollAccountCreated.setVotingId(pollAccount.getVotingId());
            pollAccountCreated.setEmail(pollAccount.getEmail());
            personListToUpdate.addPollAccount(pollAccountCreated);

            commitTransaction("Could not add poll account to favorite list");
        } catch (TopiaException e) {
            throw new PollenTechnicalException(e);
        }
    }

    public void editPollAccountToFavoriteList(PersonList favoriteList,
                                              PollAccount pollAccount) throws ParticipantAlreadyFoundInListException, FavoriteListNotFoundException, ParticipantNotFoundException {

        Preconditions.checkNotNull(favoriteList);
        Preconditions.checkNotNull(pollAccount);

        // can't accept poll account without name
        Preconditions.checkArgument(
                StringUtils.isNotEmpty(pollAccount.getVotingId()));

        // can't accept poll account without email
        Preconditions.checkArgument(
                StringUtils.isNotEmpty(pollAccount.getEmail()));

        PollAccountDAO dao = getDAO(PollAccount.class);
        PersonListDAO personListDAO = getDAO(PersonList.class);

        try {

            PersonList personListToUpdate =
                    personListDAO.findByTopiaId(favoriteList.getTopiaId());

            if (personListToUpdate == null) {
                throw new FavoriteListNotFoundException();
            }

            PollAccount pollAccountToUpdate =
                    dao.findByTopiaId(pollAccount.getTopiaId());

            if (pollAccountToUpdate == null) {
                throw new ParticipantNotFoundException();
            }

            // check there is another poll account in this list with same id
            boolean pollAccountExists = dao.isPollAccountExists(personListToUpdate, pollAccount);
            if (pollAccountExists) {
                throw new ParticipantAlreadyFoundInListException();
            }

            pollAccountToUpdate.setVotingId(pollAccount.getVotingId());
            pollAccountToUpdate.setEmail(pollAccount.getEmail());
            commitTransaction("Could not edit poll account favorite list");

        } catch (TopiaException e) {
            throw new PollenTechnicalException(e);
        }
    }

    public void removePollAccountToFavoriteList(PersonList favoriteList,
                                                PollAccount pollAccount) throws ParticipantNotFoundException, FavoriteListNotFoundException, ParticipantNotFoundInListException {

        Preconditions.checkNotNull(favoriteList);
        Preconditions.checkNotNull(pollAccount);

        PollAccountDAO dao = getDAO(PollAccount.class);
        PersonListDAO personListDAO = getDAO(PersonList.class);

        try {

            PersonList personListToUpdate =
                    personListDAO.findByTopiaId(favoriteList.getTopiaId());

            if (personListToUpdate == null) {
                throw new FavoriteListNotFoundException();
            }

            PollAccount pollAccountToRemove =
                    dao.findByTopiaId(pollAccount.getTopiaId());

            if (pollAccountToRemove == null) {

                throw new ParticipantNotFoundException();
            }

            PollAccount pollAccountInList =
                    personListToUpdate.getPollAccountByTopiaId(pollAccount.getTopiaId());

            if (pollAccountInList == null) {

                throw new ParticipantNotFoundInListException();
            }

            personListToUpdate.removePollAccount(pollAccountToRemove);

            commitTransaction("Could not remove pollaccount to favorite list");
        } catch (TopiaException e) {
            throw new PollenTechnicalException(e);
        }
    }

    public List<PollAccount> importFromCsvfile(String filename,
                                               File file) throws FavoriteListImportException {

        List<PollAccount> results = Lists.newArrayList();
        Locale locale = getLocale();
        PollAccountDAO dao = getDAO(PollAccount.class);
        BufferedReader reader = null;
        try {
            reader = new BufferedReader(new FileReader(file));
            String line;
            int lineNumber = 0;
            while ((line = reader.readLine()) != null) {
                line = line.trim();
                if (StringUtils.isBlank(line) || line.startsWith("#")) {

                    // comment line
                    continue;
                }
                lineNumber++;

                int spaceIndex = line.indexOf(' ');
                String email;
                String votingId;
                if (spaceIndex == -1) {
                    // only email
                    email = line;
                    votingId = line;
                } else {
                    // email + votingId
                    email = line.substring(0, spaceIndex);
                    votingId = line.substring(spaceIndex);
                }
                email = email.trim();
                votingId = votingId.trim();

                if (!StringUtil.isEmail(email)) {

                    // email is not valid
                    String error = l_(locale, "pollen.error.import.invalid.email", lineNumber, email);
                    throw new FavoriteListImportException(filename, error, null);
                }

                PollAccount account = newInstance(dao);
                account.setEmail(email);
                account.setVotingId(votingId);
                results.add(account);
            }

            if (log.isInfoEnabled()) {
                log.info(results.size() + " comptes importés.");
            }
            reader.close();

        } catch (FileNotFoundException ex) {
            // should never happens ?
            throw new PollenTechnicalException(ex);
        } catch (IOException e) {
            if (log.isErrorEnabled()) {
                log.error("Could not close reader", e);
            }
        } finally {
            IOUtils.closeQuietly(reader);
        }
        return results;
    }

    public List<PollAccount> importFromLDAP(String url) throws FavoriteListImportException {

        long start = System.nanoTime();

        List<PollAccount> results = Lists.newArrayList();
        try {

            // Initialisation du contexte
            Properties env = new Properties();
//            if (server != null) {
//                env.put(Context.INITIAL_CONTEXT_FACTORY,
//                        "com.sun.jndi.ldap.LdapCtxFactory");
//                env.put(Context.PROVIDER_URL, "ldap://" + server + "/");
//            }
            DirContext ictx = new InitialDirContext(env);

            // Recherche en profondeur
            SearchControls control = new SearchControls();
            control.setSearchScope(SearchControls.SUBTREE_SCOPE);

            // Création des comptes avec les résultats de la recherche
            NamingEnumeration<SearchResult> e = ictx.search(url, null,
                                                            control);
            while (e.hasMore()) {
                SearchResult r = e.next();

                if (log.isDebugEnabled()) {
                    log.debug("Result: " + r.getName() + "(object: "
                              + r.getClassName() + ")");
                }

                Attribute nameAttr = r.getAttributes().get("cn");
                Attribute emailAttr = r.getAttributes().get("mail");

                if (nameAttr != null) {
                    PollAccountDAO dao = getDAO(PollAccount.class);
                    PollAccount account = newInstance(dao);
                    account.setVotingId(nameAttr.get().toString());
                    account.setEmail(emailAttr.get().toString());
                    results.add(account);

                    if (log.isDebugEnabled()) {
                        log.debug("New account - name: "
                                  + nameAttr.get().toString() + ", email: "
                                  + emailAttr.get().toString());
                    }
                }
            }
        } catch (NamingException ex) {
            log.error("Exception de nommage lors de l'import depuis LDAP", ex);
            throw new FavoriteListImportException("LDAP", ex);
        }

        long duration = (System.nanoTime() - start) / 1000000000;
        if (log.isInfoEnabled()) {
            log.info(results.size() + " comptes importés en " + duration
                     + " sec.");
        }

        return results;
    }

    public PersonList newFavoriteList() {
        PersonListDAO dao = getDAO(PersonList.class);
        PersonList personList = newInstance(dao);
        return personList;
    }

    public PollAccount newPollAccountForFavoriteList() {
        PollAccountDAO dao = getDAO(PollAccount.class);
        PollAccount pollAccount = newInstance(dao);
        return pollAccount;
    }
}
