/*
 * #%L
 * Pollen :: Services
 * $Id: PreventRuleService.java 3761 2012-12-01 12:34:54Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.5.1/pollen-services/src/main/java/org/chorem/pollen/services/impl/PreventRuleService.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.services.impl;

import com.google.common.base.Preconditions;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.PreventRule;
import org.chorem.pollen.services.PollenNotifierWorker;
import org.chorem.pollen.services.PollenServiceSupport;

import java.util.Date;

/**
 * Manage prevent rules of a poll.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.3
 */
public class PreventRuleService extends PollenServiceSupport {

    public static final String SCOPE_VOTE = "vote";

    public static final String SCOPE_REMINDER = "rappel";

    /** Logger. */
    private static final Log log = LogFactory.getLog(PreventRuleService.class);

    public PreventRule createAddVotePreventRule() {
        PreventRule rule = newInstance(getDAO(PreventRule.class));
        rule.setScope(SCOPE_VOTE);
        rule.setSensibility(1);
        rule.setRepeated(true);
        return rule;
    }

    public PreventRule createRemindPreventRule(int sensibility) {
        PreventRule rule = newInstance(getDAO(PreventRule.class));
        rule.setScope(SCOPE_REMINDER);
        rule.setSensibility(sensibility);
        rule.setRepeated(false);
        return rule;
    }

    public void onVoteAdded(Poll poll) {

        if (poll.getEndDate() != null) {

            for (PreventRule rule : poll.getPreventRule()) {

                // send to creator a email to say hey a new vote was added!

                if (canExecuteRule(SCOPE_VOTE, rule, 1)) {

                    // ok can execute this task

                    EmailService emailService = newService(EmailService.class);

                    // let's send an email
                    emailService.onVoteAdded(poll);

                    // and update prevent rule
                    commitTransaction("Could not update prevent rule");
                }
            }
        }
    }

    public void onPollToRemind(Poll poll, Date now) {

        Preconditions.checkNotNull(poll, "Poll can notbe nul.");

        Date endDate = poll.getEndDate();
        Preconditions.checkNotNull(
                endDate, "End date can not be null to remind");

        // envoi d'un email si endDate-nowDate <= preventRuleSensibility

        // timeValue = endDate-nowDate
        long timeValue = endDate.getTime() - now.getTime();

        int hourBeforeEnd = (int) (timeValue / PollenNotifierWorker.ONE_HOUR);

        if (log.isDebugEnabled()) {
            log.debug("Now: " + now.getTime() + " End: "
                      + endDate.getTime() + " timeValue: "
                      + timeValue + "(" + hourBeforeEnd + "h)");
        }

        for (PreventRule rule : poll.getPreventRule()) {

            if (canExecuteRule(SCOPE_REMINDER, rule, hourBeforeEnd)) {

                // ok can execute this task

                if (log.isInfoEnabled()) {
                    log.info("Execute reminder rule on poll " + poll.getTitle());
                }
                EmailService emailService = newService(EmailService.class);

                // let's send an email
                emailService.onPollReminder(poll);

                // this rule must be desactivate now
                rule.setActive(false);

                // and update prevent rule
                commitTransaction("Could not update prevent rule");
            }
        }
    }

    protected boolean canExecuteRule(String scope, PreventRule preventRule,
                                     Integer value) {

        Preconditions.checkNotNull(scope);
        Preconditions.checkNotNull(value);

        boolean result = false;

        if (preventRule.isActive()) {

            // alive rule

            if (scope.equals(preventRule.getScope())) {

                // rule in good scope

                int sensibility = preventRule.getSensibility();

                if (preventRule.isRepeated() && sensibility != 0) {
                    value = value % sensibility + sensibility;
                }

                if (value == sensibility) {

                    // ok good value, can execute this rule
                    result = true;
                }

            }
        }

        return result;
    }

}
