/*
 * #%L
 * Pollen :: Services
 * 
 * $Id: PollResultsService.java 3395 2012-05-28 15:34:46Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.3.1/pollen-services/src/main/java/org/chorem/pollen/services/impl/PollResultsService.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.services.impl;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.PollenFunctions;
import org.chorem.pollen.bean.PollResult;
import org.chorem.pollen.bean.PollResultList;
import org.chorem.pollen.business.persistence.Choice;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.Result;
import org.chorem.pollen.business.persistence.ResultDAO;
import org.chorem.pollen.common.ChoiceType;
import org.chorem.pollen.common.PollType;
import org.chorem.pollen.common.VoteCountingType;
import org.chorem.pollen.services.PollenServiceSupport;
import org.chorem.pollen.services.exceptions.PollNotFoundException;
import org.chorem.pollen.votecounting.business.NumberMethod;
import org.chorem.pollen.votecounting.dto.ChoiceDTO;
import org.chorem.pollen.votecounting.dto.PollDTO;
import org.chorem.pollen.votecounting.dto.PollExportDTO;
import org.chorem.pollen.votecounting.dto.VoteCountingResultDTO;
import org.chorem.pollen.votecounting.services.ImportExportService;
import org.chorem.pollen.votecounting.services.VoteCountingService;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;

import static org.nuiton.i18n.I18n.l_;

/**
 * Deals with poll results.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.3
 */
public class PollResultsService extends PollenServiceSupport {

    /** log. */
    private static final Log log = LogFactory.getLog(PollResultsService.class);

    public PollResultList getResults(String pollId) throws PollNotFoundException {

        PollService pollService = newService(PollService.class);
        Poll poll = pollService.getPollByPollId(pollId);

        if (poll == null) {
            throw new PollNotFoundException();
        }
        PollResultList result;
        if (poll.getPollType() == PollType.GROUP) {
            result = generateResults(poll, null, true, true);
        } else {
            result = generateResults(poll, null, true, false);
        }
        return result;
    }

    /**
     * Génère les résultats d'un sondage (= un dépouillement).
     *
     * @param poll         le sondage
     * @param voteCounting type de dépouillement
     * @param byGroup      résultats par groupe
     * @param groupOnly    résultats uniquement par groupe
     * @return les résultats du sondage
     */
    private PollResultList generateResults(Poll poll,
                                           VoteCountingType voteCounting,
                                           boolean byGroup,
                                           boolean groupOnly) {


        if (log.isDebugEnabled()) {
            log.debug(poll.getPollId() + " (" + voteCounting
                      + ") has results: " + hasResults(poll, voteCounting));
        }

        // Dépouillement du sondage.
        // S'il n'existe pas de résultats ou si le sondage est encore ouvert.
        //if (!hasResults(ePoll, voteCounting) || !ePoll.getIsClosed()) {
        PollDTO dto = PollenFunctions.POLL_TO_BEAN.apply(poll);

        if (voteCounting != null) {
            dto.setVoteCounting(voteCounting);
        }

        // clear result (they will be regenerated ! (a big shame))
        poll.clearResult();

        VoteCountingService service = new VoteCountingService();

        VoteCountingResultDTO result = service.execute(dto);

        ResultDAO daoResult = getDAO(Result.class);

        for (ChoiceDTO choice : result.getChoices()) {

            Result eResult = create(daoResult);

            Choice eChoice = poll.getChoiceByTopiaId(choice.getIdChoice());

            poll.addResult(eResult);

            eResult.setName(eChoice.getName());
            eResult.setByGroup(result.isByGroup());
            eResult.setPoll(poll);
            eResult.setResultValue(String.valueOf(choice.getValue()));
            eResult.setVoteCountingType(result.getTypeVoteCounting());
        }

        commitTransaction("Could not generate poll result");

        // Conversion et trie des résultats
        List<PollResult> list = createPollResults(poll);

        VoteCountingType voteCountingType = poll.getVoteCountingType();

        Iterator<PollResult> it = list.iterator();
        while (it.hasNext()) {
            PollResult currentResult = it.next();

            // Cas d'un dépouillement particulier
            // Suppression des resultats qui ne sont pas de se dépouillement
            if (voteCounting != null
                && currentResult.getVoteCountingType() != voteCounting) {
                it.remove();
            }

            // Cas d'un sondage de type GROUP avec filtre
            if (byGroup) {

                // Filtre group : Suppression resultats non group
                if (groupOnly && !currentResult.isByGroup()) {
                    it.remove();
                }

                // Filtre non group : Suppression resultats group
                else if (!groupOnly && currentResult.isByGroup()) {
                    it.remove();
                }

                // Suppression des autres resultats de depouillements differents
                if (currentResult.getVoteCountingType() != voteCountingType) {
                    it.remove();
                }
            }
        }

        PollResultList resultList = new PollResultList();
        resultList.setPollResults(list);
        resultList.setVoteCountingResult(result);
        return resultList;
    }

    public String exportPolltoXml(String pollId) throws PollNotFoundException {

        // Recherche du sondage

        PollService pollService = newService(PollService.class);

        Poll poll = pollService.getPollByPollId(pollId);

        if (poll == null) {
            throw new PollNotFoundException();
        }

        PollDTO dto = PollenFunctions.POLL_TO_BEAN.apply(poll);

        VoteCountingService service = new VoteCountingService();

        List<VoteCountingResultDTO> results = Lists.newArrayList();

        VoteCountingResultDTO result = service.executeVoteCounting(dto);

        results.add(result);

        if (poll.getPollType() == PollType.GROUP) {
            VoteCountingResultDTO groupResult =
                    service.executeGroupCounting(dto);
            results.add(groupResult);
        }

        // Transformation du sondage
        PollExportDTO pollExport = new PollExportDTO();
        pollExport.setPollId(dto.getPollId());
        pollExport.setPoll(dto);
        pollExport.setVoteCountingResults(results);

        // Export du sondage
        ImportExportService serviceExport = new ImportExportService();
        String content = serviceExport.exportToXml(pollExport);

        return content;
    }

    /**
     * Retourne la liste des résultats d'un sondage sous forme de DTOs.
     *
     * @param poll le sondage (entitée Poll)
     * @return la liste des DTO Result
     */
    protected List<PollResult> createPollResults(Poll poll) {
        List<PollResult> results = Lists.newArrayList();
        for (Result res : poll.getResult()) {
            PollResult dto = new PollResult();
            dto.setId(res.getTopiaId());
            dto.setPollId(poll.getPollId());
            String resName = res.getName();
            dto.setName(resName);
            dto.setHidden(resName != null &&
                          resName.startsWith(NumberMethod.HIDDEN_PREFIX));
            dto.setValue(res.getResultValue());
            dto.setByGroup(res.isByGroup());
            dto.setChoiceType(poll.getChoiceType());
            dto.setVoteCountingType(poll.getVoteCountingType());
            results.add(dto);
        }
        return results;
    }

    public String getResultValue(Choice choice,
                                 Collection<PollResult> results) {

        String val = "";
        for (PollResult result : results) {
            if (result.getName().equals(choice.getName())) {
                val = removeTrailing0(result.getValue());
                break;
            }
        }
        return val;
    }

    public List<PollResult> createNumberVoteCountingResult(
            PollResultList resultListDTO,
            Multimap<String, String> choicesResults,
            Multimap<String, String> subtitles) {
        Preconditions.checkNotNull(resultListDTO);
        Preconditions.checkNotNull(choicesResults);
        Preconditions.checkNotNull(subtitles);

        List<PollResult> results = resultListDTO.getPollResults();

        /*
        *  liste des vrais choix (les choix pas cachés).
        *  cette variable sera affectée à la variable results
        */
        List<PollResult> results2 = Lists.newArrayList();

        // Ajout des résultats des choix cachés
        for (PollResult result : results) {
            if (result.isHidden()) {
                String name = result.getName();
                int indexOf = name.indexOf('#');
                String choice = name.substring(
                        NumberMethod.HIDDEN_PREFIX.length(), indexOf);

                String votingId = name.substring(indexOf + 1);
                String value = result.getValue();

                choicesResults.put(choice, votingId);
                choicesResults.put(choice, value);

            } else {
                results2.add(result);
            }
        }

        Locale locale = getLocale();

        // Récupération du choix correspondant au résultat
        VoteCountingResultDTO voteCountingResult = resultListDTO
                .getVoteCountingResult();

        for (String choiceName : choicesResults.keySet()) {
            Collection<String> values = choicesResults.get(choiceName);

            for (ChoiceDTO choice : voteCountingResult.getChoices()) {
                if (choiceName.equals(choice.getName())) {
                    subtitles.put(
                            choiceName,
                            l_(locale, "pollen.common.numberVote-total"));
                    subtitles.put(
                            choiceName,
                            String.valueOf(choice.getValue()));

                    subtitles.put(
                            choiceName,
                            l_(locale, "pollen.common.numberVote-average"));
                    subtitles.put(
                            choiceName,
                            String.valueOf(choice.getAverage()));

                    subtitles.put(
                            choiceName,
                            l_(locale, "pollen.common.numberVote-blank-votes"));
                    subtitles.put(
                            choiceName,
                            String.valueOf(choice.getNbBlankVotes()));

                    subtitles.put(
                            choiceName,
                            l_(locale, "pollen.common.numberVote-total-votes"));
                    subtitles.put(
                            choiceName,
                            String.valueOf(values.size() / 2));
                    break;
                }
            }
        }

        if (results2.size() == 1) {
            results2.remove(0);
        }
        return results2;
    }

    public List<PollResult> getTopRanking(List<PollResult> results) {
        List<PollResult> ranking = Lists.newArrayList(results);

        Collections.sort(ranking, new Comparator<PollResult>() {
            @Override
            public int compare(PollResult o1, PollResult o2) {
                Double result1 = Double.parseDouble(o1.getValue());
                Double result2 = Double.parseDouble(o2.getValue());
                int comp = 0; // résultat de la comparaison

                if (result1 > result2) {
                    comp = -1;
                } else if (result1 < result2) {
                    comp = 1;
                }
                if (comp == 0) {
                    comp = o1.getName().compareTo(o2.getName());
                }
                return comp;
            }
        });

        List<PollResult> topRanking = Lists.newArrayList();

        if (CollectionUtils.isNotEmpty(ranking)) {
            String winValue = ranking.get(0).getValue();
            for (PollResult r : ranking) {
                if (winValue.equals(r.getValue())) {
                    topRanking.add(r);
                }
            }
        }
        return topRanking;
    }

    /**
     * Supprime le 0 final d'un nombre à virgule. Le résultat peut-être un
     * double : 1,0 -> 1 et 1,2 -> 1,2.
     *
     * @param val le nombre
     * @return le nombre sans 0 final
     */
    protected String removeTrailing0(String val) {
        if (val.endsWith(".0")) {
            val = val.substring(0, val.indexOf('.'));
        }
        return val;
    }

//    /**
//     * Création des résultats de sondage à partir d'un dto de résultats de
//     * sondage.
//     *
//     * @param ePoll poll à remplir
//     * @param dto   le dto des résultats de sondage.
//     */
//    protected void populateResultEntities(Poll ePoll,
//                                          VoteCountingResultDTO dto) {
//        for (ChoiceDTO choice : dto.getChoices()) {
//
//            ResultDAO daoResult = getDAO(Result.class);
//            Result eResult = create(daoResult);
//
//            Choice eChoice = ePoll.getChoiceByTopiaId(choice.getIdChoice());
//
//            ePoll.addResult(eResult);
//
//            eResult.setName(eChoice.getName());
//            eResult.setByGroup(dto.isByGroup());
//            eResult.setPoll(ePoll);
//            eResult.setResultValue(String.valueOf(choice.getValue()));
//            eResult.setVoteCountingType(dto.getTypeVoteCounting());
//        }
//    }

    /**
     * Retourne vrai si le sondage a des résultats.
     *
     * @param poll le sondage
     * @param type type de dépouillement
     * @return vrai si le sondage a des résultats
     */
    protected boolean hasResults(Poll poll, VoteCountingType type) {
        boolean hasresult = false;

        if (!poll.isResultEmpty()) {

            if (type == null) {
                hasresult = true;
            } else {
                for (Result result : poll.getResult()) {
                    if (type.equals(result.getVoteCountingType())) {
                        hasresult = true;
                        break;
                    }
                }
            }
        }
        return hasresult;
    }

//    public String importPoll(String filePath) {
//        return importPoll(filePath, null);
//    }
//
//    public String importPoll(String filePath, UserAccount user) {
//        String topiaId = "";
//
//        // Import du sondage
//        ServiceExport serviceExport = new ServiceExportImpl();
//        PollExportDTO pollExportDTO = serviceExport.executeImport(filePath);
//
//        // Transformation du sondage
//        // FIXME do not call a Service from an other one
//        PollService servicePoll = newService(PollService.class);
//        org.chorem.pollen.business.dto.PollDTO poll = DataVoteCountingConverter
//                .createPollDTO(pollExportDTO.getPoll());
//
//        // Enregistrement du sondage
//        if (user != null) {
//            poll.setUserId(user.getId());
//        }
//        topiaId = servicePoll.createPoll(poll);
//
//        if (log.isInfoEnabled()) {
//            log.info("Poll imported: " + topiaId);
//        }
//
//        return topiaId;
//    }

    /**
     * Retourne une chaîne contenant les résultats du sondage.
     *
     * @param poll    le sondage
     * @param results les résultats du sondage
     * @return les résultats sous forme de chaine de caractères
     */
    public String getResultsAsString(Poll poll, Collection<Result> results) {

        DateFormat dateFormat = new SimpleDateFormat();
        StringBuilder res = new StringBuilder("");
        Iterator<Result> it = results.iterator();
        while (it.hasNext()) {
            Result result = it.next();
            if (poll.getChoiceType() == ChoiceType.DATE) {
                Date date = new Date(Long.parseLong(result.getName()));
                res.append(dateFormat.format(date));
            } else {
                res.append(result.getName());
            }
            res.append("=").append(removeTrailing0(result.getResultValue()));
            if (it.hasNext()) {
                res.append(", ");
            }
        }
        return res.toString();
    }
}
