/*
 * #%L
 * Pollen :: Persistence
 * $Id: PollenConfiguration.java 3698 2012-09-23 13:01:38Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.5.5/pollen-persistence/src/main/java/org/chorem/pollen/PollenConfiguration.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.builder.ReflectionToStringBuilder;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.business.persistence.ChoiceType;
import org.chorem.pollen.business.persistence.PollCommentVisibility;
import org.chorem.pollen.business.persistence.PollType;
import org.chorem.pollen.business.persistence.PollVoteVisibility;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.util.ArgumentsParserException;
import org.nuiton.util.FileUtil;
import org.nuiton.util.Version;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.Collections;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.List;
import java.util.Properties;

/**
 * Pollen configuration.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.3
 */
public class PollenConfiguration {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(PollenConfiguration.class);

    /** Delegate application config object containing configuration. */
    private ApplicationConfig applicationConfig;

    public PollenConfiguration() {
        this("pollen.properties", null);
    }

    public PollenConfiguration(String file,
                               Properties defaultOptions) {

        if (log.isInfoEnabled()) {
            log.info(this + " is initializing...");
        }
        loadApplicationConfig(file, defaultOptions);

        // application url is a mandatory property
        String applicationUrl =
                applicationConfig.getOption(
                        PollenConfigurationOption.APPLICATION_URL.key);
        if (StringUtils.isBlank(applicationUrl)) {
            throw new IllegalStateException(
                    "No application url defined in configuration, please fill the property '" +
                    PollenConfigurationOption.APPLICATION_URL.key + "'");
        }

        createDirectory(PollenConfigurationOption.EMAIL_DIR);
        createDirectory(PollenConfigurationOption.FEED_DIR);
        createDirectory(PollenConfigurationOption.IMG_DIR);
        createDirectory(PollenConfigurationOption.TMP_DIR);

        if (log.isDebugEnabled()) {
            log.debug("parsed options in config file" +
                      applicationConfig.getOptions());
        }
    }

    public URL getApplicationUrl() {
        URL result = applicationConfig.getOptionAsURL(
                PollenConfigurationOption.APPLICATION_URL.key
        );
        return result;
    }

    public ChoiceType getDefaultChoiceType() {
        ChoiceType result = applicationConfig.getOption(
                ChoiceType.class,
                PollenConfigurationOption.DEFAULT_CHOICE_TYPE.key);
        return result;
    }

    public int getDefaultVoteCountingType() {
        int result = applicationConfig.getOptionAsInt(
                PollenConfigurationOption.DEFAULT_VOTE_COUNTING_TYPE.key);
        return result;
    }

    public PollType getDefaultPollType() {
        PollType result = applicationConfig.getOption(
                PollType.class,
                PollenConfigurationOption.DEFAULT_POLL_TYPE.key);
        return result;
    }

    public PollVoteVisibility getDefaultPollVoteVisibility() {
        PollVoteVisibility result = applicationConfig.getOption(
                PollVoteVisibility.class,
                PollenConfigurationOption.DEFAULT_POLL_VOTE_VISIBILITY.key);
        return result;
    }

    public PollCommentVisibility getDefaultPollCommentVisibility() {
        PollCommentVisibility result = applicationConfig.getOption(
                PollCommentVisibility.class,
                PollenConfigurationOption.DEFAULT_POLL_COMMENT_VISIBILITY.key);
        return result;
    }

    public String getAdminLogin() {
        String result = applicationConfig.getOption(
                PollenConfigurationOption.ADMIN_LOGIN.key);
        return result;
    }

    public String getAdminPassword() {
        String result = applicationConfig.getOption(
                PollenConfigurationOption.ADMIN_PASSWORD.key);
        return result;
    }

    public String getAdminEmail() {
        String result = applicationConfig.getOption(
                PollenConfigurationOption.ADMIN_EMAIL.key);
        return result;
    }

    public String getContactEmail() {
        String result = applicationConfig.getOption(
                PollenConfigurationOption.CONTACT_EMAIL.key);
        return result;
    }

    public String getEmailHost() {
        String result = applicationConfig.getOption(
                PollenConfigurationOption.EMAIL_HOST.key);
        return result;
    }

    public int getEmailPort() {
        int result = applicationConfig.getOptionAsInt(
                PollenConfigurationOption.EMAIL_PORT.key);
        return result;
    }

    public String getEmailFrom() {
        String result = applicationConfig.getOption(
                PollenConfigurationOption.EMAIL_FROM.key);
        return result;
    }

    public File getEmailDirectory() {
        File result = applicationConfig.getOptionAsFile(
                PollenConfigurationOption.EMAIL_DIR.key);
        return result;
    }

    public File getFeedDirectory() {
        File result = applicationConfig.getOptionAsFile(
                PollenConfigurationOption.FEED_DIR.key);
        return result;
    }

    public File getImageDirectory() {
        File result = applicationConfig.getOptionAsFile(
                PollenConfigurationOption.IMG_DIR.key);
        return result;
    }

    public File getTemporaryDirectory() {
        File result = applicationConfig.getOptionAsFile(
                PollenConfigurationOption.TMP_DIR.key);
        return result;
    }

    protected File getDataDirectory() {
        File result = applicationConfig.getOptionAsFile(
                PollenConfigurationOption.DATA_DIR.key);
        return result;
    }

    public int getNbVotesPerPage() {
        int result = applicationConfig.getOptionAsInt(
                PollenConfigurationOption.NB_VOTES_PER_PAGE.key);
        return result;
    }

    public String getCharset() {
        String result = applicationConfig.getOption(
                PollenConfigurationOption.CHARSET.key);
        return result;
    }

    public Version getVersion() {
        Version result = applicationConfig.getOptionAsVersion("pollen.version");
        return result;
    }

    public void setAdminLogin(String adminLogin) {
        setOption(PollenConfigurationOption.ADMIN_LOGIN.key, adminLogin);
    }

    public void setAdminPassword(String adminPassword) {
        setOption(PollenConfigurationOption.ADMIN_PASSWORD.key, adminPassword);
    }

    public void setAdminEmail(String adminEmail) {
        setOption(PollenConfigurationOption.ADMIN_EMAIL.key, adminEmail);
    }

    public void setContactEmail(String contactEmail) {
        setOption(PollenConfigurationOption.CONTACT_EMAIL.key, contactEmail);
    }

    public void setEmailHost(String emailHost) {
        setOption(PollenConfigurationOption.EMAIL_HOST.key, emailHost);
    }

    public void setEmailPort(int emailPort) {
        setOption(PollenConfigurationOption.EMAIL_PORT.key, String.valueOf(emailPort));
    }

    public void setEmailFrom(String emailFrom) {
        setOption(PollenConfigurationOption.EMAIL_FROM.key, emailFrom);
    }

    public void setEmailDirectory(File emailDirectory) {
        setOption(PollenConfigurationOption.EMAIL_DIR.key,
                  emailDirectory.getAbsolutePath());
    }

    public void setFeedDirectory(File feedDirectory) {
        setOption(PollenConfigurationOption.FEED_DIR.key,
                  feedDirectory.getAbsolutePath());
    }

    public void setImageDirectory(File imageDirectory) {
        setOption(PollenConfigurationOption.IMG_DIR.key,
                  imageDirectory.getAbsolutePath());
    }

    public void setNbVotesPerPage(int nbVotesPerPage) {
        setOption(PollenConfigurationOption.NB_VOTES_PER_PAGE.key,
                  String.valueOf(nbVotesPerPage));
    }

    public void setCharset(String charset) {
        applicationConfig.setOption(
                PollenConfigurationOption.CHARSET.key, charset);
    }

    @Override
    public String toString() {
        return ReflectionToStringBuilder.toString(this);
    }

    public Properties getProperties() {
        Properties result = applicationConfig.getFlatOptions();
        return result;
    }

    public boolean getOptionAsBoolean(String propertyName) {
        Boolean result = applicationConfig.getOptionAsBoolean(propertyName);
        return result != null && result;
    }

    protected void loadApplicationConfig(String file, Properties defaultOptions) {
        applicationConfig = new ApplicationConfig(file);
        try {
            applicationConfig.loadDefaultOptions(
                    PollenConfigurationOption.values());

            if (defaultOptions != null) {
                Enumeration<?> enumeration = defaultOptions.propertyNames();
                while (enumeration.hasMoreElements()) {
                    String key = (String) enumeration.nextElement();
                    String property = defaultOptions.getProperty(key);
                    applicationConfig.setDefaultOption(key, property);
                }
            }
            applicationConfig.parse();
        } catch (ArgumentsParserException e) {
            throw new PollenTechnicalException(
                    "Could not parse configuration", e);
        }
    }

    protected File getOptionAsFile(String key) {
        return applicationConfig.getOptionAsFile(key);
    }

    protected void setOption(String key, String value) {
        applicationConfig.setOption(key, value);
    }

    /**
     * Creates a directory given the configuration given key.
     *
     * @param key the configuration option key which contains the location of
     *            the directory to create
     */
    private void createDirectory(PollenConfigurationOption key) {

        File directory = applicationConfig.getOptionAsFile(key.getKey());

        Preconditions.checkNotNull(
                directory,
                "Could not find directory " + directory + " (key " +
                key +
                "in your configuration file named echobase.properties)"
        );
        if (log.isInfoEnabled()) {
            log.info(key + " = " + directory);
        }
        try {
            FileUtil.createDirectoryIfNecessary(directory);
        } catch (IOException e) {
            throw new PollenTechnicalException(
                    "Could not create directory " + directory, e);
        }
    }

    public static final Comparator<PollenConfigurationOption> POLLEN_CONFIGURATION_OPTION_COMPARATOR = new Comparator<PollenConfigurationOption>() {
        @Override
        public int compare(PollenConfigurationOption o1, PollenConfigurationOption o2) {
            return o1.key.compareTo(o2.key);
        }
    };

    public String printConfig() {
        StringBuilder builder = new StringBuilder();
        List<PollenConfigurationOption> options = Lists.newArrayList(PollenConfigurationOption.values());
        Collections.sort(options, POLLEN_CONFIGURATION_OPTION_COMPARATOR);
        for (PollenConfigurationOption option : options) {
            builder.append(String.format("\n%1$-40s = %2$s",
                                         option.getKey(),
                                         applicationConfig.getOption(option)));
        }
        return builder.toString();
    }
}
