/*
 * This file is part of the OWL API.
 *
 * The contents of this file are subject to the LGPL License, Version 3.0.
 *
 * Copyright (C) 2011, The University of Manchester
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see http://www.gnu.org/licenses/.
 *
 *
 * Alternatively, the contents of this file may be used under the terms of the Apache License, Version 2.0
 * in which case, the provisions of the Apache License Version 2.0 are applicable instead of those above.
 *
 * Copyright 2011, University of Manchester
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.semanticweb.owlapi.io;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.zip.GZIPOutputStream;

import org.semanticweb.owlapi.model.IRI;
import org.semanticweb.owlapi.model.OWLRuntimeException;

/**
 * An ontology document target which can write to a GZIP stream. Notice that
 * this works best when the output stream is closed explicitly in the client
 * code.
 * 
 * @author ignazio
 * @since 3.4.8
 */
public class GZipStreamDocumentTarget implements OWLOntologyDocumentTarget {

    private OutputStream outputStream;
    private GZIPOutputStream zippedStream;

    /**
     * @param os
     *        the actual file
     */
    public GZipStreamDocumentTarget(OutputStream os) {
        outputStream = os;
    }

    @Override
    public boolean isWriterAvailable() {
        return isOutputStreamAvailable();
    }

    @Override
    public Writer getWriter() {
        if (!isWriterAvailable()) {
            throw new UnsupportedOperationException(
                    "writer not available; check with isWriterAvailable() first.");
        }
        return new OutputStreamWriter(getOutputStream());
    }

    @Override
    public boolean isOutputStreamAvailable() {
        return true;
    }

    @Override
    public OutputStream getOutputStream() {
        if (zippedStream == null) {
            try {
                zippedStream = new GZIPOutputStream(outputStream);
            } catch (FileNotFoundException e) {
                throw new OWLRuntimeException(e);
            } catch (IOException e) {
                throw new OWLRuntimeException(e);
            }
        }
        return zippedStream;
    }

    @Override
    public boolean isDocumentIRIAvailable() {
        return false;
    }

    @Override
    public IRI getDocumentIRI() {
        throw new UnsupportedOperationException(
                "iri not available; check with isDocumentIRIAvailable() first");
    }
}
