package fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.frequency;

/*
 * #%L
 * Tutti :: UI
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import fr.ifremer.tutti.type.WeightUnit;
import org.jdesktop.swingx.table.TableColumnModelExt;
import org.jfree.data.xy.XYSeries;
import org.nuiton.jaxx.application.swing.table.AbstractApplicationTableModel;
import org.nuiton.jaxx.application.swing.table.ColumnIdentifier;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.List;

import static org.nuiton.i18n.I18n.n;

/**
 * Model of the species frequency table.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.2
 */
public class BenthosFrequencyTableModel extends AbstractApplicationTableModel<BenthosFrequencyRowModel> {

    private static final long serialVersionUID = 1L;

    public static final ColumnIdentifier<BenthosFrequencyRowModel> LENGTH_STEP = ColumnIdentifier.newId(
            BenthosFrequencyRowModel.PROPERTY_LENGTH_STEP,
            n("tutti.editBenthosFrequencies.table.header.lengthStep"),
            n("tutti.editBenthosFrequencies.table.header.lengthStep"));

    public static final ColumnIdentifier<BenthosFrequencyRowModel> NUMBER = ColumnIdentifier.newId(
            BenthosFrequencyRowModel.PROPERTY_NUMBER,
            n("tutti.editBenthosFrequencies.table.header.number"),
            n("tutti.editBenthosFrequencies.table.header.number"));

    public static final ColumnIdentifier<BenthosFrequencyRowModel> WEIGHT = ColumnIdentifier.newId(
            BenthosFrequencyRowModel.PROPERTY_WEIGHT,
            n("tutti.editBenthosFrequencies.table.header.weight"),
            n("tutti.editBenthosFrequencies.table.header.weight"));

    private final BenthosFrequencyUIModel uiModel;

    private final BenthosFrequencyUIModelCache modelCache;

    protected final XYSeries series;

    /**
     * Weight unit.
     *
     * @since 2.5
     */
    protected final WeightUnit weightUnit;

    protected transient PropertyChangeListener onLengthStepChangedListener;

    protected transient PropertyChangeListener onWeightChangedListener;

    protected transient PropertyChangeListener onNumberChangedListener;

    public BenthosFrequencyTableModel(WeightUnit weightUnit,
                                      TableColumnModelExt columnModel,
                                      BenthosFrequencyUIModel uiModel) {
        super(columnModel, true, true);
        this.weightUnit = weightUnit;
        this.uiModel = uiModel;
        this.modelCache = uiModel.cache;
        this.series = uiModel.dataset.getSeries(0);
        setNoneEditableCols();
    }

    @Override
    public BenthosFrequencyRowModel createNewRow() {
        return createNewRow(true);
    }

    public BenthosFrequencyRowModel createNewRow(boolean attachListeners) {
        Float defaultStep = null;

        int rowCount = getRowCount();
        if (rowCount > 0) {

            BenthosFrequencyRowModel rowModel = getEntry(rowCount - 1);
            Float lengthStep = rowModel.getLengthStep();
            if (lengthStep != null) {
                defaultStep = uiModel.getLengthStep(lengthStep + uiModel.getStep());
            }
        }
        BenthosFrequencyRowModel result = new BenthosFrequencyRowModel(weightUnit);

        if (attachListeners) {
            attachListeners(result);
        }

        result.setLengthStepCaracteristic(uiModel.getLengthStepCaracteristic());
        result.setLengthStep(defaultStep);
        result.setValid(defaultStep != null);

        return result;
    }

    @Override
    public void setValueAt(Object aValue,
                           int rowIndex,
                           int columnIndex,
                           ColumnIdentifier<BenthosFrequencyRowModel> propertyName,
                           BenthosFrequencyRowModel entry) {
        super.setValueAt(aValue, rowIndex, columnIndex, propertyName, entry);
        // TODO Rebuild the computedWeight if possible...
    }

    @Override
    protected void onRowAdded(int rowIndex, BenthosFrequencyRowModel row) {
        uiModel.recomputeCanEditLengthStep();
    }

    @Override
    protected void onRowUpdated(int rowIndex, BenthosFrequencyRowModel row) {
        uiModel.recomputeCanEditLengthStep();
    }

    @Override
    protected void onRowRemoved(int rowIndex, BenthosFrequencyRowModel row) {
        dettachListeners(row);
        uiModel.recomputeCanEditLengthStep();
    }

    @Override
    protected void onBeforeRowsChanged(List<BenthosFrequencyRowModel> oldRows) {

        if (oldRows != null) {

            for (BenthosFrequencyRowModel row : oldRows) {
                dettachListeners(row);
            }

        }

    }

    @Override
    protected void onRowsChanged(List<BenthosFrequencyRowModel> newRows) {

        if (newRows != null) {

            for (BenthosFrequencyRowModel row : newRows) {
                dettachListeners(row); // prevent leaks!
                attachListeners(row);
            }

        }

        uiModel.recomputeCanEditLengthStep();

    }

    public void decrementNumberForLengthStep(Float lengthStep) {

        BenthosFrequencyRowModel row = modelCache.getRowCache().get(lengthStep);

        if (row != null) {

            Integer number = row.getNumber();

            if (number != null) {

                if (number > 1) {
                    row.setNumber(number - 1);
                } else {
                    row.setNumber(null);
                }

                int rowIndex = getRowIndex(row);

                fireTableRowsUpdated(rowIndex, rowIndex);

            }

            uiModel.recomputeCanEditLengthStep();

        }

    }

    private PropertyChangeListener getOnLengthStepChangedListener() {
        if (onLengthStepChangedListener == null) {
            onLengthStepChangedListener = new PropertyChangeListener() {
                @Override
                public void propertyChange(PropertyChangeEvent evt) {

                    Float oldValue = (Float) evt.getOldValue();
                    if (oldValue != null) {

                        modelCache.removeLengthStep(oldValue);
                        if (series.indexOf(oldValue) >= 0) {
                            series.remove(oldValue);
                        }

                    }

                    BenthosFrequencyRowModel row = (BenthosFrequencyRowModel) evt.getSource();

                    Float newValue = (Float) evt.getNewValue();
                    if (newValue != null) {

                        modelCache.addLengthStep(row);

                        if (row.getNumber() != null) {

                            series.addOrUpdate(newValue, row.getNumber());

                        }

                    }

                    uiModel.recomputeCanEditLengthStep();
                    uiModel.recomputeRowsValidateState();
                    uiModel.updateEmptyRow(row);

                    // Can recompute total number and weight only after valid flag change
                    uiModel.recomputeTotalNumberAndWeight();

                    fireTableDataChanged();

                }
            };
        }
        return onLengthStepChangedListener;
    }

    private PropertyChangeListener getOnNumberChangedListener() {
        if (onNumberChangedListener == null) {
            onNumberChangedListener = new PropertyChangeListener() {
                @Override
                public void propertyChange(PropertyChangeEvent evt) {

                    BenthosFrequencyRowModel row = (BenthosFrequencyRowModel) evt.getSource();

                    Integer newValue = (Integer) evt.getNewValue();

                    Float lengthStep = row.getLengthStep();

                    if (lengthStep != null) {

                        if (newValue == null) {

                            // remove the value for the lengthStep
                            if (series.indexOf(lengthStep) >= 0) {
                                series.remove(lengthStep);
                            }

                        } else {

                            series.addOrUpdate(lengthStep, row.getNumber());

                        }

                    }

                    uiModel.recomputeCanEditLengthStep();
                    uiModel.recomputeRowValidState(row);
                    uiModel.updateEmptyRow(row);

                    // Can recompute total number and weight only after valid flag change
                    uiModel.recomputeTotalNumberAndWeight();

                }
            };
        }
        return onNumberChangedListener;
    }

    private PropertyChangeListener getOnWeightChangedListener() {
        if (onWeightChangedListener == null) {
            onWeightChangedListener = new PropertyChangeListener() {
                @Override
                public void propertyChange(PropertyChangeEvent evt) {

                    BenthosFrequencyRowModel row = (BenthosFrequencyRowModel) evt.getSource();
                    modelCache.updateRowWithWeight(row);

                    uiModel.recomputeRowsValidateState();
                    uiModel.updateEmptyRow(row);

                    // Can recompute total number and weight only after valid flag change
                    uiModel.recomputeTotalNumberAndWeight();

                }
            };
        }
        return onWeightChangedListener;
    }

    private void dettachListeners(BenthosFrequencyRowModel result) {

        result.removePropertyChangeListener(BenthosFrequencyRowModel.PROPERTY_LENGTH_STEP, getOnLengthStepChangedListener());
        result.removePropertyChangeListener(BenthosFrequencyRowModel.PROPERTY_WEIGHT, getOnWeightChangedListener());
        result.removePropertyChangeListener(BenthosFrequencyRowModel.PROPERTY_NUMBER, getOnNumberChangedListener());

    }

    private void attachListeners(BenthosFrequencyRowModel result) {

        result.addPropertyChangeListener(BenthosFrequencyRowModel.PROPERTY_LENGTH_STEP, getOnLengthStepChangedListener());
        result.addPropertyChangeListener(BenthosFrequencyRowModel.PROPERTY_WEIGHT, getOnWeightChangedListener());
        result.addPropertyChangeListener(BenthosFrequencyRowModel.PROPERTY_NUMBER, getOnNumberChangedListener());

    }

}
