/*
 * #%L
 * Vradi :: Services
 * 
 * $Id: FilterListQueryHandler.java 1715 2010-10-27 19:21:28Z tchemit $
 * $HeadURL: svn+ssh://sletellier@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.4.0/vradi-services/src/main/java/com/jurismarches/vradi/services/search/FilterListQueryHandler.java $
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package com.jurismarches.vradi.services.search;

import java.util.List;
import java.util.Stack;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.PrefixQuery;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.TermRangeQuery;
import org.apache.lucene.search.WildcardQuery;

import com.jurismarches.vradi.services.search.FilterList.Operator;

/**
 * <code>FilterListHandler</code> handles lucene queries and assembles them into
 * a <code>FilterList</code> tree.
 * 
 * @author $Author: tchemit $
 * @version $Revision: 1715 $ $Date: 2010-10-27 21:21:28 +0200 (mer., 27 oct. 2010) $
 * @since 0.0.4
 */
public class FilterListQueryHandler implements QueryHandler {
    private static final Log log = LogFactory.getLog(FilterListQueryHandler.class);

    protected Stack<FilterList> filterListStack = new Stack<FilterList>();

    public FilterListQueryHandler() {
        FilterList filterList = new FilterList();
        filterListStack.push(filterList);
    }

    @Override
    public void prefixQuery(PrefixQuery query, BooleanClause booleanClause) {
        Term term = query.getPrefix();
        Filter filter = new CompareFilter(term.field(), term.text() + "*");
        filter(filter, booleanClause);
    }

    @Override
    public void termQuery(TermQuery query, BooleanClause booleanClause) {
        Term term = query.getTerm();
        Filter filter = new CompareFilter(term.field(), term.text());
        filter(filter, booleanClause);
    }

    @Override
    public void rangeQuery(TermRangeQuery query, BooleanClause booleanClause) {
        String field = query.getField();
        String lowerTerm = query.getLowerTerm();
        String upperTerm = query.getUpperTerm();
        
        Filter filter = new RangeFilter(field, lowerTerm, upperTerm);
        filter(filter, booleanClause);
    }

    @Override
    public void wildcardQuery(WildcardQuery query, BooleanClause booleanClause) {
        Term term = query.getTerm();
        Filter filter = new CompareFilter(term.field(), term.text());
        filter(filter, booleanClause);
    }

    @Override
    public void group(BooleanClause booleanClause) {
        if (booleanClause == null) {
            return;
        }
        
        FilterList.Operator operator = Operator.getOperator(booleanClause);
        FilterList filterList = new FilterList(operator);
        
        if (log.isTraceEnabled()) {
            log.trace("group: " + operator.toString());
        }
        
        FilterList peek = filterListStack.peek();
        peek.addFilter(filterList);

        filterListStack.push(filterList);
    }

    @Override
    public void ungroup() {
        if (filterListStack.size() > 1) {
            filterListStack.pop();
        }
    }

    public FilterList getFilterList() {
        FilterList list = filterListStack.get(0);
        
        if (log.isDebugEnabled()) {
            traceFilterList(list, "");
        }
        
        return list;
    }
    
    public static String getQueryFromFilterList(FilterList list) {
        if(list == null) {
            return null;
        }
        FilterList parameter;
        List<Filter> filters = list.getFilters();
        if(filters.size() == 1 && filters.get(0) instanceof FilterList) {
            parameter = (FilterList)filters.get(0);
        } else {
            parameter = list;
        }
        String[] result = recomposeQuery(parameter);
        return result[0] == null ? result[1] :
                String.format("%s:(%s)", result[0], result[1]);
    }

    static void traceFilterList(FilterList list, String spaces) {
        Operator operator = list.getOperator();
        log.debug(String.format("%s%s [", spaces, operator.toString()));
        spaces = spaces + "  ";
        List<Filter> filters = list.getFilters();
        for (Filter filter : filters) {
            if (filter instanceof FilterList) {
                traceFilterList((FilterList) filter, spaces);
            } else {
                log.debug(String.format("%s%s", spaces, filter.toString()));
            }
        }
        spaces = spaces.substring(0, spaces.length() - 2);
        log.debug(spaces + "]");
    }
    
    private void filter(Filter filter, BooleanClause booleanClause) {
        FilterList peek = filterListStack.peek();
        FilterList.Operator operator = Operator.getOperator(booleanClause);
        
        if (operator == FilterList.Operator.FILTER_NOT &&
                peek.getOperator() != operator) {
            
            group(booleanClause);
            peek = filterListStack.peek();
            peek.addFilter(filter);
            ungroup();
            
        } else {
            peek.addFilter(filter);
        }
    }

    private static String[] recomposeQuery(FilterList filterList) {
        String[] result = new String[2];

        List<Filter> filters = filterList.getFilters();
        for (Filter filter : filters) {
            String term = null;
            String value = null;
            String openSeparator = null;
            String closeSeparator = null;
            
            if(filter instanceof FilterList) {
                FilterList lFilter = (FilterList) filter;
                String[] listResult = recomposeQuery(lFilter);
                term = listResult[0];
                value = listResult[1];
                openSeparator="(";
                closeSeparator=")";

            } else if(filter instanceof CompareFilter) {
                CompareFilter cFilter = (CompareFilter) filter;
                term = cFilter.getName();
                value = "\"" + cFilter.getValue() + "\"";
                openSeparator="";
                closeSeparator="";
            }
            
            //if it is the first filter
            if(result[0] == null && result[1] == null) {
                result[0] = term;
                result[1] = String.format("%s%s%s",openSeparator, value,
                        closeSeparator);
                
            } else if(result[0] != null) {
                //if the previous filter was concerning the same term
                if(result[0].equals(term)) {
                    result[1] = String.format("%s %s %s%s%s", result[1],
                            filterList.getOperator().getValue(),
                            openSeparator, value, closeSeparator);
                
                //if the previous filter was concerning another term
                } else {
                    result[1] = String.format("%s:(%s) %s %s:(%s)",
                            result[0], result[1],
                            filterList.getOperator().getValue(),
                            term, value);
                    result[0] = null;
                }
                
            } else {
                  result[1] = String.format("%s %s %s:(%s)", result[1],
                            filterList.getOperator().getValue(),
                            term, value);
            }
        }

        return result;
    }


}
