/*
 * #%L
 * SGQ :: Business
 * $Id: BatchImportModel.java 320 2013-04-17 08:26:15Z echatellier $
 * $HeadURL: https://svn.codelutin.com/sgq-ch/tags/sgq-ch-1.1.6/sgq-business/src/main/java/com/herbocailleau/sgq/business/services/csv/BatchImportModel.java $
 * %%
 * Copyright (C) 2012 - 2013 Herboristerie Cailleau
 * %%
 * Herboristerie Cailleau - Tous droits réservés
 * #L%
 */

package com.herbocailleau.sgq.business.services.csv;

import static org.nuiton.i18n.I18n._;

import java.text.ParseException;
import java.util.Date;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.nuiton.util.csv.ValueParser;
import org.nuiton.util.csv.ext.AbstractImportModel;

import com.herbocailleau.sgq.business.SgqBusinessException;
import com.herbocailleau.sgq.business.SgqUtils;
import com.herbocailleau.sgq.entities.Batch;
import com.herbocailleau.sgq.entities.Client;
import com.herbocailleau.sgq.entities.Country;
import com.herbocailleau.sgq.entities.Supplier;

/**
 * Modèle d'import du fichier CSV du fichier numero lot (ancien fichier
 * excel circulant dans l'herboristerie cailleau , faisant office de
 * référence et contenant des informations corrigées par rapport au fichier
 * dbase).
 * 
 * "DATE_R";"CODF_R";"FOUR_R";"COMM_R";"COLIS_R";"POIDS_R";"PRO_R";"PAYS_R";
 * "CODP_R";"LOT_R";"DLUO_R";"CLI_R";"CODC_R";"ECH_R";"REM_R";"IG_R";"Date épuissé";"DMES"
 * 
 * @author echatellier
 */
public class BatchImportModel extends AbstractImportModel<BatchBean> {

    public BatchImportModel(final Map<String, Supplier> suppliers,
            final Map<String, Client> clients) {
        super(';');
        newMandatoryColumn("DATE_R", Batch.PROPERTY_ENTRY_DATE, new ValueParser<Date>() {
            @Override
            public Date parse(String value) throws ParseException {
                return SgqUtils.parseSgqDate(value);
            }
        });
        newMandatoryColumn("CODF_R", Batch.PROPERTY_SUPPLIER, new ValueParser<Supplier>() {
            @Override
            public Supplier parse(String value) throws ParseException {
                Supplier supplier = suppliers.get(value);
                if (supplier == null) {
                    throw new SgqBusinessException(_("Impossible de trouver le fournisseur %s", value));
                }
                return supplier;
            }
        });
        newIgnoredColumn("FOUR_R");
        newMandatoryColumn("COMM_R", Batch.PROPERTY_ORDER_NUMBER, new ValueParser<Integer>() {
            @Override
            public Integer parse(String value) throws ParseException {
                return Integer.valueOf(value);
            }
        });
        newMandatoryColumn("COLIS_R", Batch.PROPERTY_PACKAGE_COUNT, new ValueParser<Integer>() {
            @Override
            public Integer parse(String value) throws ParseException {
                return Integer.valueOf(value);
            }
        });
        newMandatoryColumn("POIDS_R", Batch.PROPERTY_QUANTITY, new ValueParser<Double>() {
            @Override
            public Double parse(String value) throws ParseException {
                return Double.valueOf(value.replace(',','.').replaceAll("[^\\d.]",""));
            }
        });
        newMandatoryColumn("PRO_R", BatchBean.PROPERTY_PRODUCT_CODE, new ValueParser<String>() {
            @Override
            public String parse(String value) throws ParseException {
                if (StringUtils.length(value) != 6) {
                    throw new SgqBusinessException(_("Code produit invalide %s ", value));
                }
                return value;
            }
        });
        newIgnoredColumn("PAYS_R");
        newMandatoryColumn("CODP_R", Batch.PROPERTY_ORIGIN, new ValueParser<Country>() {
            @Override
            public Country parse(String value) throws ParseException {
                return Country.valueOf(value);
            }
        });
        newMandatoryColumn("LOT_R", Batch.PROPERTY_NUMBER, new ValueParser<Integer>() {
            @Override
            public Integer parse(String value) throws ParseException {
                return Integer.valueOf(value);
            }
        });
        newMandatoryColumn("DLUO_R", Batch.PROPERTY_DLUO, new ValueParser<Date>() {
            @Override
            public Date parse(String value) throws ParseException {
                return SgqUtils.parseSgqDate(value);
            }
        });
        newIgnoredColumn("CLI_R");
        newMandatoryColumn("CODC_R", Batch.PROPERTY_DEDICATED_CLIENT, new ValueParser<Client>() {
            @Override
            public Client parse(String value) throws ParseException {
                Client client = null;
                // il n'est pas obligatoire que le lot soit dédié à un client
                if (StringUtils.isNotBlank(value)) {
                    client = clients.get(value);
                    if (client == null) {
                        throw new SgqBusinessException(_("Impossible de trouver le client %s", value));
                    }
                }
                return client;
            }
        });
        newMandatoryColumn("ECH_R", Batch.PROPERTY_SAMPLE_CODE);
        newMandatoryColumn("REM_R", Batch.PROPERTY_COMMENT);
        newMandatoryColumn("IG_R", Batch.PROPERTY_INVALID, new ValueParser<Boolean>() {
            // si cette colonne est valuée, toute la ligne doit être ingorée
            @Override
            public Boolean parse(String value) throws ParseException {
                Boolean result = Boolean.FALSE;
                if ("O".equalsIgnoreCase(value)) {
                    result = Boolean.TRUE;
                }
                return result;
            }
        });
        newMandatoryColumn("Date épuissé", Batch.PROPERTY_EXPIRED_DATE, new ValueParser<Date>() {
            @Override
            public Date parse(String value) throws ParseException {
                Date result = null;
                // la date peut être vide (=pas épuisé)
                if (StringUtils.isNotBlank(value)) {
                    result = SgqUtils.parseSgqDate(value);
                }
                return result;
            }
        });
        newMandatoryColumn("DMES", Batch.PROPERTY_DMES, new ValueParser<Date>() {
            @Override
            public Date parse(String value) throws ParseException {
                Date result = null;
                // la date peut être vide (pas de dmes)
                if (StringUtils.isNotBlank(value)) {
                    result = SgqUtils.parseSgqDate(value);
                }
                return result;
            }
        });
    }

    @Override
    public BatchBean newEmptyInstance() {
        return new BatchBean();
    }
}
