/*
 * Copyright (c) 2008-2015, Hazelcast, Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.hazelcast.client.impl.protocol.template;

import com.hazelcast.annotation.GenerateCodec;
import com.hazelcast.annotation.Nullable;
import com.hazelcast.annotation.Request;
import com.hazelcast.annotation.Since;
import com.hazelcast.client.impl.protocol.constants.ResponseMessageConst;
import com.hazelcast.client.impl.protocol.constants.EventMessageConst;
import com.hazelcast.nio.Address;
import com.hazelcast.nio.serialization.Data;

@GenerateCodec(id = TemplateConstants.CLIENT_TEMPLATE_ID, name = "Client", ns = "Hazelcast.Client.Protocol.Codec")
public interface ClientMessageTemplate {

    /**
     * @param username             Name of the user for authentication.
     * @param password             Password for the user.
     * @param uuid                 Unique string identifying the connected client uniquely. This string is generated by the owner member server
     *                             on initial connection. When the client connects to a non-owner member it sets this field on the request.
     * @param ownerUuid            Unique string identifying the server member uniquely.
     * @param isOwnerConnection    You must set this field to true while connecting to the owner member, otherwise set to false.
     * @param clientType           The type of the client. E.g. JAVA, CPP, CSHARP, etc.
     * @param serializationVersion client side supported version to inform server side
     * @param clientHazelcastVersion The Hazelcast version of the client. (e.g. 3.7.2)
     * @return Returns the address, uuid and owner uuid.
     */
    @Request(id = 2, retryable = true, response = ResponseMessageConst.AUTHENTICATION)
    Object authentication(String username, String password, @Nullable String uuid, @Nullable String ownerUuid,
                          boolean isOwnerConnection, String clientType, byte serializationVersion,
                          @Since (value = "1.3") String clientHazelcastVersion);

    /**
     * @param credentials          Secret byte array for authentication.
     * @param uuid                 Unique string identifying the connected client uniquely. This string is generated by the owner member server
     *                             on initial connection. When the client connects to a non-owner member it sets this field on the request.
     * @param ownerUuid            Unique string identifying the server member uniquely.
     * @param isOwnerConnection    You must set this field to true while connecting to the owner member, otherwise set to false.
     * @param clientType           The type of the client. E.g. JAVA, CPP, CSHARP, etc.
     * @param serializationVersion client side supported version to inform server side
     * @param clientHazelcastVersion The Hazelcast version of the client. (e.g. 3.7.2)
     * @return Returns the address, uuid and owner uuid.
     */

    @Request(id = 3, retryable = true, response = ResponseMessageConst.AUTHENTICATION)
    Object authenticationCustom(Data credentials, @Nullable String uuid, @Nullable String ownerUuid, boolean isOwnerConnection,
                                String clientType, byte serializationVersion,
                                @Since (value = "1.3") String clientHazelcastVersion);

    /**
     * @param localOnly if true only master node sends events, otherwise all registered nodes send all membership
     *                  changes.
     * @return Returns the registration id for the listener.
     */
    @Request(id = 4, retryable = false, response = ResponseMessageConst.STRING,
            event = {EventMessageConst.EVENT_MEMBER, EventMessageConst.EVENT_MEMBERLIST, EventMessageConst.EVENT_MEMBERATTRIBUTECHANGE})
    Object addMembershipListener(boolean localOnly);

    /**
     * @param name        The distributed object name for which the proxy is being created for.
     * @param serviceName The name of the service. Possible service names are:
     *                    "hz:impl:listService"
     *                    "hz:impl:queueService"
     *                    "hz:impl:setService"
     *                    "hz:impl:atomicLongService"
     *                    "hz:impl:atomicReferenceService"
     *                    "hz:impl:countDownLatchService"
     *                    "hz:impl:idGeneratorService"
     *                    "hz:impl:semaphoreService"
     *                    "hz:impl:executorService"
     *                    "hz:impl:mapService"
     *                    "hz:impl:mapReduceService"
     *                    "hz:impl:multiMapService"
     *                    "hz:impl:quorumService"
     *                    "hz:impl:replicatedMapService"
     *                    "hz:impl:ringbufferService"
     *                    "hz:core:proxyService"
     *                    "hz:impl:reliableTopicService"
     *                    "hz:impl:topicService"
     *                    "hz:core:txManagerService"
     *                    "hz:impl:xaService"
     */
    @Request(id = 5, retryable = false, response = ResponseMessageConst.VOID)
    void createProxy(String name, String serviceName, Address target);

    /**
     * @param name        The distributed object name for which the proxy is being destroyed for.
     * @param serviceName The name of the service. Possible service names are:
     *                    "hz:impl:listService"
     *                    "hz:impl:queueService"
     *                    "hz:impl:setService"
     *                    "hz:impl:atomicLongService"
     *                    "hz:impl:atomicReferenceService"
     *                    "hz:impl:countDownLatchService"
     *                    "hz:impl:idGeneratorService"
     *                    "hz:impl:semaphoreService"
     *                    "hz:impl:executorService"
     *                    "hz:impl:mapService"
     *                    "hz:impl:mapReduceService"
     *                    "hz:impl:multiMapService"
     *                    "hz:impl:quorumService"
     *                    "hz:impl:replicatedMapService"
     *                    "hz:impl:ringbufferService"
     *                    "hz:core:proxyService"
     *                    "hz:impl:reliableTopicService"
     *                    "hz:impl:topicService"
     *                    "hz:core:txManagerService"
     *                    "hz:impl:xaService"
     */

    @Request(id = 6, retryable = false, response = ResponseMessageConst.VOID)
    void destroyProxy(String name, String serviceName);

    /**
     * @return The partition list for each member address.
     */
    @Request(id = 8, retryable = false, response = ResponseMessageConst.PARTITIONS)
    Object getPartitions();

    @Request(id = 9, retryable = false, response = ResponseMessageConst.VOID)
    void removeAllListeners();

    /**
     * @param localOnly if true only node that has the partition sends the request, if false
     *                  sends all partition lost events.
     * @return The listener registration id.
     */
    @Request(id = 10, retryable = false, response = ResponseMessageConst.STRING, event = {EventMessageConst.EVENT_PARTITIONLOST})
    Object addPartitionLostListener(boolean localOnly);

    /**
     * @param registrationId The id assigned during the listener registration.
     * @return true if the listener existed and removed, false otherwise.
     */
    @Request(id = 11, retryable = true, response = ResponseMessageConst.BOOLEAN)
    Object removePartitionLostListener(String registrationId);

    /**
     * @return An array of distributed object info in the cluster.
     */
    @Request(id = 12, retryable = false, response = ResponseMessageConst.LIST_DISTRIBUTED_OBJECT)
    Object getDistributedObjects();

    /**
     * @param localOnly If set to true, the server adds the listener only to itself, otherwise the listener is is added for all
     *                  members in the cluster.
     * @return The registration id for the distributed object listener.
     */
    @Request(id = 13, retryable = false, response = ResponseMessageConst.STRING, event = {EventMessageConst.EVENT_DISTRIBUTEDOBJECT})
    Object addDistributedObjectListener(boolean localOnly);

    /**
     * @param registrationId The id assigned during the registration.
     * @return true if the listener existed and removed, false otherwise.
     */
    @Request(id = 14, retryable = true, response = ResponseMessageConst.BOOLEAN)
    Object removeDistributedObjectListener(String registrationId);

    @Request(id = 15, retryable = true, response = ResponseMessageConst.VOID)
    void ping();
}
