/* *##%
 * Copyright (c) 2009 ruchaud. All rights reserved.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *##%*/

package org.sharengo.wikitty.jms;

import java.util.Date;
import java.util.Enumeration;

import javax.jms.DeliveryMode;
import javax.jms.Destination;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.XATopicSession;
import javax.transaction.TransactionManager;
import javax.transaction.xa.XAResource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.sharengo.wikitty.WikittyTransaction;

/**
 * @author ruchaud
 * @version $Revision: 1 $
 *
 * Last update: $Date: 2010-04-16 10:29:38 +0200 (ven., 16 avril 2010) $
 * by : $Author: echatellier $
 */
public class WikittyJMSUtil {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(WikittyJMSUtil.class);

    /** topic factory name in JMS*/
    static final public String TOPIC_CONNECTION_FACTORY = "TopicConnectionFactory";
    /** topic use for WikittyStorage in JMS*/
    static final public String TOPIC_WIKITTY_STORAGE = "WikittyStorageTopic";
    /** topic use for WikittyExtensionStorage in JMS*/
    static final public String TOPIC_EXTENSION_STORAGE = "WikittyExtensionStorageTopic";

    /** specify store action in message */
    static final public String ACTION_STORE = "store";
    /** specify delete action in message */
    static final public String ACTION_DELETE = "delete";
    /** specify clear action in message */
    static final public String ACTION_CLEAR = "clear";

    /** parameter wikitties in message with List<Wikitty>*/
    static final public String PARAM_WIKITTIES = "wikitties";
    /** parameter disableAutoVersionIncrement in message with boolean */
    static final public String PARAM_DISABLE_AUTO_VERSION_INCREMENT = "disableAutoVersionIncrement";
    /** parameter ids in message with List<String> */
    static final public String PARAM_IDS = "ids";
    /** parameter action in message specifying action to perform */
    static final public String PARAM_ACTION = "action";
    /** parameter ids in message with List<WikittyExtension> */
    static final public String PARAM_EXTENSIONS = "extensions";
    
    /** provider url for jndi */
    static final public String JNDI_PROVIDER_URL = "java.naming.provider.url";

    /**
     * Add topicSession XAResource in transaction
     * @param transaction transaction
     * @param topicSession session
     * @throws Exception
     */
    public static void addXAResource(WikittyTransaction transaction,
            XATopicSession topicSession) throws Exception {
        XAResource xAResource = topicSession.getXAResource();
        TransactionManager transactionManager = transaction.getTransactionManager();
        transactionManager.getTransaction().enlistResource(xAResource);
    }
    
    /**
     * Log message
     * @param logger
     * @param message
     */
    public static void logMessageHeaders(Log logger, Message message) {
        if (message == null) {
            logger.warn("message is null");
            return;
        }

        try {
            // destination
            Destination dest = message.getJMSDestination();
            logger.debug("JMSDestination: " + dest);

            // delivery mode
            int delMode = message.getJMSDeliveryMode();
            if (delMode == DeliveryMode.NON_PERSISTENT) {
                logger.debug("JMSDeliveryMode: non-persistent");

            } else if (delMode == DeliveryMode.PERSISTENT) {
                logger.debug("JMSDeliveryMode: persistent");

            } else {
                logger.error("JMSDeliveryMode: neither persistent nor non-persistent");
            }

            /*
             * Expiration time. If value is 0 (the default), the
             * message never expires.
             */
            long expiration = message.getJMSExpiration();

            if (expiration != 0) {
                Date expTime = new Date(expiration);
                logger.debug("JMSExpiration: " + expTime);

            } else {
                logger.debug("JMSExpiration: " + expiration);
            }

            // priority
            int priority = message.getJMSPriority();
            logger.debug("JMSPriority: " + priority);

            // message ID
            String msgID = message.getJMSMessageID();
            logger.debug("JMSMessageID: " + msgID);

            // timestamp.
            long timestamp = message.getJMSTimestamp();
            if (timestamp != 0) {
                Date timestampTime = new Date(timestamp);
                logger.debug("JMSTimestamp: " + timestampTime);

            } else {
                logger.debug("JMSTimestamp: " + timestamp);
            }

            // correlation ID
            String correlID = message.getJMSCorrelationID();
            logger.debug("JMSCorrelationID: " + correlID);

            // Display the ReplyTo destination
            Destination replyTo = message.getJMSReplyTo();
            logger.debug("JMSReplyTo: " + replyTo);

            // redelivered value (usually false)
            boolean redelivered = message.getJMSRedelivered();
            logger.debug("JMSRedelivered: " + redelivered);

            // JMSType
            String type = message.getJMSType();
            logger.debug("JMSType: " + type);

            // client properties.
            Enumeration e = message.getPropertyNames();
            while (e.hasMoreElements()) {
                String propertyName = (String) e.nextElement();

                logger.debug("Client property " + propertyName + ": "
                        + message.getObjectProperty(propertyName));
            }

        } catch (JMSException e) {
            logger.error("JMSException occurred: ", e);

        } catch (Exception e) {
            logger.error("Exception occurred: ", e);
        }
    }
}
