package org.planx.xmlstore.references;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.util.concurrent.Future;
import java.util.concurrent.ExecutionException;
import org.planx.msd.Discriminator;
import org.planx.msd.DiscriminatorFactory;
import org.planx.msd.util.DiscriminatorAdapter;
import org.planx.xmlstore.Reference;
import org.planx.xmlstore.io.Streamer;
import org.planx.xmlstore.io.Streamers;
import org.planx.xmlstore.util.Concurrency;

/**
 * A proxy <code>Reference</code> with an associated <code>Reference</code>
 * which can be unresolved. If any method is invoked on a <code>ReferenceProxy</code>,
 * and the associated <code>Reference</code> is unresolved, the thread will block
 * until the reference is resolved.
 *
 * @author Kasper Bøgebjerg
 * @author Henning Niss
 * @author Thomas Ambus
 */
public class ReferenceProxy implements Reference {
    public static final byte CLASS_ID = 0x04;
    private Future<Reference> future;

    public ReferenceProxy(Future<Reference> future) {
        this.future = future;
    }

    public Reference get() {
        try {
            return future.get();
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        } catch (ExecutionException e) {
            throw new RuntimeException(e);
        }
    }

    public static Reference resolve(Reference vref) {
        if (vref instanceof ReferenceProxy) {
            return resolve(((ReferenceProxy) vref).get());
        }
        return vref;
    }

    public static Streamer<ReferenceProxy> getStreamer() {
        final Streamer<Reference> vrefStreamer =
            Streamers.getPolymorphicStreamer(Reference.class);

        return new Streamer<ReferenceProxy>() {
            public void toStream(DataOutput out, ReferenceProxy vref) throws IOException {
                vrefStreamer.toStream(out, resolve(vref));
            }
            public ReferenceProxy fromStream(DataInput in) throws IOException {
                Reference vref = vrefStreamer.fromStream(in);
                Future<Reference> future = Concurrency.future(vref);
                return new ReferenceProxy(future);
            }
        };
    }

    public static Discriminator<ReferenceProxy> getDiscriminator() {
        return new DiscriminatorAdapter<ReferenceProxy,Reference>
            (DiscriminatorFactory.instance().getPolymorphicDiscriminator(Reference.class)) {
            protected Reference transform(ReferenceProxy vref) {
                return resolve(vref);
            }
        };
    }
}
