package org.planx.xmlstore.nodes;

import java.io.IOException;
import java.lang.ref.SoftReference;
import java.util.List;
import org.planx.msd.lang.EquivalenceClass;
import org.planx.msd.lang.EquivalenceClassDiscriminable;
import org.planx.xmlstore.*;
import org.planx.xmlstore.io.LocalLocator;

/**
 * @author Kasper Bøgebjerg
 * @author Henning Niss
 * @author Thomas Ambus
 */
public abstract class NodeProxy extends AbstractNode {
    private SoftReference<SystemNode> nodeRef;
    private SoftReference<LocalLocator> loc;

    protected NodeProxy(Reference ref) {
        if (ref == null) throw new NullPointerException("Reference is null");
        if (ref instanceof LocalLocator) setLoc((LocalLocator) ref);
        this.ref = ref;
    }

    protected NodeProxy(LocalLocator l, Reference ref) {
        if (l == null && ref == null) throw new NullPointerException();
        this.ref = ref;
        setLoc(l);
    }

    protected void setLoc(LocalLocator l) {
        this.loc = new SoftReference<LocalLocator>(l);
    }

    protected LocalLocator getLoc() {
        return (loc==null)?null:loc.get();
    }

    private SystemNode doGenerate() {
        SystemNode node = (nodeRef == null) ? null : nodeRef.get();
        if (node == null) {
            try {
                node = generate();
            } catch (UnknownReferenceException e) {
                throw new RuntimeException(e);
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
            nodeRef = new SoftReference<SystemNode>(node);
        }
        if (node == this) throw new IllegalStateException
            ("NodeProxy generated itself, loc="+getLoc()+", ref="+ref);
        return node;
    }

    protected abstract SystemNode generate() throws UnknownReferenceException, IOException;

    public SystemNode get() {
        SystemNode node = doGenerate().get();
        if (node.getLocator() != null) setLoc(node.getLocator());
        else node.setLocator(getLoc());
        return node;
    }

    public boolean isMutable() {
        return doGenerate().isMutable();
    }

    public byte getType() {
        return doGenerate().getType();
    }

    public String getNodeValue() {
        return doGenerate().getNodeValue();
    }

    public List<SystemNode> getChildren() {
        return doGenerate().getChildren();
    }

    public List<Attribute> getAttributes() {
        return doGenerate().getAttributes();
    }

    public String getAttribute(String attrName) {
        return doGenerate().getAttribute(attrName);
    }

    public String[] getAttributeNames() {
        return doGenerate().getAttributeNames();
    }

    public boolean isShared() {
        return doGenerate().isShared();
    }

    public void setShared(boolean isShared) {
        doGenerate().setShared(isShared);
    }

    public int getHeight() {
        return doGenerate().getHeight();
    }

    public void setHeight(int height) {
        doGenerate().setHeight(height);
    }

    public LocalLocator getLocator() {
        LocalLocator l = getLoc();
        if (l == null && ref instanceof LocalLocator) {
            l = (LocalLocator) ref;
            setLoc(l); // fixme, necessary?
        }
        return l;
    }

    public void setLocator(LocalLocator l) {
        setLoc(l);
        doGenerate().setLocator(l); // fixme, necessary?
    }

    public void setChild(int index, SystemNode child) {
        doGenerate().setChild(index, child);
    }

    /**
     * If the argument <code>Node</code> is also a <code>NodeProxy</code>,
     * attempts to compare the two nodes using references, otherwise the
     * proxied node is loaded and a structural comparison is (probably)
     * performed.
     */
    public boolean equals(Object o) {
        if (o == this) return true;
        if (o instanceof NodeProxy) {
            NodeProxy p = (NodeProxy) o;
            Reference ref1 = getLoc();
            Reference ref2 = p.getLoc();
            if (ref1 != null && ref2 != null && ref1.equals(ref2)) return true;
            ref1 = getReference();
            ref2 = p.getReference();
            if (ref1 != null && ref2 != null && ref1.equals(ref2)) return true;
        }
        if (!(o instanceof Node)) return false;
        return doGenerate().equals(o);
    }

    public int hashCode() {
        return doGenerate().hashCode();
    }

    public String toString() {
        return doGenerate().toString();
    }

    /**
     * Unload the proxied node from memory. Later uses of the proxy's
     * methods will attempt to generate it again.
     */
    public void unload() {
        nodeRef = null;
        loc = null;
    }
}
