package org.planx.xmlstore.io;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;

/**
 * A <code>Streamer</code> is capable of writing and reading objects of
 * a specific class to and from streams. The <code>Streamer</code> must
 * ensure that the format written by the <code>toStream</code> method
 * is compatible with the <code>fromStream</code> method. Moreover,
 * <i>all</i> data written by the <code>toStream</code> method must
 * be read by the <code>fromStream</code> method, so that the position
 * of the stream is ready for the next <code>Streamer</code>.
 * <p>
 * Compared to <code>Serializable</code> the advantage of using
 * <code>Streamer</code>s is that the identity of the class is not
 * written to the stream, thus making the output more compact. Furthermore,
 * multiple <code>Streamer</code>s can be defined for the same class so
 * that the output format is configurable and not fixed at one representation.
 * Note, that if a <code>Streamer</code> requires some kind of polymorphy,
 * it can simply write the identity of the class to the output itself
 * if needed.
 *
 * @author Thomas Ambus
 */
public interface Streamer<E> {
    /**
     * Writes the internal state of the object to the output stream
     * in a format that can later be read by the same <code>Streamer</code>
     * using the <code>fromStream</code> method.
     **/
    public void toStream(DataOutput out, E obj) throws IOException;

    /**
     * Reads the internal state of the object from the input stream.
     **/
    public E fromStream(DataInput in) throws IOException;
}
