package org.planx.xmlstore.io;

import java.io.*;
import org.planx.msd.Discriminator;
import org.planx.msd.Memory;

/**
 * @author Thomas Ambus
 */
public class MemoryFileSystem implements FileSystem {
    private FileSystemIdentifier fsi;
    private ByteOutputStream bout;
    private DataOutputStream out;

    public MemoryFileSystem() {
        this(1024);
    }

    public MemoryFileSystem(int capacity) {
        this(capacity, new FileSystemIdentifier());
    }

    public MemoryFileSystem(int capacity, FileSystemIdentifier fsi) {
        this.fsi = fsi;
        bout = new ByteOutputStream(capacity);
        out = new DataOutputStream(bout);
    }

    public FileSystemIdentifier currentIdentifier() {
        return fsi;
    }

    /**
     * Always succeeds and allocates space at end of buffer, writes must
     * be done before the next allocation.
     */
    public LocalLocator allocate() throws IOException {
        return new LocalLocator(bout.size(), 0, fsi);
    }

    public LocalLocator allocate(int size) throws IOException {
        return new LocalLocator(bout.size(), size, fsi);
    }

    public LocalLocator allocate(int size, FileSystemIdentifier id)
                                                throws IOException {
        if (!fsi.equals(id))
            throw new IllegalArgumentException
                ("FileSystemIdentifier not accepted "+id+" must be "+fsi);
        return allocate(size);
    }

    public LocalLocator all() {
        return new LocalLocator(0, bout.size(), fsi);
    }

    /**
     * Does nothing.
     */
    public void free(LocalLocator loc) throws IOException,
                                    UnknownLocatorException {
        checkLocator(loc);
    }

    /**
     * Frees all data and generates a new <code>FileSystemIdentifier</code>.
     */
    public void clear() {
        clear(new FileSystemIdentifier());
    }

    /**
     * Frees all data and sets the <code>FileSystemIdentifier</code> to
     * the specified object.
     */
    public void clear(FileSystemIdentifier fsi) {
        bout.reset();
        this.fsi = fsi;
    }

    public DataInput getInput(LocalLocator l) throws IOException,
                                            UnknownLocatorException {
        checkLocator(l);
        byte[] buf = bout.toByteArray(); // this does NOT copy! (see below)
        ByteArrayInputStream bin = new ByteArrayInputStream(
                        buf, l.getOff(), buf.length-l.getOff());
        return new DataInputStream(bin);
    }

    /**
     * Only append allowed.
     */
    public DataOutput getOutput(LocalLocator l) throws IOException,
                                        UnknownLocatorException {
        checkLocator(l);
        if (l.getOff() != bout.size()) throw new IOException
            ("Only append allowed. LocalLocator offset "+l.getOff()+
                            ", expected "+bout.size());
        return out;
    }

    public void copy(FileSystem fs, LocalLocator fromLoc, LocalLocator locTo)
                       throws IOException, UnknownLocatorException {
        checkLocator(locTo);
        if (locTo.getOff() != bout.size()) throw new IOException
            ("Only appending allowed, toLoc="+locTo+", expected offset "+
                                bout.size());
        DataInput in = fs.getInput(fromLoc);
        bout.copy(in, fromLoc.getLen());
    }

    public long size() throws IOException {
        return bout.size();
    }

    /**
     * No effect.
     */
    public void close() throws IOException {
    }

    public boolean isContained(LocalLocator l) {
        if (l == null) return false;
        return fsi.equals(l.getFileSystemId());
    }

    private void checkLocator(LocalLocator l) throws UnknownLocatorException {
        if (!fsi.equals(l.getFileSystemId()))
            throw new UnknownLocatorException
                ("LocalLocator does not originate from this FileSystem");
    }
}

class ByteOutputStream extends ByteArrayOutputStream {
    public ByteOutputStream(int capacity) {
        super(capacity);
    }

    public byte[] toByteArray() {
        return buf;  // do not copy to new array as in super class
    }

    public void copy(DataInput in, int len) throws IOException {
        int newcount = count + len;
        if (newcount > buf.length) {
            byte newbuf[] = new byte[Math.max(buf.length << 1, newcount)];
            System.arraycopy(buf, 0, newbuf, 0, count);
            buf = newbuf;
        }
        in.readFully(buf, count, len);
        count = newcount;
    }
}

