package org.planx.xmlstore;

import java.util.List;

/**
 * An XML Node. Currently, only element nodes and character data nodes are supported.
 *
 * @author Kasper Bøgebjerg
 * @author Henning Niss
 * @author Thomas Ambus
 */
public interface Node {
    /**
     * Node type element. The <code>Node</code> has a value, attributes, and
     * children.
     */
    public static final byte ELEMENT = 0x00;

    /**
     * Node type character data. The <code>Node</code> only has a value.
     */
    public static final byte CHARDATA = 0x01;

    /**
     * Returns <code>true</code> if this <code>Node</code> is mutable,
     * false otherwise.
     *
     * @return  <code>true</code> if mutable, <code>false</code> if immutable.
     */
    public boolean isMutable();

    /**
     * Returns the node type of this <code>Node</code>.
     *
     * @return  The node type
     */
    public byte getType();

    /**
     * Returns the value of this <code>Node</code>. When the node represents
     * a character data node, the returned value will be the character data.
     * When the node represents an element node, the returned value will
     * by the tag name of the element.
     *
     * @return  The node value.
     */
    public String getNodeValue();

    /**
     * Returns the attributes of this <code>Node</code>. In case
     * the node represents character data, the empty list is returned.
     *
     * @return An immutable list of the attributes of this <code>Node</code>.
     */
    public List<Attribute> getAttributes();

    /**
     * Returns the attribute with the given name. In case the
     * node represents a character data node the returned value is null.
     *
     * @return  The attribute named by the argument.
     */
    public String getAttribute(String attrName);

    /**
     * Returns the names of this <code>Node</code>'s attributes.
     * If the node represents character data the returned array is empty.
     *
     * @return  The attribute names of the node.
     */
    public String[] getAttributeNames();

    /**
     * Returns the children of this <code>Node</code>.
     * If the node represents character data the empty list is returned.
     *
     * @return  An immutable list of the children of this <code>Node</code>.
     */
    public List<? extends Node> getChildren();

    /**
     * Returns <code>true</code> if and only if the specified object is also a
     * <code>Node</code>, they have the same type, their node values are equal,
     * their attributes are equals as a set, and their children are recursively
     * equal according to this definition.
     */
    public boolean equals(Object o);

    /**
     * Recursively computes a hash code based on the node type, node value,
     * attributes, and children.
     */
    public int hashCode();
}
