package org.planx.msd.list;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import org.planx.msd.*;
import org.planx.msd.array.*;
import org.planx.msd.util.*;
import org.planx.util.Association;

/**
 * A <code>Discriminator</code> capable of discriminating a multiset of
 * <code>List</code>s considered as bags. That is, two lists are considered
 * equivalent if one is a permutation of the other.
 * <p>
 * This implementation uses a <code>ShortArrayDiscriminator</code> to
 * discriminate the lists after they have been weak sorted and elements
 * replaced by <code>short</code> tokens. This gives the restriction
 * that there must be at most 2^16 different elements
 * (equivalence classes) in the input.
 * <p>
 * <b>Note that this implementation is not synchronized.</b> If multiple
 * threads access an instance of this class concurrently, it must be
 * synchronized externally.
 *
 * @author Thomas Ambus
 */
public class ShortBagDiscriminator<T> extends AbstractDiscriminator<List<T>> {
    private Discriminator<T> disc;
    private Discriminator<short[]> listDisc;
    private boolean doRemoveDuplicates;

    /**
     * Constructs a new <code>ShortBagDiscriminator</code> where the
     * specified <code>Discriminator</code> is capable of discriminating
     * the elements contained in the bags. The instance will reuse the
     * memory of the specified <code>Memory</code>.
     */
    public ShortBagDiscriminator(Discriminator<T> d, Memory memory) {
        this(d, memory, false);
    }

    ShortBagDiscriminator(Discriminator<T> d, Memory memory,
                                boolean doRemoveDuplicates) {
        disc = d;
        listDisc = new ShortArrayDiscriminator(memory);
        this.doRemoveDuplicates = doRemoveDuplicates;
    }

    public <U,S> Collection<List<S>> discriminate(List<? extends U> values,
                                        Extractor<U,? extends List<T>,S> e) {

        // Create list of all contained elements and make total order

        int insize = values.size();
        List<IWrap<T>> in = new ArrayList<IWrap<T>>();
        List<Association<short[],S>> sorted =
            new ArrayList<Association<short[],S>>(insize);

        for (int i=0; i<insize; i++) {
            // Create input
            U elm = values.get(i);
            List<T> list = e.getLabel(elm);
            for (T t : list) {
                in.add(new IWrap<T>(t,i));
            }
            // Prepare result
            short[] newList = new short[list.size()];
            S value = e.getValue(elm);
            sorted.add(new Association<short[],S>(newList,value));
        }

        Extractor<IWrap<T>,T,IWrap<T>> ext1 =
                new Extractor<IWrap<T>,T,IWrap<T>>() {
            public T getLabel(IWrap<T> elm) {
                return elm.obj;
            }
            public IWrap<T> getValue(IWrap<T> elm) {
                return elm;
            }
        };
        Collection<List<IWrap<T>>> order = disc.discriminate(in, ext1);

        // Loop through contained elements in order and re-assign them to
        // lists in this order

        if (order.size() > 1<<Short.SIZE)
            throw new IndexOutOfBoundsException(
                "Too many distinct equivalence classes");

        int[] sizes = new int[insize];
        short unique = 0;
        for (List<IWrap<T>> group : order) {
            int prevIdx = -1;
            for (IWrap<T> wrap : group) {
                int idx = wrap.i;
                if (!doRemoveDuplicates || (doRemoveDuplicates &&
                                                    idx != prevIdx)) {
                    short[] list = sorted.get(idx).getFirst();
                    list[sizes[idx]++] = unique;
                }
                prevIdx = idx;
            }
            unique++;
        }

        // Unfortunately some of the short arrays might be too big
        // if duplicates were removed

        if (doRemoveDuplicates) {
            for (int i=0; i<insize; i++) {
                Association<short[],S> pair = sorted.get(i);
                short[] ss = pair.getFirst();
                int n = sizes[i];
                if (n < ss.length) {
                    short[] ns = new short[n];
                    System.arraycopy(ss,0,ns,0,n);
                    pair.setFirst(ns);
                }
            }
        }

        // List discrimination using shorts as tokens for T

        Extractor<Association<short[],S>,short[],S> ext2 =
            Discriminators.pairExtractor();
        return listDisc.discriminate(sorted, ext2);
    }

    private static class IWrap<T> {
        T obj;
        int i;
        public IWrap(T obj, int i) {
            this.obj=obj;
            this.i=i;
        }
    }
}
