package org.planx.msd.list;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import org.planx.msd.*;
import org.planx.msd.number.IntegerDiscriminator;
import org.planx.util.Association;
import org.planx.util.Pair;
import org.planx.msd.util.*;

/**
 * A <code>Discriminator</code> capable of discriminating a multiset of
 * <code>List</code>s considered as bags. That is, two lists are considered
 * equivalent if one is a permutation of the other.
 * <p>
 * <b>Note that this implementation is not synchronized.</b> If multiple
 * threads access an instance of this class concurrently, it must be
 * synchronized externally.
 *
 * @author Thomas Ambus
 */
public class BagDiscriminator<T> extends AbstractDiscriminator<List<T>> {
    private Discriminator<T> disc;
    private Discriminator<List<Integer>> listDisc;
    private boolean doRemoveDuplicates;

    /**
     * Constructs a new <code>BagDiscriminator</code> where the specified
     * <code>Discriminator</code>
     * is capable of discriminating the elements contained in the bags.
     * The instance will reuse the memory of the specified <code>Memory</code>.
     */
    public BagDiscriminator(Discriminator<T> d, Memory memory) {
        this(d, memory, false);
    }

    BagDiscriminator(Discriminator<T> d, Memory memory,
                            boolean doRemoveDuplicates) {
        disc = d;
        listDisc = new SizeRandomAccessListDiscriminator<Integer>(
                         new IntegerDiscriminator(memory), memory);
        this.doRemoveDuplicates = doRemoveDuplicates;
    }

    public <U,S> Collection<List<S>> discriminate(List<? extends U> values,
                                        Extractor<U,? extends List<T>,S> e) {

        // Create list of all contained elements and make total order
        int insize = values.size();
        List<IWrap<T>> in = new ArrayList<IWrap<T>>();
        List<Pair<List<Integer>,S>> sorted =
            new ArrayList<Pair<List<Integer>,S>>(insize);

        for (int i=0; i<insize; i++) {
            // Create input
            U elm = values.get(i);
            List<T> list = e.getLabel(elm);
            for (T t : list) {
                in.add(new IWrap<T>(t,i));
            }
            // Prepare result
            List<Integer> newList = new ArrayList<Integer>(list.size());
            S value = e.getValue(elm);
            sorted.add(new Association<List<Integer>,S>(newList,value));
        }

        Extractor<IWrap<T>,T,IWrap<T>> ext1 =
                new Extractor<IWrap<T>,T,IWrap<T>>() {
            public T getLabel(IWrap<T> elm) {
                return elm.obj;
            }
            public IWrap<T> getValue(IWrap<T> elm) {
                return elm;
            }
        };
        Collection<List<IWrap<T>>> order = disc.discriminate(in, ext1);

        // Loop through contained elements in order and re-assign
        // them to lists in this order

        int unique = 0;
        for (List<IWrap<T>> group : order) {
            int prevIdx = -1;
            Integer canon = new Integer(unique);
            for (IWrap<T> wrap : group) {
                int idx = wrap.i;
                if (!doRemoveDuplicates || (doRemoveDuplicates &&
                                            idx != prevIdx)) {
                    Pair<List<Integer>,S> p = sorted.get(idx);
                    List<Integer> list = p.getFirst();
                    list.add(canon);
                }
                prevIdx = idx;
            }
            unique++;
        }

        // List discrimination using integers as tokens for T

        Extractor<Pair<List<Integer>,S>,List<Integer>,S> ext2 =
            Discriminators.pairExtractor();
        return listDisc.discriminate(sorted, ext2);
    }

    private static class IWrap<T> {
        T obj;
        int i;
        public IWrap(T obj, int i) {
            this.obj=obj;
            this.i=i;
        }
    }
}
