/*
 * Decompiled with CFR 0.152.
 */
package org.openrewrite.jgit.internal.storage.file;

import java.io.File;
import java.io.IOException;
import java.nio.file.NoSuchFileException;
import java.nio.file.attribute.BasicFileAttributes;
import java.time.Duration;
import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.Locale;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import org.openrewrite.jgit.annotations.NonNull;
import org.openrewrite.jgit.util.FS;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileSnapshot {
    private static final Logger LOG = LoggerFactory.getLogger(FileSnapshot.class);
    public static final long UNKNOWN_SIZE = -1L;
    private static final Instant UNKNOWN_TIME = Instant.ofEpochMilli(-1L);
    private static final Object MISSING_FILEKEY = new Object();
    private static final DateTimeFormatter dateFmt = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss.nnnnnnnnn").withLocale(Locale.getDefault()).withZone(ZoneId.systemDefault());
    public static final FileSnapshot DIRTY = new FileSnapshot(UNKNOWN_TIME, UNKNOWN_TIME, -1L, Duration.ZERO, MISSING_FILEKEY);
    public static final FileSnapshot MISSING_FILE = new FileSnapshot(Instant.EPOCH, Instant.EPOCH, 0L, Duration.ZERO, MISSING_FILEKEY){

        @Override
        public boolean isModified(File path) {
            return FS.DETECTED.exists(path);
        }
    };
    private final Instant lastModified;
    private volatile Instant lastRead;
    private boolean cannotBeRacilyClean;
    private final long size;
    private FS.FileStoreAttributes fileStoreAttributeCache;
    private boolean useConfig;
    private final Object fileKey;
    private final File file;
    private boolean sizeChanged;
    private boolean fileKeyChanged;
    private boolean lastModifiedChanged;
    private boolean wasRacyClean;
    private long delta;
    private long racyThreshold;

    public static FileSnapshot save(File path) {
        return new FileSnapshot(path);
    }

    public static FileSnapshot saveNoConfig(File path) {
        return new FileSnapshot(path, false);
    }

    private static Object getFileKey(BasicFileAttributes fileAttributes) {
        Object fileKey = fileAttributes.fileKey();
        return fileKey == null ? MISSING_FILEKEY : fileKey;
    }

    @Deprecated
    public static FileSnapshot save(long modified) {
        Instant read = Instant.now();
        return new FileSnapshot(read, Instant.ofEpochMilli(modified), -1L, FS.FileStoreAttributes.FALLBACK_TIMESTAMP_RESOLUTION, MISSING_FILEKEY);
    }

    public static FileSnapshot save(Instant modified) {
        Instant read = Instant.now();
        return new FileSnapshot(read, modified, -1L, FS.FileStoreAttributes.FALLBACK_TIMESTAMP_RESOLUTION, MISSING_FILEKEY);
    }

    protected FileSnapshot(File file) {
        this(file, true);
    }

    protected FileSnapshot(File file, boolean useConfig) {
        this.file = file;
        this.lastRead = Instant.now();
        this.useConfig = useConfig;
        BasicFileAttributes fileAttributes = null;
        try {
            fileAttributes = FS.DETECTED.fileAttributes(file);
        }
        catch (NoSuchFileException e) {
            this.lastModified = Instant.EPOCH;
            this.size = 0L;
            this.fileKey = MISSING_FILEKEY;
            return;
        }
        catch (IOException e) {
            LOG.error(e.getMessage(), (Throwable)e);
            this.lastModified = Instant.EPOCH;
            this.size = 0L;
            this.fileKey = MISSING_FILEKEY;
            return;
        }
        this.lastModified = fileAttributes.lastModifiedTime().toInstant();
        this.size = fileAttributes.size();
        this.fileKey = FileSnapshot.getFileKey(fileAttributes);
        if (LOG.isDebugEnabled()) {
            LOG.debug("file={}, create new FileSnapshot: lastRead={}, lastModified={}, size={}, fileKey={}", new Object[]{file, dateFmt.format(this.lastRead), dateFmt.format(this.lastModified), this.size, this.fileKey.toString()});
        }
    }

    private FileSnapshot(Instant read, Instant modified, long size, @NonNull Duration fsTimestampResolution, @NonNull Object fileKey) {
        this.file = null;
        this.lastRead = read;
        this.lastModified = modified;
        this.fileStoreAttributeCache = new FS.FileStoreAttributes(fsTimestampResolution);
        this.size = size;
        this.fileKey = fileKey;
    }

    @Deprecated
    public long lastModified() {
        return this.lastModified.toEpochMilli();
    }

    public Instant lastModifiedInstant() {
        return this.lastModified;
    }

    public long size() {
        return this.size;
    }

    public boolean isModified(File path) {
        Object currFileKey;
        long currSize;
        Instant currLastModified;
        try {
            BasicFileAttributes fileAttributes = FS.DETECTED.fileAttributes(path);
            currLastModified = fileAttributes.lastModifiedTime().toInstant();
            currSize = fileAttributes.size();
            currFileKey = FileSnapshot.getFileKey(fileAttributes);
        }
        catch (NoSuchFileException e) {
            currLastModified = Instant.EPOCH;
            currSize = 0L;
            currFileKey = MISSING_FILEKEY;
        }
        catch (IOException e) {
            LOG.error(e.getMessage(), (Throwable)e);
            currLastModified = Instant.EPOCH;
            currSize = 0L;
            currFileKey = MISSING_FILEKEY;
        }
        this.sizeChanged = this.isSizeChanged(currSize);
        if (this.sizeChanged) {
            return true;
        }
        this.fileKeyChanged = this.isFileKeyChanged(currFileKey);
        if (this.fileKeyChanged) {
            return true;
        }
        this.lastModifiedChanged = this.isModified(currLastModified);
        return this.lastModifiedChanged;
    }

    public void setClean(FileSnapshot other) {
        Instant now = other.lastRead;
        if (!this.isRacyClean(now)) {
            this.cannotBeRacilyClean = true;
        }
        this.lastRead = now;
    }

    public void waitUntilNotRacy() throws InterruptedException {
        long timestampResolution = this.fileStoreAttributeCache().getFsTimestampResolution().toNanos();
        while (this.isRacyClean(Instant.now())) {
            TimeUnit.NANOSECONDS.sleep(timestampResolution);
        }
    }

    public boolean equals(FileSnapshot other) {
        boolean sizeEq = this.size == -1L || other.size == -1L || this.size == other.size;
        return this.lastModified.equals(other.lastModified) && sizeEq && Objects.equals(this.fileKey, other.fileKey);
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FileSnapshot)) {
            return false;
        }
        FileSnapshot other = (FileSnapshot)obj;
        return this.equals(other);
    }

    public int hashCode() {
        return Objects.hash(this.lastModified, this.size, this.fileKey);
    }

    boolean wasSizeChanged() {
        return this.sizeChanged;
    }

    boolean wasFileKeyChanged() {
        return this.fileKeyChanged;
    }

    boolean wasLastModifiedChanged() {
        return this.lastModifiedChanged;
    }

    boolean wasLastModifiedRacilyClean() {
        return this.wasRacyClean;
    }

    public long lastDelta() {
        return this.delta;
    }

    public long lastRacyThreshold() {
        return this.racyThreshold;
    }

    public String toString() {
        if (this == DIRTY) {
            return "DIRTY";
        }
        if (this == MISSING_FILE) {
            return "MISSING_FILE";
        }
        return "FileSnapshot[modified: " + dateFmt.format(this.lastModified) + ", read: " + dateFmt.format(this.lastRead) + ", size:" + this.size + ", fileKey: " + this.fileKey + "]";
    }

    private boolean isRacyClean(Instant read) {
        this.racyThreshold = this.getEffectiveRacyThreshold();
        this.delta = Duration.between(this.lastModified, read).toNanos();
        boolean bl = this.wasRacyClean = this.delta <= this.racyThreshold;
        if (LOG.isDebugEnabled()) {
            LOG.debug("file={}, isRacyClean={}, read={}, lastModified={}, delta={} ns, racy<={} ns", new Object[]{this.file, this.wasRacyClean, dateFmt.format(read), dateFmt.format(this.lastModified), this.delta, this.racyThreshold});
        }
        return this.wasRacyClean;
    }

    private long getEffectiveRacyThreshold() {
        long minRacyInterval;
        long timestampResolution = this.fileStoreAttributeCache().getFsTimestampResolution().toNanos();
        long max = Math.max(timestampResolution, minRacyInterval = this.fileStoreAttributeCache().getMinimalRacyInterval().toNanos());
        return max < 100000000L ? max * 5L / 2L : max * 5L / 4L;
    }

    private boolean isModified(Instant currLastModified) {
        boolean bl = this.lastModifiedChanged = !this.lastModified.equals(currLastModified);
        if (this.lastModifiedChanged) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("file={}, lastModified changed from {} to {}", new Object[]{this.file, dateFmt.format(this.lastModified), dateFmt.format(currLastModified)});
            }
            return true;
        }
        if (this.cannotBeRacilyClean) {
            LOG.debug("file={}, cannot be racily clean", (Object)this.file);
            return false;
        }
        if (!this.isRacyClean(this.lastRead)) {
            LOG.debug("file={}, is unmodified", (Object)this.file);
            return false;
        }
        LOG.debug("file={}, is racily clean", (Object)this.file);
        return true;
    }

    private boolean isFileKeyChanged(Object currFileKey) {
        boolean changed;
        boolean bl = changed = currFileKey != MISSING_FILEKEY && !currFileKey.equals(this.fileKey);
        if (changed) {
            LOG.debug("file={}, FileKey changed from {} to {}", new Object[]{this.file, this.fileKey, currFileKey});
        }
        return changed;
    }

    private boolean isSizeChanged(long currSize) {
        boolean changed;
        boolean bl = changed = currSize != -1L && currSize != this.size;
        if (changed) {
            LOG.debug("file={}, size changed from {} to {} bytes", new Object[]{this.file, this.size, currSize});
        }
        return changed;
    }

    private FS.FileStoreAttributes fileStoreAttributeCache() {
        if (this.fileStoreAttributeCache == null) {
            this.fileStoreAttributeCache = this.useConfig ? FS.getFileStoreAttributes(this.file.toPath().getParent()) : FS.FileStoreAttributes.FALLBACK_FILESTORE_ATTRIBUTES;
        }
        return this.fileStoreAttributeCache;
    }
}

