/* *##%
 * Copyright (C) 2010 Code Lutin, Chatellier Eric
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package org.nuiton.wikitty;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.Set;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.ListenerSet;

/**
 * Wikitty service notifier.
 * 
 * Currently based on jgroups.
 * 
 * @author chatellier
 * @version $Revision: 161 $
 * 
 * Last update : $Date: 2010-06-28 17:56:28 +0200 (lun., 28 juin 2010) $
 * By : $Author: echatellier $
 */
public class WikittyServiceNotifier implements WikittyService {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(WikittyServiceNotifier.class);

    /**
     * Utilisation du canal de communication basé sur jgroups avec
     * comme identifiant d'application le nom de canal en option.
     * 
     * Si ce nom est vide, jgroups n'est pas utilisé.
     * Si {@link #WIKITTY_EVENT_PROPAGATE_OPTION} est a true et que cette
     * options est vide, une exception est levée.
     */
    static public final String WIKITTY_EVENT_JGROUPCHANNELNAME_OPTION = "wikitty.service.event.jgroupschannelname";

    /**
     * Indique si les objects sont propages (true) vers les autres caches ou
     * simplement supprimes des autres caches (false). Default to {@code false}.
     */
    static public final String WIKITTY_EVENT_PROPAGATE_OPTION = "wikitty.service.event.propagateEvent";

    /**
     * Indique si les evenement sont propagés aux autres listener. Sinon on est
     * juste listener des evenements recus. C'est pourquoi le listener
     * {@link #notifier} doit toujours être ajouté.
     * 
     * @see WikittyServiceNotifier#WIKITTY_EVENT_PROPAGATE_OPTION
     */
    protected boolean propagateCache = false;

    /** Service to delegate. */
    protected WikittyService ws;

    /** Wikitty service listener (all event). */
    protected ListenerSet<WikittyServiceListener> allWikittyServiceListeners;

    /** Wikitty service listener (only for local event). */
    protected ListenerSet<WikittyServiceListener> localWikittyServiceListeners;

    /** Wikitty service listener (only for remote event). */
    protected ListenerSet<WikittyServiceListener> remoteWikittyServiceListeners;

    /** JGroup notifier. */
    protected WikittyServiceListener notifier;

    /**
     * Default constructor.
     * 
     * @param ws delegate service
     */
    public WikittyServiceNotifier(WikittyService ws) {
        this(ws, null);
    }

    /**
     * Constructor with configuration.
     * 
     * @param ws delegate service
     * @param props properties (can be null)
     */
    public WikittyServiceNotifier(WikittyService ws, Properties props) {
        // service
        this.ws = ws;

        // listeners
        allWikittyServiceListeners = new ListenerSet<WikittyServiceListener>();
        localWikittyServiceListeners = new ListenerSet<WikittyServiceListener>();
        remoteWikittyServiceListeners = new ListenerSet<WikittyServiceListener>();
        
        addJGroupNotifier(props);
    }

    /**
     * Add jgroup listener if enabled by configuration.
     * 
     * @param props properties (can be null)
     */
    protected void addJGroupNotifier(Properties props) {

        // can be null according to default constructor
        if (props != null) {
            
            // get propagate cache property
            String propagateCacheOption = props.getProperty(WIKITTY_EVENT_PROPAGATE_OPTION, "false");
            propagateCache = "true".equalsIgnoreCase(propagateCacheOption);
            if (log.isDebugEnabled()) {
                log.debug("Set propagateCache option to " + propagateCache);
            }
            
            // add notifier as listener
            String jgroupChannel = props.getProperty(WIKITTY_EVENT_JGROUPCHANNELNAME_OPTION);
            if (!StringUtils.isBlank(jgroupChannel)) {
                notifier = new JGroupsNotifier(this, jgroupChannel, propagateCache);
                addWikittyServiceListener(notifier, ServiceListenerType.ALL); // weak reference
            }
            else if (propagateCache) {
                throw new IllegalArgumentException("Can't use propagate cache without a valid jgroups channel name !!!");
            }
            else if (log.isDebugEnabled()) {
                log.debug("JGroup synchronisation channel not used ");
            }
        }

    }

    @Override
    public void addWikittyServiceListener(WikittyServiceListener listener, ServiceListenerType type) {
        // not delegated
        switch (type) {
            case ALL : allWikittyServiceListeners.add(listener); break;
            case LOCAL : localWikittyServiceListeners.add(listener); break;
            case REMOTE : remoteWikittyServiceListeners.add(listener); break;
        }
    }

    @Override
    public void removeWikittyServiceListener(WikittyServiceListener listener, ServiceListenerType type) {
        // not delegated
        switch (type) {
            case ALL : allWikittyServiceListeners.remove(listener); break;
            case LOCAL : localWikittyServiceListeners.remove(listener); break;
            case REMOTE : remoteWikittyServiceListeners.remove(listener); break;
        }
    }

    @Override
    public String login(String login, String password) {
        return ws.login(login, password);
    }

    @Override
    public void logout(String securityToken) {
        ws.logout(securityToken);
    }

    @Override
    public boolean canWrite(String securityToken, Wikitty wikitty) {
        return ws.canWrite(securityToken, wikitty);
    }

    @Override
    public boolean canDelete(String securityToken, String wikittyId) {
        return ws.canDelete(securityToken, wikittyId);
    }

    @Override
    public boolean canRead(String securityToken, String wikittyId) {
        return ws.canRead(securityToken, wikittyId);
    }

    public void clear(String securityToken) {
        ws.clear(securityToken);
    }

    @Override
    public UpdateResponse store(String securityToken, Wikitty wikitty) {
        UpdateResponse updateResponse = ws.store(securityToken, wikitty);
        // update wikitty
        updateResponse.update(wikitty);
        // notify listeners
        firePutWikitty(wikitty);
        return updateResponse;
    }

    @Override
    public UpdateResponse store(String securityToken, Collection<Wikitty> wikitties) {
        UpdateResponse updateResponse = ws.store(securityToken, wikitties);
        // update wikitty
        for (Wikitty wikitty : wikitties) {
            updateResponse.update(wikitty);
        }
        // notify listeners
        firePutWikitty(wikitties.toArray(new Wikitty[wikitties.size()]));
        return updateResponse;
    }

    @Override
    public UpdateResponse store(String securityToken, Collection<Wikitty> wikitties,
            boolean disableAutoVersionIncrement) {
        UpdateResponse updateResponse = ws.store(
                securityToken, wikitties, disableAutoVersionIncrement);
        // update wikitty
        for (Wikitty wikitty : wikitties) {
            updateResponse.update(wikitty);
        }
        // notify listeners
        firePutWikitty(wikitties.toArray(new Wikitty[wikitties.size()]));
        return updateResponse;
    }

    @Override
    public UpdateResponse store(String securityToken, WikittyTransaction transaction,
            Collection<Wikitty> wikitties, boolean disableAutoVersionIncrement) {
        UpdateResponse updateResponse = ws.store(securityToken, transaction,
                wikitties, disableAutoVersionIncrement);
        // update wikitty
        for (Wikitty wikitty : wikitties) {
            updateResponse.update(wikitty);
        }
        // notify listeners
        firePutWikitty(wikitties.toArray(new Wikitty[wikitties.size()]));
        return updateResponse;
    }

    @Override
    public List<String> getAllExtensionIds(String securityToken) {
        // no notification
        return ws.getAllExtensionIds(securityToken);
    }

    @Override
    public List<String> getAllExtensionsRequires(
            String securityToken, String extensionName) {
        // no notification
        return ws.getAllExtensionsRequires(securityToken, extensionName);
    }

    @Override
    public UpdateResponse storeExtension(String securityToken, WikittyExtension ext) {
        UpdateResponse updateResponse = ws.storeExtension(securityToken, ext);
        // notify listeners
        firePutExtension(ext);
        return updateResponse;
    }

    @Override
    public UpdateResponse storeExtension(
            String securityToken, Collection<WikittyExtension> exts) {
        UpdateResponse updateResponse = ws.storeExtension(securityToken, exts);
        // notify listeners
        firePutExtension(exts.toArray(new WikittyExtension[exts.size()]));
        return updateResponse;
    }

    @Override
    public UpdateResponse storeExtension(String securityToken, 
            WikittyTransaction transaction, Collection<WikittyExtension> exts) {
        UpdateResponse updateResponse = ws.storeExtension(
                securityToken, transaction, exts);
        // no notification called by #storeExtension(Collection<WikittyExtension>)
        return updateResponse;
    }

    @Override
    public WikittyExtension restoreExtension(String securityToken, String id) {
        // no notification
        return ws.restoreExtension(securityToken, id);
    }

    @Override
    public WikittyExtension restoreExtension(String securityToken, 
            WikittyTransaction transaction, String id) {
        // no notification
        return ws.restoreExtension(securityToken, transaction, id);
    }

    @Override
    public WikittyExtension restoreExtensionLastVersion(
            String securityToken, String name) {
        // no notification
        return ws.restoreExtensionLastVersion(securityToken, name);
    }

    @Override
    public WikittyExtension restoreExtensionLastVersion(
            String securityToken, WikittyTransaction transaction, String name) {
        // no notification
        return ws.restoreExtensionLastVersion(securityToken, transaction, name);
    }

    @Override
    public Wikitty restore(String securityToken, String id) {
        // no notification
        return ws.restore(securityToken, id);
    }

    @Override
    public List<Wikitty> restore(String securityToken, List<String> ids) {
        // no notification
        return ws.restore(securityToken, ids);
    }

    @Override
    public List<Wikitty> restore(String securityToken, WikittyTransaction transaction, List<String> ids) {
        // no notification
        return ws.restore(securityToken, transaction, ids);
    }

    @Override
    public void delete(String securityToken, String id) {
        ws.delete(securityToken, id);
        // notify listeners
        fireRemoveWikitty(id);
    }

    @Override
    public void delete(String securityToken, Collection<String> ids) {
        ws.delete(securityToken, ids);
        // notify listeners
        fireRemoveWikitty(ids.toArray(new String[ids.size()]));
    }

    @Override
    public PagedResult<String> findAllByCriteria(String securityToken, Criteria criteria) {
        // no notification
        return ws.findAllByCriteria(securityToken, criteria);
    }

    @Override
    public PagedResult<String> findAllByCriteria(String securityToken,
            WikittyTransaction transaction, Criteria criteria) {
        // no notification
        return ws.findAllByCriteria(securityToken, transaction, criteria);
    }

    @Override
    public Wikitty findByCriteria(String securityToken, Criteria criteria) {
        // no notification
        return ws.findByCriteria(securityToken, criteria);
    }

    @Override
    public void addLabel(String securityToken, String wikittyId, String label) {
        // no notification
        // TODO EC20100607 fixme : need notification ?
        ws.addLabel(securityToken, wikittyId, label);
    }

    @Override
    public PagedResult<String> findAllByLabel(String securityToken,
            String label, int firstIndex, int endIndex) {
        // no notification
        return ws.findAllByLabel(securityToken, label, firstIndex, endIndex);
    }

    @Override
    public Wikitty findByLabel(String securityToken, String label) {
        // no notification
        return ws.findByLabel(securityToken, label);
    }

    @Override
    public Set<String> findAllAppliedLabels(String securityToken, String wikittyId) {
        // no notification
        return ws.findAllAppliedLabels(securityToken, wikittyId);
    }

    @Override
    public Tree restoreTree(String securityToken, String wikittyId) {
        // no notification
        return ws.restoreTree(securityToken, wikittyId);
    }

    @Override
    public Entry<TreeNode, Integer> restoreNode(String securityToken, String wikittyId,
            Criteria filter) {
        // no notification
        return ws.restoreNode(securityToken, wikittyId, filter);
    }

    @Override
    public Map<TreeNode, Integer> restoreChildren(String securityToken, 
            String wikittyId, Criteria filter) {
        // no notification
        return ws.restoreChildren(securityToken, wikittyId, filter);
    }

    @Override
    public Wikitty restoreVersion(
            String securityToken, String wikittyId, String version) {
        // no notification
        return ws.restoreVersion(securityToken, wikittyId, version);
    }

    @Override
    public UpdateResponse syncEngin(String securityToken) {
        // no notification
        return ws.syncEngin(securityToken);
    }

    /**
     * Build event to fire and call {@link #firePutWikitty(WikittyServiceEvent)}.
     *
     * @param ws data
     */
    protected void firePutWikitty(Wikitty... ws) {
        WikittyServiceEvent event = new WikittyServiceEvent(ws);
        Set<String> ids = new HashSet<String>();
        Map<String, Set<String>> idsExtension = new HashMap<String, Set<String>>();
        Map<String, String> idsVersion = new HashMap<String, String>();
        
        for (Wikitty w : ws) {
            // ids
            ids.add(w.getId());
            // extension
            Set<String> extension = new HashSet<String>();
            extension.addAll(w.getExtensionNames());
            idsExtension.put(w.getId(), extension);
            // version
            idsVersion.put(w.getId(), w.getVersion());
        }
        
        event.setIds(ids);
        event.setIdExtensions(idsExtension);
        event.setIdVersions(idsVersion);
        firePutWikitty(event);
    }

    /**
     * Build event to fire and call {@link #fireRemoveWikitty(WikittyServiceEvent)}.
     *
     * @param wikittyIds wikitty ids
     */
    protected void fireRemoveWikitty(String... wikittyIds) {
        WikittyServiceEvent event = new WikittyServiceEvent(ws);
        Set<String> ids = new HashSet<String>();
        
        for (String wikittyId : wikittyIds) {
            // ids
            ids.add(wikittyId);
        }
        
        event.setIds(ids);
        fireRemoveWikitty(event);
    }

    /**
     * Build event to fire and call {@link #fireClearWikitty(WikittyServiceEvent)}.
     */
    protected void fireClearWikitty() {
        WikittyServiceEvent event = new WikittyServiceEvent(ws);
        fireClearWikitty(event);
    }

    /**
     * Build event to fire and call {@link #firePutExtension(WikittyServiceEvent)}.
     *
     * @param ws data
     */
    protected void firePutExtension(WikittyExtension... exts) {
        WikittyServiceEvent event = new WikittyServiceEvent(ws);
        Set<String> ids = new HashSet<String>();
        Map<String, Set<String>> idsExtension = new HashMap<String, Set<String>>();

        for (WikittyExtension ext : exts) {
            // ids
            ids.add(ext.getId());
            // extension
            Set<String> extension = new HashSet<String>();
            extension.add(ext.getName());

            // also add requires fields
            extension.add(ext.requires);

            idsExtension.put(ext.getId(), extension);
        }

        event.setIds(ids);
        event.setIdExtensions(idsExtension);
        firePutExtension(event);
    }

    /**
     * Build event to fire and call {@link #fireRemoveExtension(WikittyServiceEvent)}.
     *
     * @param ws data
     */
    protected void fireRemoveExtension(WikittyExtension... exts) {
        WikittyServiceEvent event = new WikittyServiceEvent(ws);
        Set<String> ids = new HashSet<String>();
        
        for (WikittyExtension ext : exts) {
            // ids
            ids.add(ext.getId());
        }
        
        event.setIds(ids);
        fireRemoveExtension(event);
    }

    /**
     * Build event to fire and call {@link #fireClearExtension(WikittyServiceEvent)}.
     *
     * @param ws data
     */
    protected void fireClearExtension() {
        WikittyServiceEvent event = new WikittyServiceEvent(ws);
        fireClearExtension(event);
    }

    /**
     * Fire event to all registred listener.
     * 
     * Take care about {@link WikittyServiceEvent#isRemote()} for fire.
     *
     * @param event event to fire
     */
    protected void firePutWikitty(WikittyServiceEvent event) {
        for (WikittyServiceListener l : allWikittyServiceListeners) {
            l.putWikitty(event);
        }
        if (event.isRemote()) {
            for (WikittyServiceListener l : localWikittyServiceListeners) {
                l.putWikitty(event);
            }
        }
        else {
            for (WikittyServiceListener l : remoteWikittyServiceListeners) {
                l.putWikitty(event);
            }
        }
    }

    /**
     * Fire event to all registred listener.
     * 
     * Take care about {@link WikittyServiceEvent#isRemote()} for fire.
     *
     * @param event event to fire
     */
    protected void fireRemoveWikitty(WikittyServiceEvent event) {
        for (WikittyServiceListener l : allWikittyServiceListeners) {
            l.removeWikitty(event);
        }
        if (event.isRemote()) {
            for (WikittyServiceListener l : localWikittyServiceListeners) {
                l.removeWikitty(event);
            }
        }
        else {
            for (WikittyServiceListener l : remoteWikittyServiceListeners) {
                l.removeWikitty(event);
            }
        }
    }

    /**
     * Fire event to all registred listener.
     * 
     * Take care about {@link WikittyServiceEvent#isRemote()} for fire.
     *
     * @param event event to fire
     */
    protected void fireClearWikitty(WikittyServiceEvent event) {
        for (WikittyServiceListener l : allWikittyServiceListeners) {
            l.clearWikitty(event);
        }
        if (event.isRemote()) {
            for (WikittyServiceListener l : localWikittyServiceListeners) {
                l.clearWikitty(event);
            }
        }
        else {
            for (WikittyServiceListener l : remoteWikittyServiceListeners) {
                l.clearWikitty(event);
            }
        }
    }

    /**
     * Fire event to all registred listener.
     * 
     * Take care about {@link WikittyServiceEvent#isRemote()} for fire.
     *
     * @param event event to fire
     */
    protected void firePutExtension(WikittyServiceEvent event) {
        for (WikittyServiceListener l : allWikittyServiceListeners) {
            l.putExtension(event);
        }
        if (event.isRemote()) {
            for (WikittyServiceListener l : localWikittyServiceListeners) {
                l.putExtension(event);
            }
        }
        else {
            for (WikittyServiceListener l : remoteWikittyServiceListeners) {
                l.putExtension(event);
            }
        }
    }

    /**
     * Fire event to all registred listener.
     * 
     * Take care about {@link WikittyServiceEvent#isRemote()} for fire.
     *
     * @param event event to fire
     */
    protected void fireRemoveExtension(WikittyServiceEvent event) {
        for (WikittyServiceListener l : allWikittyServiceListeners) {
            l.removeExtension(event);
        }
        if (event.isRemote()) {
            for (WikittyServiceListener l : localWikittyServiceListeners) {
                l.removeExtension(event);
            }
        }
        else {
            for (WikittyServiceListener l : remoteWikittyServiceListeners) {
                l.removeExtension(event);
            }
        }
    }

    /**
     * Fire event to all registred listener.
     * 
     * Take care about {@link WikittyServiceEvent#isRemote()} for fire.
     *
     * @param event event to fire
     */
    protected void fireClearExtension(WikittyServiceEvent event) {
        for (WikittyServiceListener l : allWikittyServiceListeners) {
            l.clearExtension(event);
        }
        if (event.isRemote()) {
            for (WikittyServiceListener l : localWikittyServiceListeners) {
                l.clearExtension(event);
            }
        }
        else {
            for (WikittyServiceListener l : remoteWikittyServiceListeners) {
                l.clearExtension(event);
            }
        }
    }
}
