package org.nuiton.topia.persistence.support;

/*
 * #%L
 * ToPIA :: Persistence
 * $Id: TopiaJpaSupport.java 2981 2014-01-17 17:38:55Z athimel $
 * $HeadURL: https://svn.nuiton.org/topia/tags/topia-3.0-beta-7/topia-persistence/src/main/java/org/nuiton/topia/persistence/support/TopiaJpaSupport.java $
 * %%
 * Copyright (C) 2004 - 2014 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import java.util.List;
import java.util.Map;

/**
 * This API provides methods to use persistence using JPA queries
 *
 * @author Arnaud Thimel : thimel@codelutin.com
 * @since 3.0
 */
public interface TopiaJpaSupport {

    /**
     * Allow to do some JPA-QL query
     * <p/>
     * WARNING : Depending on the registered service, this method may not
     * support something else than queries on TopiaEntity
     *
     * @param jpaql      the JPA-QL query
     * @param parameters a map which keys are the attribute names and values are the attributes expected values
     * @return The result list
     */
    <T> List<T> findAll(String jpaql,
                        Map<String, Object> parameters);

    /**
     * Allow to do some JPA-QL query using the given bounds.
     * <p/>
     * No lower bound : <code>startIndex</code> = 0.<br/>
     * No upper bound : <code>endIndex</code> = -1.
     * <p/>
     * WARNING : Depending on the registered service, this method may not
     * support something else than queries on TopiaEntity
     *
     * @param jpaql      the JPA-QL query
     * @param startIndex first index of entity to return
     * @param endIndex   last index of entity to return
     * @param parameters a map which keys are the attribute names and values are the attributes expected values
     * @return The result list
     */
    <T> List<T> find(String jpaql,
                     int startIndex,
                     int endIndex,
                     Map<String, Object> parameters);

    /**
     * Allow to do some JPA-QL query and return an unique result. If nothing if
     * found by the query, will return null. If more than one result is found,
     * will throw an exception.
     * <p/>
     * WARNING : Depending on the registered service, this method may not
     * support something else than queries on TopiaEntity
     *
     * @param jpaql      the JPA-QL query
     * @param parameters a map which keys are the attribute names and values are the attributes expected values
     * @return The result instance or null
     */
    <T> T findUnique(String jpaql,
                     Map<String, Object> parameters);

    /**
     * Execute JPA-QL operation on data (Update, Delete).
     *
     * @param jpaql      the JPA-QL query
     * @param parameters a map which keys are the attribute names and values are the attributes expected values
     * @return The number of entities updated or deleted.
     */
    int execute(String jpaql,
                Map<String, Object> parameters);

    /**
     * Tells to the context if it has to use a flush mode before each query.
     * <p/>
     * By default, we use a flush mode, but in some case it costs to much doing
     * this, that's why you can disable it setting the value to {@code false}.
     *
     * @param useFlushMode the new value to set
     * @since 2.5
     */
    void setUseFlushMode(boolean useFlushMode);

    /**
     * Persist the given transient instance, first assigning a generated identifier. This method is JPA implementation
     * independent.
     * <p/>
     * This method is "inspired" of the Hibernate's Session#save method.
     *
     * @param object a transient instance of a persistent class
     */
    void save(Object object);

    /**
     * Update the persistent instance with the identifier of the given detached instance.
     * <p/>
     * This method is "inspired" of the Hibernate's Session#update method.
     *
     * @param object a detached instance containing updated state
     */
    void update(Object object);

    /**
     * Either {@link #save(Object)} or {@link #update(Object)} the given instance.
     * <p/>
     * This method is "inspired" of the Hibernate's Session#saveOrUpdate method.
     *
     * @param object a transient or detached instance containing new or updated state
     * @see #save(java.lang.Object)
     * @see #update(Object object)
     */
    void saveOrUpdate(Object object);

    /**
     * Remove a persistent instance.
     * <p/>
     * This method is "inspired" of the Hibernate's Session#delete method.
     *
     * @param object the instance to be removed
     */
    void delete(Object object);

}
