/*
 * #%L
 * ToPIA :: Persistence
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2004 - 2014 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.topia.generator;

import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Set;

import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.eugene.models.object.ObjectModelType;
import org.nuiton.eugene.java.ObjectModelTransformerToJava;
import org.nuiton.eugene.models.object.*;
import org.nuiton.eugene.models.object.xml.ObjectModelAttributeImpl;
import org.nuiton.eugene.models.object.xml.ObjectModelEnumerationImpl;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.TopiaRuntimeException;
import org.nuiton.topia.framework.TopiaContextImplementor;
import org.nuiton.topia.persistence.TopiaDAO;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.topia.persistence.TopiaEntityEnum;
import org.nuiton.topia.persistence.util.EntityOperator;
import org.nuiton.topia.persistence.util.EntityOperatorStore;
import org.nuiton.topia.persistence.util.TopiaEntityHelper;

import java.lang.reflect.Array;
import java.util.Arrays;
import java.util.List;






/**
 * Created: 13 nov. 2009 09:05:17
 *
 * @author tchemit <chemit@codelutin.com>
 * @version $Id$
 * @since 2.3.0
 * @plexus.component role="org.nuiton.eugene.Template" role-hint="org.nuiton.topia.generator.DAOHelperTransformer"
 */
public class DAOHelperTransformer extends ObjectModelTransformerToJava {

    private static final Log log =
            LogFactory.getLog(DAOHelperTransformer.class);

    @Override
    public void transformFromModel(ObjectModel model) {
        String packageName = getDefaultPackageName();
        String modelName = model.getName();
        String daoHelperClazzName = modelName + "DAOHelper";
        String entityEnumName = modelName + "EntityEnum";

        List<ObjectModelClass> classes =
                TopiaGeneratorUtil.getEntityClasses(model, true);

        boolean generateOperator =
                TopiaGeneratorUtil.shouldGenerateOperatorForDAOHelper(model);

        boolean generateStandaloneEnum =
                TopiaGeneratorUtil.shouldGenerateStandaloneEnumForDAOHelper(model);

        ObjectModelClass daoHelper = createClass(daoHelperClazzName,
                                                 packageName);

        ObjectModelEnumeration entityEnum;

        if (generateStandaloneEnum) {
            if (log.isDebugEnabled()) {
                log.debug("Will generate standalone " + entityEnumName +
                     " in package " + packageName);
            }
            entityEnum = createEnumeration(entityEnumName, packageName);
            addImport(entityEnum, TopiaEntity.class);
            addImport(entityEnum, EntityOperatorStore.class);
            addImport(entityEnum, Arrays.class);
            addImport(entityEnum, TopiaRuntimeException.class);
            addImport(entityEnum, ArrayUtils.class);
            addImport(entityEnum, TopiaEntityHelper.class);

        } else {
            entityEnum = (ObjectModelEnumerationImpl)
                    addInnerClassifier(daoHelper,
                                       ObjectModelType.OBJECT_MODEL_ENUMERATION,
                                       entityEnumName
                    );
            addImport(daoHelper, TopiaRuntimeException.class);
            addImport(daoHelper, TopiaEntityEnum.class);
            addImport(daoHelper, EntityOperatorStore.class);
            addImport(daoHelper, Arrays.class);
            addImport(daoHelper, ArrayUtils.class);
            addImport(daoHelper, TopiaEntityHelper.class);
        }

        // generate DAOHelper
        createDAOHelper(model,
                        daoHelper,
                        daoHelperClazzName,
                        entityEnumName,
                        generateOperator,
                        classes
        );

        // generate TopiaEntityEnum
        createEntityEnum(entityEnum,
                         daoHelperClazzName,
                         entityEnumName,
                         generateOperator,
                         generateStandaloneEnum,
                         classes
        );
    }

    protected void createDAOHelper(ObjectModel model,
                                   ObjectModelClass daoHelper,
                                   String daoHelperClazzName,
                                   String entityEnumName,
                                   boolean generateOperator,
                                   List<ObjectModelClass> classes) {

        String modelName = model.getName();
        String modelVersion = model.getVersion();

        addImport(daoHelper, TopiaContextImplementor.class);
        addImport(daoHelper, TopiaDAO.class);
        addImport(daoHelper, TopiaEntity.class);
        addImport(daoHelper, TopiaContext.class);
        addImport(daoHelper, Array.class);

        if (generateOperator) {
            addImport(daoHelper,EntityOperator.class);
            addImport(daoHelper, EntityOperatorStore.class);
        }

        // add non public constructor
        ObjectModelOperation constructor =
                addConstructor(daoHelper, ObjectModelJavaModifier.PROTECTED);
        setOperationBody(constructor," ");

        ObjectModelOperation op;

        // getModelVersion method
        op = addOperation(daoHelper, "getModelVersion", "String", ObjectModelJavaModifier.PUBLIC, ObjectModelJavaModifier.STATIC);
        setOperationBody(op, ""
+"\n"
+"        return \""+modelVersion+"\";\n"
+"    "
        );

        // getModelName method
        op = addOperation(daoHelper, "getModelName", "String", ObjectModelJavaModifier.PUBLIC, ObjectModelJavaModifier.STATIC);
        setOperationBody(op, ""
+"\n"
+"        return \""+modelName+"\";\n"
+"    "
        );


        for (ObjectModelClass clazz : classes) {
            String clazzName = clazz.getName();
            String daoClazzName = clazzName + "DAO";

            // specialized getXXXDao method
            op = addOperation(daoHelper, "get" + daoClazzName, clazz.getPackageName() + '.' + daoClazzName, ObjectModelJavaModifier.PUBLIC, ObjectModelJavaModifier.STATIC);
            addParameter(op, TopiaContext.class, "context");
            addImport(daoHelper, clazz);
            addException(op, TopiaException.class);
            setOperationBody(op, ""
+"\n"
+"        TopiaContextImplementor ci = (TopiaContextImplementor) context;\n"
+"        "+daoClazzName+" result = ci.getDAO("+clazzName+".class, "+daoClazzName+".class);\n"
+"        return result;\n"
+"    "
            );

        }

        // generic getDao method
        op = addOperation(daoHelper, "getDAO", "<T extends TopiaEntity, D extends TopiaDAO<? super T>> D", ObjectModelJavaModifier.PUBLIC, ObjectModelJavaModifier.STATIC);
        addParameter(op, TopiaContext.class, "context");
        addParameter(op, "Class<T>", "klass");
        addException(op, TopiaException.class);
        setOperationBody(op, ""
+"\n"
+"        TopiaContextImplementor ci = (TopiaContextImplementor) context;\n"
+"        "+entityEnumName+" constant = "+entityEnumName+".valueOf(klass);\n"
+"        D dao = (D) ci.getDAO(constant.getContract());\n"
+"        return dao;\n"
+"    "
        );

        op = addOperation(daoHelper, "getDAO", "<T extends TopiaEntity, D extends TopiaDAO<? super T>> D", ObjectModelJavaModifier.PUBLIC, ObjectModelJavaModifier.STATIC);
        addParameter(op, TopiaContext.class, "context");
        addParameter(op, "T", "entity");
        addException(op, TopiaException.class);
        setOperationBody(op, ""
+"\n"
+"        TopiaContextImplementor ci = (TopiaContextImplementor) context;\n"
+"        "+entityEnumName+" constant = "+entityEnumName+".valueOf(entity);\n"
+"        D dao = (D) ci.getDAO(constant.getContract());\n"
+"        return dao;\n"
+"    "
        );

        // getContractClass method
        op = addOperation(daoHelper, "getContractClass", "<T extends TopiaEntity> Class<T>", ObjectModelJavaModifier.PUBLIC, ObjectModelJavaModifier.STATIC);
        addParameter(op, "Class<T>", "klass");
        setOperationBody(op, ""
+"\n"
+"        "+entityEnumName+" constant = "+entityEnumName+".valueOf(klass);\n"
+"        return (Class<T>) constant.getContract();\n"
+"    "
        );

        // getImplementationClass method
        op = addOperation(daoHelper, "getImplementationClass", "<T extends TopiaEntity> Class<T>", ObjectModelJavaModifier.PUBLIC, ObjectModelJavaModifier.STATIC);
        addParameter(op, "Class<T>", "klass");
        setOperationBody(op, ""
+"\n"
+"        "+entityEnumName+" constant = "+entityEnumName+".valueOf(klass);\n"
+"        return (Class<T>) constant.getImplementation();\n"
+"    "
        );

        // getContractClasses method
        op = addOperation(daoHelper, "getContractClasses", "Class<? extends TopiaEntity>[]", ObjectModelJavaModifier.PUBLIC, ObjectModelJavaModifier.STATIC);
        setOperationBody(op, ""
+"\n"
+"        "+entityEnumName+"[] values = "+entityEnumName+".values();\n"
+"        Class<? extends TopiaEntity>[] result = (Class<? extends TopiaEntity>[]) Array.newInstance(Class.class, values.length);\n"
+"        for (int i = 0; i < values.length; i++) {\n"
+"            result[i] = values[i].getContract();\n"
+"        }\n"
+"        return result;\n"
+"    "
        );

        // getImplementationClasses method
        op = addOperation(daoHelper, "getImplementationClasses", "Class<? extends TopiaEntity>[]", ObjectModelJavaModifier.PUBLIC, ObjectModelJavaModifier.STATIC);
        setOperationBody(op, ""
+"\n"
+"        "+entityEnumName+"[] values = "+entityEnumName+".values();\n"
+"        Class<? extends TopiaEntity>[] result = (Class<? extends TopiaEntity>[]) Array.newInstance(Class.class, values.length);\n"
+"        for (int i = 0; i < values.length; i++) {\n"
+"            result[i] = values[i].getImplementation();\n"
+"        }\n"
+"        return result;\n"
+"    "
        );

        // getImplementationClassesAsString method
        op = addOperation(daoHelper, "getImplementationClassesAsString", "String", ObjectModelJavaModifier.PUBLIC, ObjectModelJavaModifier.STATIC);
        setOperationBody(op, ""
+"\n"
+"        StringBuilder buffer = new StringBuilder();\n"
+"        for (Class<? extends TopiaEntity> aClass : getImplementationClasses()) {\n"
+"            buffer.append(',').append(aClass.getName());\n"
+"        }\n"
+"        return buffer.substring(1);\n"
+"    "
        );

        // getContracts method
        op = addOperation(daoHelper, "getContracts", entityEnumName+"[]", ObjectModelJavaModifier.PUBLIC, ObjectModelJavaModifier.STATIC);
        setOperationBody(op, ""
+"\n"
+"        return "+entityEnumName+".values();\n"
+"    "
        );

        if (generateOperator) {
            // getOperator method
            op = addOperation(daoHelper, "getOperator", "<T extends TopiaEntity> EntityOperator<T>", ObjectModelJavaModifier.PUBLIC, ObjectModelJavaModifier.STATIC);
            addParameter(op,"Class<T>","klass");
            setOperationBody(op, ""
+"\n"
+"        "+entityEnumName+" constant = "+entityEnumName+".valueOf(klass);\n"
+"        return EntityOperatorStore.getOperator(constant);\n"
+"    "
            );
        }
    }

    protected void createEntityEnum(ObjectModelEnumeration entityEnum,
                                    String daoHelperClazzName,
                                    String entityEnumName,
                                    boolean generateOperator,
                                    boolean generateStandaloneEnum,
                                    List<ObjectModelClass> classes) {

        ObjectModelAttributeImpl attr;
        ObjectModelOperation op;

        addInterface(entityEnum, TopiaEntityEnum.class);

        for (ObjectModelClass clazz : classes) {
            String clazzName = clazz.getName();

            boolean withNatural = false;
            boolean withNotNull = false;
            StringBuilder naturalIdsParams = new StringBuilder();
            StringBuilder notNullParams = new StringBuilder();

            Set<ObjectModelAttribute> naturalIdsAttributes = TopiaGeneratorUtil.getNaturalIdAttributes(clazz);
            for (ObjectModelAttribute attribute: naturalIdsAttributes) {
                withNatural = true;
                // attribut metier
                naturalIdsParams.append(", \"").append(attribute.getName()).append("\"");
            }
            Set<ObjectModelAttribute> notNullIdsAttributes = TopiaGeneratorUtil.getNotNullAttributes(clazz);
            for (ObjectModelAttribute attribute : notNullIdsAttributes) {
                withNotNull = true;
                // attribut not-null
                notNullParams.append(", \"").append(attribute.getName()).append("\"");
            }

            StringBuilder params = new StringBuilder(clazzName + ".class");

            String dbSchema = TopiaGeneratorUtil.getDbSchemaNameTagValue(clazz, model);
            if (dbSchema == null) {
                params.append(", null");
            } else {
                params.append(", \"").append(dbSchema.toLowerCase()).append("\"");
            }

            String dbTable  = TopiaGeneratorUtil.getDbName(clazz);
            params.append(", \"").append(dbTable.toLowerCase()).append("\"");

            if (withNotNull) {
                params.append(", new String[]{ ").append(notNullParams.substring(2)).append(" }");
            } else {
                params.append(", ArrayUtils.EMPTY_STRING_ARRAY");
            }
            if (withNatural) {
                params.append(", ").append(naturalIdsParams.substring(2));
            }
            addLiteral(entityEnum, clazzName + '(' + params.toString() + ')');

            if (generateStandaloneEnum) {
                addImport(entityEnum, clazz);
            }
        }

        attr = (ObjectModelAttributeImpl) addAttribute(entityEnum, "contract", "Class<? extends TopiaEntity>");
        attr.setDocumentation("The contract of the entity.");

        attr = (ObjectModelAttributeImpl) addAttribute(entityEnum, "dbSchemaName", "String");
        attr.setDocumentation("The optional name of database schema of the entity (if none was filled, will be {@code null}).");

        attr = (ObjectModelAttributeImpl) addAttribute(entityEnum, "dbTableName", "String");
        attr.setDocumentation("The name of the database table for the entity.");

        attr = (ObjectModelAttributeImpl) addAttribute(entityEnum, "implementationFQN", "String");
        attr.setDocumentation("The fully qualified name of the implementation of the entity.");

        attr = (ObjectModelAttributeImpl) addAttribute(entityEnum, "implementation", "Class<? extends TopiaEntity>");
        attr.setDocumentation("The implementation class of the entity (will be lazy computed at runtime).");

        attr = (ObjectModelAttributeImpl) addAttribute(entityEnum, "naturalIds", "String[]");
        attr.setDocumentation("The array of property involved in the natural key of the entity.");

        attr = (ObjectModelAttributeImpl) addAttribute(entityEnum, "notNulls", "String[]");
        attr.setDocumentation("The array of not null properties of the entity.");

        // constructor
        op = addConstructor(entityEnum, ObjectModelJavaModifier.PACKAGE);
        addParameter(op,"Class<? extends TopiaEntity >","contract");
        addParameter(op,"String","dbSchemaName");
        addParameter(op,"String","dbTableName");
        addParameter(op,"String[]","notNulls");
        addParameter(op,"String...","naturalIds");
        setOperationBody(op, ""
+"\n"
+"        this.contract = contract;\n"
+"        this.dbSchemaName = dbSchemaName;\n"
+"        this.dbTableName = dbTableName;\n"
+"        this.notNulls = notNulls;\n"
+"        this.naturalIds = naturalIds;\n"
+"        implementationFQN = contract.getName() + \"Impl\";\n"
+"    "
        );

        // getContract method
        op = addOperation(entityEnum, "getContract", "Class<? extends TopiaEntity>", ObjectModelJavaModifier.PUBLIC);
        addAnnotation(entityEnum,op,Override.class.getSimpleName());
        setOperationBody(op, ""
+"\n"
+"        return contract;\n"
+"    "
        );

        // dbSchemaName method
        op = addOperation(entityEnum, "dbSchemaName", String.class, ObjectModelJavaModifier.PUBLIC);
        addAnnotation(entityEnum,op,Override.class.getSimpleName());
        setOperationBody(op, ""
+"\n"
+"        return dbSchemaName;\n"
+"    "
        );

        // dbTableName method
        op = addOperation(entityEnum, "dbTableName", String.class, ObjectModelJavaModifier.PUBLIC);
        addAnnotation(entityEnum,op,Override.class.getSimpleName());
        setOperationBody(op, ""
+"\n"
+"        return dbTableName;\n"
+"    "
        );

        // getNaturalIds method
        op = addOperation(entityEnum, "getNaturalIds", "String[]", ObjectModelJavaModifier.PUBLIC);
        addAnnotation(entityEnum,op,Override.class.getSimpleName());
        setOperationBody(op, ""
+"\n"
+"        return naturalIds;\n"
+"    "
        );

        // isUseNaturalIds method
        op = addOperation(entityEnum, "isUseNaturalIds", "boolean", ObjectModelJavaModifier.PUBLIC);
        addAnnotation(entityEnum,op,Override.class.getSimpleName());
        setOperationBody(op, ""
+"\n"
+"        return naturalIds.length > 0;\n"
+"    "
        );

        // getNotNulls method
        op = addOperation(entityEnum, "getNotNulls", "String[]", ObjectModelJavaModifier.PUBLIC);
        addAnnotation(entityEnum,op,Override.class.getSimpleName());
        setOperationBody(op, ""
+"\n"
+"        return notNulls;\n"
+"    "
        );

        // isUseNotNulls method
        op = addOperation(entityEnum, "isUseNotNulls", "boolean", ObjectModelJavaModifier.PUBLIC);
        addAnnotation(entityEnum,op,Override.class.getSimpleName());
        setOperationBody(op, ""
+"\n"
+"        return notNulls.length > 0;\n"
+"    "
        );

        // getImplementationFQN method
        op = addOperation(entityEnum, "getImplementationFQN","String",ObjectModelJavaModifier.PUBLIC);
        addAnnotation(entityEnum,op,Override.class.getSimpleName());
        setOperationBody(op, ""
+"\n"
+"        return implementationFQN;\n"
+"    "
        );

        // setImplementationFQN method
        op = addOperation(entityEnum, "setImplementationFQN","void",ObjectModelJavaModifier.PUBLIC);
        addAnnotation(entityEnum,op,Override.class.getSimpleName());
        addParameter(op,"String","implementationFQN");
        if (generateOperator) {
               setOperationBody(op, ""
+"\n"
+"        this.implementationFQN = implementationFQN;\n"
+"        implementation = null;\n"
+"        // reinit the operators store\n"
+"        EntityOperatorStore.clear();\n"
+"    "
            );
        } else {
            setOperationBody(op, ""
+"\n"
+"        this.implementationFQN = implementationFQN;\n"
+"        this.implementation = null;\n"
+"    "
            );
        }

        // accept method
        op = addOperation(entityEnum, "accept","boolean",ObjectModelJavaModifier.PUBLIC);
        addAnnotation(entityEnum,op,Override.class.getSimpleName());
        addParameter(op,"Class<? extends TopiaEntity>","klass");
        setOperationBody(op, ""
+"\n"
+"        return "+daoHelperClazzName+".getContractClass(klass) == contract;\n"
+"    "
        );

        // getImplementation method
        op = addOperation(entityEnum, "getImplementation","Class<? extends TopiaEntity>",ObjectModelJavaModifier.PUBLIC);
        addAnnotation(entityEnum,op,Override.class.getSimpleName());
        setOperationBody(op, ""
+"\n"
+"        if (implementation == null) {\n"
+"        try {\n"
+"                implementation = (Class<? extends TopiaEntity>) Class.forName(implementationFQN);\n"
+"            } catch (ClassNotFoundException e) {\n"
+"                throw new TopiaRuntimeException(\"could not find class \" + implementationFQN, e);\n"
+"            }\n"
+"        }\n"
+"        return implementation;\n"
+"    "
        );

        // valueOf method
        op = addOperation(entityEnum, "valueOf", entityEnumName, ObjectModelJavaModifier.PUBLIC,ObjectModelJavaModifier.STATIC);
        addParameter(op,"TopiaEntity","entity");
        setOperationBody(op, ""
+"\n"
+"        return valueOf(entity.getClass());\n"
+"    "
        );

        // valueOf method
        op = addOperation(entityEnum, "valueOf", entityEnumName, ObjectModelJavaModifier.PUBLIC,ObjectModelJavaModifier.STATIC);
        addParameter(op,"Class<?>","klass");
        setOperationBody(op, ""
+"\n"
+"        if (klass.isInterface()) {\n"
+"           return valueOf(klass.getSimpleName());\n"
+"        }\n"
+"\n"
+"        Class<?> contractClass = TopiaEntityHelper.getContractClass("+entityEnumName+".values(), (Class) klass);\n"
+"\n"
+"        if (contractClass != null) {\n"
+"\n"
+"            return valueOf(contractClass.getSimpleName());\n"
+"        }\n"
+"\n"
+"        throw new IllegalArgumentException(\"no entity defined for the class \" + klass + \" in : \" + Arrays.toString("+entityEnumName+".values()));\n"
+"    "
        );
    }
}
