/*
 * #%L
 * Nuiton Utils :: Nuiton Validator
 * 
 * $Id: FrenchSiretFieldValidator.java 2240 2011-11-30 08:52:51Z athimel $
 * $HeadURL: http://svn.nuiton.org/svn/nuiton-utils/tags/nuiton-utils-2.5/nuiton-validator/src/main/java/org/nuiton/validator/xwork2/field/FrenchSiretFieldValidator.java $
 * %%
 * Copyright (C) 2011 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.validator.xwork2.field;

import com.opensymphony.xwork2.validator.ValidationException;

import java.lang.reflect.Array;
import java.util.Collection;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.lang3.StringUtils;

/**
 * Validator for French SIRET numbers
 * <p/>
 * Siret can be in:
 * <li>String format: "44211670300038"
 * <li>long, int: 44211670300038
 * <li>Array or Collection of something: [4,4,2,1,1,6,7,0,,3,0,0,0,3,8] or ["442","116","703", "0003", "8"]
 *
 * @author jcouteau <couteau@codelutin.com>
 * @since 2.3
 *        Validation do the Luhn checksum too
 */
public class FrenchSiretFieldValidator extends NuitonFieldValidatorSupport {

    protected static final String SIRET_REGEXP = "[0-9]{14}";

    protected static final Pattern p = Pattern.compile(SIRET_REGEXP);

    @Override
    public void validateWhenNotSkip(Object object) throws ValidationException {

        String fieldName = getFieldName();
        Object value = getFieldValue(fieldName, object);

        if (value == null) {
            // no value defined
            return;
        }
        String siret;

        if (value.getClass().isArray()) {
            // le siret est stocker dans un tableau, par exemple un byte[]
            siret = "";
            for (int i = 0; i < Array.getLength(value); i++) {
                siret += String.valueOf(Array.get(value, i));
            }
        } else if (value instanceof Collection<?>) {
            // le siret est stocker dans une collection,
            // ca doit pas arriver souvent :D, mais autant le gerer
            siret = "";
            for (Object o : (Collection<?>) value) {
                siret += String.valueOf(o);
            }
        } else {
            // sinon dans tous les autres cas (String, int, long, BigInteger ...)
            // on prend le toString
            siret = String.valueOf(value);
        }

        if (StringUtils.isEmpty(siret)) {
            // no value defined
            return;
        }

        // Remove any space
        siret = siret.replaceAll(" ", "");

        Matcher m = p.matcher(siret);
        if (!m.matches() || !luhnChecksum(siret)) {
            addFieldError(fieldName, object);
        }
    }

    @Override
    public String getValidatorType() {
        return "frenchSiret";
    }

    /**
     * Verifie la validite d'un numero en suivant l'algorithme Luhn tel que d'ecrit
     * dans <a href="http://fr.wikipedia.org/wiki/Luhn">wikipedia</a>
     * <p/>
     * Algo:
     * en fonction de la position du numero dans la sequence,
     * on multiplie pas 1 (pour les impaires) ou par 2 pour les paires
     * (1 etant le numero le plus a droite)
     * On fait la somme de tous les chiffres qui resulte de ces multiplications
     * (si un resultat etait 14, on ne fait pas +14 mais +1+4)
     * <p/>
     * Si le résultat de cette somme donne un reste de 0 une fois divisé par 10
     * le numero est valide.
     *
     * @param siret une chaine composer que de chiffre
     * @return vrai si on a reussi a valider le numero
     */
    public static boolean luhnChecksum(String siret) {

        char[] tab = siret.toCharArray();
        int sum = 0;
        for (int i = tab.length - 1; i >= 0; i--) {
            // recuperation de la valeur
            int n = getDigit(tab[i]);

            // 1ere phase il faut faire la multiplication par 1 ou 2

            // il faut faire x1 pour les paires et x2 sur les impaires.
            // en prenant en compte que le numero siret le plus a droite est le
            // 1 et le plus a gauche le 14.
            // mais comme en informatique on commence a 0 :D
            // il faut faire +1 sur l'indice puis un simple module 2 + 1
            // nous convient
            n *= (i + 1) % 2 + 1;

            // 2eme phase il faut faire l'addition

            // si une fois multiplie il est superieur a 9, il faut additionner
            // toutes ces constituante, mais comme il ne peut pas etre superieur
            // a 18, cela revient a retrancher 9
            if (n > 9) {
                n -= 9;
            }

            // on peut directement faire la somme
            sum += n;
        }

        // 3eme phase on verifie que c'est bien un multiple de 10
        boolean result = sum % 10 == 0;

        return result;
    }


    /**
     * Converti un char en un entier '0' => 0 et '9' => 9, et 'A' => 10 a 'Z' => 36,
     * les autres caractere sont aussi convertis pour que 'a' = 10 et 'z' = 36.
     * Pour les autres c'est un indedermine
     *
     * @param c le caractere qui doit etre converti
     * @return le chiffre
     */
    public static int getDigit(char c) {
        int result = 0;
        if (c >= '0' && c <= '9') {
            result = c - '0';
        } else if (c >= 'A' && c <= 'Z') {
            result = c - 'A' + 10;
        } else {
            result = c - 'a' + 10;
        }
        return result;
    }
}
