/*
 * #%L
 * JRedmine :: Maven plugin
 * 
 * $Id: AbstractRedmineMojo.java 157 2010-10-08 10:23:16Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jredmine/tags/jredmine-1.1.4/maven-jredmine-plugin/src/main/java/org/nuiton/jredmine/plugin/AbstractRedmineMojo.java $
 * %%
 * Copyright (C) 2009 - 2010 Tony Chemit, CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.jredmine.plugin;

import org.apache.maven.execution.MavenSession;
import org.apache.maven.model.IssueManagement;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.MavenProject;
import org.nuiton.io.rest.RestClientConfiguration;
import org.nuiton.jredmine.RedmineService;
import org.nuiton.jredmine.RedmineServiceException;
import org.nuiton.jredmine.RedmineServiceImplementor;
import org.nuiton.jredmine.model.ModelHelper;
import org.nuiton.jredmine.model.Project;
import org.nuiton.jredmine.model.User;
import org.nuiton.jredmine.model.Version;
import org.nuiton.plugin.AbstractPlugin;

import java.net.URL;
import java.text.DateFormat;
import java.text.SimpleDateFormat;

/**
 * Abstract redmine mojo.
 *
 * @author tchemit
 * @requiresOnline true
 * @requiresProject true
 * @since 1.0.0
 */
public abstract class AbstractRedmineMojo extends AbstractPlugin implements RestClientConfiguration {

    public static final String REDMINE_SYSTEM = "redmine";

    /**
     * Dependance du projet.
     *
     * @parameter default-value="${project}"
     * @required
     * @readonly
     * @since 1.0.0
     */
    protected MavenProject project;

    /**
     * The real basedir redmine url.
     * <p/>
     * If no url is given, will use the issue management url.
     *
     * @parameter expression="${redmine.url}"
     * @since 1.0.0
     */
    protected URL url;

    /**
     * The redmine's server login.
     * <p/>
     * <b>Note:</b> : this parameter is mandatory if you not use a {@code anonymous} service.
     * configuration.
     *
     * @parameter expression="${redmine.username}"
     * @since 1.0.0
     */
    protected String username;

    /**
     * The redmine's server password.
     * <p/>
     * <b>Note:</b> : this parameter is mandatory if you not use a {@code anonymous} service.
     * configuration.
     *
     * @parameter expression="${redmine.password}"
     * @since 1.0.0
     */
    protected String password;

    /**
     * The encoding used to read and write files.
     * <p/>
     * <b>Note:</b> If nothing is filled here, we will use the system
     * property {@code file.encoding}.
     *
     * @parameter expression="${redmine.encoding}" default-value="${project.build.sourceEncoding}"
     * @since 1.0.0
     */
    protected String encoding;

    /**
     * Redmine project name.
     *
     * @parameter expression="${redmine.projectId}" default-value="${project.artifactId}"
     * @required
     * @since 1.0.0
     */
    protected String projectId;

    /**
     * redmine version name.
     *
     * @parameter expression="${redmine.versionId}" default-value="${project.version}"
     * @since 1.0.0
     */
    protected String versionId;

    /**
     * Un flag pour activer le mode verbeux.
     *
     * @parameter expression="${redmine.verbose}"  default-value="${maven.verbose}"
     * @since 1.0.0
     */
    protected boolean verbose;

    /**
     * Un flag pour faire échouer le build si la configuration n'est pas ok.
     *
     * @parameter expression="${redmine.safe}"  default-value="true"
     * @since 1.0.0
     */
    protected boolean safe;

    /**
     * @parameter expression="${session}"
     * @readonly
     */
    protected MavenSession session;

    /**
     * Redmine service.
     *
     * @component
     * @since 1.0.0
     */
    protected RedmineService service;

    /** flag to load in init a required project using the {@link #projectId} name */
    private final boolean requireProject;

    /** flag to load in init a required version using the {@link #versionId} name. */
    private final boolean requireVersion;

    /** flag to load in init a required user using the user loggued to redmine server. */
    private final boolean requireUser;

    /** the project loaded in init if {@link #requireProject} flag is on */
    protected Project releaseProject;

    /** the version loaded in init if {@link #requireVersion} flag is on */
    protected Version releaseVersion;

    /** the user loaded in init if {@link #requireUser} flag is on */
    protected User releaseUser;

    /** cache of users of a given project loaded in int if {@link #requireUser} flag is on */
    protected User[] users;

    /** the date format used to write a date */
    protected DateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");

    /** flag to mark if a runOnce goal was done */
    protected boolean runOnceDone;

    /** flag to mark if service was sucessfull init */
    protected boolean serviceInit;

    protected boolean initOk = true;

    public AbstractRedmineMojo(boolean requireProject, boolean requireVersion, boolean requireUser) {
        this.requireProject = requireProject;
        this.requireVersion = requireVersion;
        this.requireUser = requireUser;
    }

    ///////////////////////////////////////////////////////////////////////////
    /// AbstractPlugin
    ///////////////////////////////////////////////////////////////////////////

    @Override
    protected void init() throws Exception {

        // check issue management

        IssueManagement issueManagement = project.getIssueManagement();

        if (issueManagement == null) {
            throw new MojoExecutionException("No Issue Management set.");
        } else if (issueManagement.getUrl() == null || issueManagement.getUrl().trim().equals("")) {
            throw new MojoExecutionException("No URL set in Issue Management.");
        } else if (issueManagement.getSystem() != null && !issueManagement.getSystem().equalsIgnoreCase(REDMINE_SYSTEM)) {
            throw new MojoExecutionException("Redmine's Plugin only supports 'redmine' Issue Management system.");
        }
        // prepare Redmine service configuration

        URL url = getRestUrl();

        if (url == null || url.toString().isEmpty()) {

            // no redmine url specified, guess it from issueManagement

            url = new URL(issueManagement.getUrl());

            if (verbose) {
                getLog().info("use the url from issue management : " + url);
            }
        }

        // apply configuration

        setRestUrl(url);

        if (verbose) {
            if (isAnonymous()) {
                getLog().info("Redmine anonymous configuration :\n>> host     : " +
                              getRestUrl());
            } else {
                getLog().info("Redmine configuration :\n>> host     : " +
                              getRestUrl() + "\n>> username : " +
                              getRestUsername());
            }
        }

        // init Redmine service

        try {

            ((RedmineServiceImplementor) service).init(this);
            serviceInit = true;
        } catch (Exception e) {
            if (isSafe()) {
                throw e;
            }
            serviceInit = false;
            initOk = false;
            getLog().error("could not init Redmine service [" + getRestUrl() + "] with user '" + getRestUsername() + "'", e);
        }

        // check project exists

        if (requireProject) {
            boolean r = initReleaseProject();
            if (!r) {
                if (isSafe()) {
                    throw new MojoExecutionException("the project '" + projectId + "' could not be retrieve from redmine server.");
                }
                initOk = false;
                return;
            }
        }

        // check user exists

        if (requireUser) {
            boolean r = initReleaseUser();
            if (!r) {
                if (isSafe()) {
                    throw new MojoExecutionException("the user '" + username + "' could not be retrieve from redmine server.");
                }
                initOk = false;
                return;
            }
        }

        // check version exists

        if (requireVersion) {
            boolean r = initReleaseVersion();
            if (!r) {
                if (isSafe()) {
                    throw new MojoExecutionException("the version '" + versionId + "' could not be retrieve from redmine server.");
                }
                initOk = false;
            }
        }
    }

    @Override
    protected boolean checkSkip() {

        if (isGoalSkip()) {
            getLog().warn("The goal is skip due to the skipGoal flag on");
            return false;
        }
        if (isRunOnce() && isRunOnceDone()) {
            getLog().info("skip goal, runOnce flag is on, and was already executed.");
            return false;
        }

        if (!serviceInit) {
            getLog().error("could not init Redmine service [" + getRestUrl() + "] with user '" + getRestUsername() + "'");
            return false;
        }

        if (requireProject && releaseProject == null) {

            getLog().error("the project '" + projectId + "' could not be retrieve from redmine server, goal is skip");
            return false;
        }
        if (requireUser && releaseUser == null) {
            getLog().error("the user '" + username + "' could not be retrieve from redmine server, goal is skip");
            return false;
        }

        if (requireVersion && releaseVersion == null) {
            getLog().error("the version '" + versionId + "' could not be retrieve from redmine server, goal is skip");
            return false;
        }

        return true;
    }

    @Override
    protected void afterExecute() {
        closeService();
    }

    /**
     * Re-expose the protected method for test purposes...
     *
     * @throws Exception if any problems
     */
    @Override
    protected abstract void doAction() throws Exception;

    protected abstract boolean isGoalSkip();

    protected abstract boolean isRunOnce();

    protected abstract boolean checkRunOnceDone();
    ///////////////////////////////////////////////////////////////////////////
    /// Plugin
    ///////////////////////////////////////////////////////////////////////////

    @Override
    public MavenProject getProject() {
        return project;
    }

    @Override
    public void setProject(MavenProject project) {
        this.project = project;
    }

    @Override
    public boolean isVerbose() {
        return verbose;
    }

    @Override
    public void setVerbose(boolean verbose) {
        this.verbose = verbose;
    }

    ///////////////////////////////////////////////////////////////////////////
    /// PluginWithencoding
    ///////////////////////////////////////////////////////////////////////////

    @Override
    public String getEncoding() {
        return encoding;
    }

    @Override
    public void setEncoding(String encoding) {
        this.encoding = encoding;
    }

    ///////////////////////////////////////////////////////////////////////////
    /// RestClientConfiguration
    ///////////////////////////////////////////////////////////////////////////

    @Override
    public String getRestPassword() {
        return password;
    }

    @Override
    public URL getRestUrl() {
        return url;
    }

    @Override
    public String getRestUsername() {
        return username;
    }

    @Override
    public void setRestPassword(String restPassword) {
        password = restPassword;
    }

    @Override
    public void setRestUrl(URL restUrl) {
        url = restUrl;
    }

    @Override
    public void setRestUsername(String restUsername) {
        username = restUsername;
    }

    ///////////////////////////////////////////////////////////////////////////
    /// Others
    ///////////////////////////////////////////////////////////////////////////

    protected boolean initReleaseProject() throws MojoExecutionException {

        if (projectId == null || projectId.trim().isEmpty()) {
            throw new MojoExecutionException("required a projectId parameter");
        }
        try {
            Project p = service.getProject(projectId);

            if (p == null) {

                return false;
            }

            releaseProject = p;
            return true;
        } catch (RedmineServiceException e) {
            getLog().warn("could not retreave project '" + projectId + "', for reason " + e.getMessage(), e);
            return false;
        }
    }

    protected boolean initReleaseVersion() throws MojoExecutionException {

        if (versionId == null || versionId.trim().isEmpty()) {
            throw new MojoExecutionException("required a versionId parameter");
        }

        try {

            Version v = service.getVersion(projectId, versionId);

            if (v == null) {

                return false;
            }

            releaseVersion = v;
            return true;
        } catch (RedmineServiceException e) {
            getLog().warn("could not retreave version '" + versionId + "', for reason " + e.getMessage(), e);
            return false;
        }
    }

    protected boolean initReleaseUser() {

        try {
            users = service.getProjectMembers(projectId);

            User user = ModelHelper.byLogin(username, users);

            if (user == null) {

                return false;
            }

            releaseUser = user;
            return true;
        } catch (RedmineServiceException e) {
            getLog().warn("could not retreave user '" + username + "', for reason " + e.getMessage(), e);
            return false;
        }
    }

    protected void closeService() {
        if (service != null) {
            RedmineServiceImplementor i;
            i = (RedmineServiceImplementor) service;

            if (i.isInit()) {
                try {
                    if (verbose) {
                        getLog().info("<<< Close redmine rest client...");
                    }
                    i.destroy();
                } catch (Exception ex) {
                    getLog().error("could not close redmine client for reason " + ex.getMessage(), ex);
                }
            }
        }
    }

    public boolean isRunOnceDone() {
        return runOnceDone;
    }

    public boolean isSafe() {
        return safe;
    }

    public void setSafe(boolean safe) {
        this.safe = safe;
    }

    public String getProjectId() {
        return projectId;
    }

    public String getVersionId() {
        return versionId;
    }
}
